/*
    File:       Controller.m

    Contains:   Central control object for sample.

    Written by: Quinn "The Eskimo!"

    Created:    Tue 10-Jun-1997

    Copyright:  (c)1997 by Apple Computer, Inc., all rights reserved.

    Change History (most recent first):

    You may incorporate this sample code into your applications without
    restriction, though the sample code has been provided "AS IS" and the
    responsibility for its operation is 100% yours.  However, what you are
    not permitted to do is to redistribute the source as "DSC Sample Code"
    after having made changes. If you're going to re-distribute the source,
    we require that you make it clear in the source that the code was
    descended from Apple Sample Code, but that you've made changes.
*/

#import "Controller.h"

#import "TransferServer.h"

// IMPORTANT: See the documentation ("ReadMe.rtf" under Supporting Files) for
// big picture information about this project.

@implementation Controller

- (id)init
    // See comments in interface part.
{
    NSPort *port1;
    NSPort *port2;
    NSArray *portArray;

    self = [super init];
    if (self != nil) {

        // First create two new ports and a new NSConnection for sending
        // and receiving Distributed Object messages through those ports.  We do
        // this (rather than use attempting to reuse the default NSConnection
        // that all applications have) because we want the ports to remain
        // anonymous.  These ports are for talking between our application's
        // threads only; we don't want them published by name on the network.
        
        port1 = [NSPort port];
        port2 = [NSPort port];
        connectionToTransferServer = [[NSConnection alloc] initWithReceivePort:port1 sendPort:port2];

        // Now set this object as the root object for the connection that we're about
        // to connect to the server.  We do this so that the server can call us
        // back to a) give us a reference to the transferServer object by calling
        // our setServer method, and b) get more information from us by calling
        // our methods (which it can do, but doesn't in this sample).
        
        [connectionToTransferServer setRootObject:self];

        // Initialise the transfer server to nil.  Note that there's a concurrency
        // hole in this implementation.  transferServer is nil when we leave this method,
        // and stays nil until TransferServer's connectWithPorts: runs in its thread
        // and sends us a setServer: message to set it.  But we leave this routine without
        // waiting for that return message.  If the new thread is unexpectedly
        // slowed down, it might be possible for someone to call our action methods
        // (eg doSomethingSlow:) before transferServer has been set up.  Of course,
        // this is *extremely* unlikely (because our action methods are only invoked
        // as a result of user action) but it's worth noting.  
        
        transferServer = nil;

        // Now put the two ports in an array and start a new thread, executing
        // TransferServer's connectWithPorts: method, with that array as its
        // argument.  Notice how the ports are reversed here, so TransferServer's
        // connectWithPorts connects its send port to our receive port and vice versa.
        //
        // I'm a bit disturbed by the possibility of portArray being cleaned up
        // before connectWithPorts: gets to run, but in practice this is not a problem.

        portArray = [NSArray arrayWithObjects:port2, port1, nil];
        [NSThread detachNewThreadSelector:@selector(connectWithPorts:)
                                toTarget:[TransferServer class]
                                withObject:portArray];
    }
    return (self);
}

- (void)awakeFromNib
{
    [mainWindow makeKeyAndOrderFront:self];
}

- (void)setServer:(id)serverObject
    // See comments in interface part.
{
    // The following line is an interesting optimisation.  We tell our proxy
    // to the transferServer object to about the methods that we're going to
    // send to the proxy.  This optimises Distributed Object's delivery of
    // messages.  [Normally when DO encounters a new method, it must first
    // conduct a transaction with the remote end to find the types for the
    // arguments of that message.  It then bundles up the method and its
    // parameters and sends it.  It also caches the response so that following
    // invokations of that method only take one transaction.  By setting
    // a protocol for the proxy, you let DO know about the messages in
    // advance, and avoid it ever having to do two transactions.]
    
    [serverObject setProtocolForProxy:@protocol(TransferServerInterface)];

    // We now retain the server object.  TransferServer's connectWithPorts:
    // method is about to release it, so we must make sure that someone
    // has a reference!  This also means that we're the *only* person with
    // a reference.
    
    [serverObject retain];
       
    // Now record the remote server object (actually its proxy) in our instance
    // variable.  Note that "(id <TransferServerInterface>)" is a form
    // of type casting.  It says that serverObject conforms to the
    // TransferServerInterface protocol.
    
    transferServer = (id <TransferServerInterface>)serverObject;
    
    return;
}

- (oneway void)outputString:(NSString *)theString
    // See comments in interface part.  Note that textField is
    // an instance variable declared in our header file that was wired
    // to the real text field in the window via Interface Builder.
{
    [textField replaceCharactersInRange:NSMakeRange(
                        [ [textField string] length ], 0)
                        withString:theString];
}

- (void)doSomethingSlow:(id)sender
    // See comments in interface part.
{
    [self outputString:@"doSomethingSlow:\n"];
    [transferServer slowTransfer:self];
}

- (void)doSomethingSlower:(id)sender
    // See comment in interface part.
{
    [self outputString:@"doSomethingSlower:\n"];
    [transferServer slowerTransfer:self];
}

@end
