// Copyright 1997-1998 Omni Development, Inc.  All rights reserved.
//
// This software may only be used and reproduced according to the
// terms in the file OmniSourceLicense.html, which should be
// distributed with this project and can also be found at
// http://www.omnigroup.com/DeveloperResources/OmniSourceLicense.html.

#import "OBPostLoader.h"

#import "OBUtilities.h"
#import "assertions.h"

#import <Foundation/Foundation.h>
#import <objc/hashtable.h>
#import <objc/objc-class.h>
#import <objc/objc-runtime.h>
#import "rcsid.h"

RCS_ID("$Header: /Network/Developer/Source/CVS/OmniGroup/OmniBase/OBPostLoader.m,v 1.14 1998/12/08 04:07:36 kc Exp $")

static NSRecursiveLock *lock = nil;
static NSHashTable     *calledImplemenations = NULL;

@interface OBPostLoader (PrivateAPI)
+ (BOOL) _processSelector: (SEL) selectorToCall inClass: (Class) aClass initialize: (BOOL) shouldInitialize;
+ (void) _preventBundleInitialization;
+ (void) _provokeBundleInitialization;
@end


@implementation OBPostLoader

+ (void) initialize;
{
    static BOOL initialized = NO;
    
    if (initialized)
        return;

    initialized = YES;

    // Set this up before we call any method implementations
    calledImplemenations = NSCreateHashTable(NSNonOwnedPointerHashCallBacks, 0);

    // This will cause an error to be reported is some goober
    // does something in the +performPosing methods that causes
    // +[NSBundle initialize] to be called.
    [self _preventBundleInitialization];

    // We have to do this before calling +[NSBundle initialize] since on
    // HP-UX calling class_poseAs() after that point can cause SEGVs.  It
    // is unclear why this is the case.
    // Also, we need -[NSBundle infoDictionary] to get replaced with
    // -replacement_infoDictionary early on (before +processDefaults is
    // called).
    [self processSelector: @selector(performPosing) initialize: NO];

    // Un-boobytrap and force a call to +[NSBundle initialize] now that we
    // have done all of the +performPosing calls
    [self _provokeBundleInitialization];

#ifdef hpux
    // Register frameworks.  This is necessary since on HP-UX, framworks
    // aren't automaticaly registered, and the +processDefaults below will
    // fail to find framework resources otherwise.  On HP-UX we need all of
    // the +registerFramework methods to get called before
    // +[NSBundle bundleForClass:] is invoked.
    [self processSelector: @selector(registerFramework) initialize: NO];
#endif
    
    // If any other bundles get loaded, make sure that we process them too.
    [[NSNotificationCenter defaultCenter] addObserver: self
                                             selector: @selector(bundleDidLoad:)
                                                 name: NSBundleDidLoadNotification
                                               object: nil];

    // Register for the multi-threadedness here so that most classes won't have to
    [[NSNotificationCenter defaultCenter] addObserver: self
                                             selector: @selector(becomingMultiThreaded:)
                                                 name: NSWillBecomeMultiThreadedNotification
                                               object: nil];
}

+ (void) processClasses;
{
    [self processSelector: @selector(performPosing) initialize: NO];
#ifdef hpux
    [self processSelector: @selector(registerFramework) initialize: NO];
#endif
    [self processSelector: @selector(didLoad) initialize: YES];
}

+ (void) processSelector: (SEL) selectorToCall initialize: (BOOL) shouldInitialize;
{
    BOOL         didInvokeSomething = YES;

    [lock lock];

    // We will collect the necessary information from the runtime before calling any method
    // implementations.  This is necessary since any once of the implementations that we call
    // could actually modify the ObjC runtime.  It could add classes or methods.

    while (didInvokeSomething) {
        NXHashTable *classHash;
        NXHashState  state;
        Class        aClass, *classes;
        unsigned int classIndex, classCount;

        // Start out claiming to have invoked nothing.  If this doesn't
        // get reset to YES below, we're done.
        didInvokeSomething = NO;

        // Gather the list of classes
        classHash = objc_getClasses();
        classCount = NXCountHashTable(classHash);
        classes = (Class *)NSZoneMalloc(NULL, NXCountHashTable(classHash) * sizeof(Class));

        state = NXInitHashState(classHash);
        classIndex = 0;
        while (NXNextHashState(classHash, &state, (void **)&aClass)) {
            OBASSERT(classIndex < classCount);
            classes[classIndex] = aClass;
            classIndex++;
        }

        // Loop over the gathered classes and process the requested implementations
        for (classIndex = 0; classIndex < classCount; classIndex++) {
            aClass = classes[classIndex];

            if ([self _processSelector: selectorToCall inClass: aClass initialize: shouldInitialize])
                didInvokeSomething = YES;
        }

        NSZoneFree(NULL, classes);
    }

    [lock unlock];
}

+ (void) bundleDidLoad: (NSNotification *) notification;
{
    [self processClasses];
}

+ (void) becomingMultiThreaded: (NSNotification *) notification;
{
    // Lets be thread-safe, shall we.
    lock = [[NSRecursiveLock alloc] init];
    
    [self processSelector: @selector(becomingMultiThreaded) initialize: NO];
}

@end




//// Private API!
// We replace this method during +[NSBundle initialize] to avoid the tons of
// file system accesses during startup that are to resolve symlinks.  It doesn't
// seem like these access are really necessary since the kernel will do them
// for us

#ifndef WIN32
@interface NSPathStore2 : NSString
- (NSString *) stringByResolvingSymlinksInPath;
@end

@implementation NSPathStore2 (OFOverrides)

- (NSString *) replacement_stringByResolvingSymlinksInPath;
{
    // Can do this since we are a string
    return self;
}

@end
#endif

static Method NSBundleInitializeMethod                 = NULL;
static IMP    originalNSBundleInitializeImplementation = NULL;

static id _NSBundleInitializationTooEarly(id self, SEL _cmd)
{
    fprintf(stderr, "+[NSBundle initialize] called too early!  This "
            "should not happen until OFPostLoader provokes it. "
            "Please rewrite your +performPosing method to not "
            "provoke +[NSBundle initialize].\n");
    abort();
    return nil;
}


@implementation OBPostLoader (PrivateAPI)

+ (BOOL) _processSelector: (SEL) selectorToCall inClass: (Class) aClass initialize: (BOOL) shouldInitialize;
{
    Class                    metaClass = aClass->isa; // we are looking at class methods
    void                    *iterator;
    struct objc_method_list *mlist;
    IMP                     *imps;
    unsigned int             impIndex, impCount, impSize;

    impSize  = 256;
    impCount = 0;
    imps = NSZoneMalloc(NULL, sizeof(IMP) * impSize);

    // Gather all the method implementations of interest on this class
    // before invoking any of them.  This is necessary since they might
    // modify the ObjC runtime.
    iterator = NULL;

    while ((mlist = class_nextMethodList(metaClass, &iterator))) {
        struct objc_method *methods;
        int                 methodCount;

        // It appears that the ObjC runtime is not thread-safe wrt dyld.
        // One thread can be in the middle of looking up something in
        // the method cache while another thread is causing dyld lazy
        // binding to occur (and thus, it is modifying the method cache).
        // Since each method list cannot span a .o file, this will cause
        // a lookup in each ObjC module, which in turn will cause dyld
        // to do its stuff.
        // Additionally, if the first method you call in any one
        // module hasn't been looked up (usually a +didLoad method)
        // you can run into crasher bug due to dyld not having
        // chewed on your code yet.

        // This uses private runtime API to make sure that the dyld
        // magic has occured on the module that contains the function
        // we are about to call.  We CANNOT simply lookup a method in
        // the module since (a) we allow duplicate definitions of the
        // methods we call and (b) if there is a duplicate, we cannot
        // be sure that we will end up hitting this method list.

        // This is not necessary on NT or HP-UX since there is no dyld
#ifdef NeXT
        {
            extern void _objc_bindModuleContainingList(struct objc_method_list *);

            _objc_bindModuleContainingList(mlist);
        }
#endif

        methodCount = mlist->method_count;
        methods     = &mlist->method_list[0];

        while (methodCount--) {
            if (methods->method_name == selectorToCall) {
                IMP imp;

                imp = methods->method_imp;

                /* Store this implementation if it hasn't already been called */
                if (!NSHashGet(calledImplemenations, imp)) {
                    if (impCount >= impSize) {
                        impSize *= 2;
                        imps = NSZoneRealloc(NULL, imps, sizeof(IMP) * impSize);
                    }

                    imps[impCount] = imp;
                    impCount++;
                    NSHashInsertKnownAbsent(calledImplemenations, imp);

#if 0 && defined(DEBUG)
                    fprintf(stderr, "Recoding +[%s %s] (0x%08x)\n", aClass->name, sel_getName(selectorToCall), imp);
#endif
                }
            }
            methods++;
        }
    }

    if (impCount) {
        if (shouldInitialize) {
            NSAutoreleasePool *pool;

            pool = [[NSAutoreleasePool alloc] init];
            // try to make sure +initialize gets called
            if (class_getClassMethod(aClass, @selector(class)))
                [aClass class];
            else if (class_getClassMethod(aClass, @selector(initialize)))
                // Avoid a compiler warning
                objc_msgSend(aClass, @selector(initialize));
            [pool release];
        }


        for (impIndex = 0; impIndex < impCount; impIndex++) {
            NSAutoreleasePool *pool;

            pool = [[NSAutoreleasePool alloc] init];
#if 0 && defined(DEBUG)
            fprintf(stderr, "Calling (0x%08x) ... ", imps[impIndex]);
#endif
            imps[impIndex](aClass, selectorToCall);
#if 0 && defined(DEBUG)
            fprintf(stderr, "done\n");
#endif
            [pool release];
        }
    }

    NSZoneFree(NULL, imps);

    return impCount != 0;
}


+ (void) _preventBundleInitialization;
{
    Class bundleClass;

    bundleClass = objc_getClass("NSBundle");
    NSBundleInitializeMethod = class_getClassMethod(bundleClass, @selector(initialize));
    originalNSBundleInitializeImplementation = NSBundleInitializeMethod->method_imp;
    NSBundleInitializeMethod->method_imp = (IMP)_NSBundleInitializationTooEarly;
}

+ (void) _provokeBundleInitialization;
{
    //time_t t;
#ifndef WIN32
    IMP oldImp;
    Class pathStoreClass;
#endif

    // Replace the original +[NSBundle initialize] implementation
    NSBundleInitializeMethod->method_imp = originalNSBundleInitializeImplementation;

    // Turn off symlink resolution in NSPathStore2.  This causes the
    // seearch through the NSBundles to be vastly faster.

    // CHRIS Kane says this is necessary on NT to get the correct
    // case in the file names since they are used as keys in a
    // dictionary.
#ifndef WIN32
    // Prevent NSPathStore2 from actually resolving symlinks.
    // Don't provoke +initialize
    pathStoreClass = objc_getClass("NSPathStore2");
    oldImp = OBReplaceMethodImplementationWithSelector(pathStoreClass,
                                                       @selector(stringByResolvingSymlinksInPath),
                                                       @selector(replacement_stringByResolvingSymlinksInPath));
#endif

    //t = time(NULL);
    //fprintf(stderr, "%s: Intializing NSBundle...\n", ctime(&t));
    [NSBundle initialize];
    //t = time(NULL);
    //fprintf(stderr, "%s: Done initializing NSBundle.\n", ctime(&t));

#ifndef WIN32
    // Put the method back in case someone really needs to resolve symlinks
    OBReplaceMethodImplementation(pathStoreClass, @selector(stringByResolvingSymlinksInPath), oldImp);
#endif
}

@end
