; -*-Emacs-Lisp-*-
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;
; File:         gnus-mailapp.el
; RCS:          /usr/local/sources/CVS/mailapp-utilities/gnus-mailapp.el,v 1.1.1.1 1996/11/24 14:56:15 tom Exp
; Description:  Make gnus work with Mail.app
; Author:       Carl Edman
; Created:      Sat Jun  5 09:49:56 1993
; Modified:     Tue May 31 16:17:51 1994 (Carl Edman) cedman@capitalist.princeton.edu
; Language:     Emacs-Lisp
; Package:      N/A
; Status:       Experimental (Do Not Distribute)
;
; (C) Copyright 1993, but otherwise this file is perfect freeware.
;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;
; This package requires Emacs 19 and Gnus 3.15+
;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;
; Load this package after loading gnus.el. You could automate that by
; adding this line to your ~/.emacs file:
;
;   (eval-after-load "gnus" '(load-library "gnus-mailapp"))
;
; To automatically mail yourself all articles you post, add these lines:
;
;   (add-hook 'gnus-inews-article-hook 'gnus-author-copy-save-in-nextmail)
;
; To use Mail.app as the default mailer inside gnus, add these lines:
;
;   (setq gnus-mail-reply-method 'gnus-mail-reply-using-nextmail
;         gnus-mail-forward-method 'gnus-mail-forward-using-nextmail
;         gnus-mail-other-window-method 'gnus-mail-other-window-using-nextmail)
;
; To have signatures which depend on the newsgroup:
;
;   (add-hook 'news-setup-hook 'gnus-insert-newsgroup-signature)
;
; To add a note to mail sent to authors of articles (when
; gnus-auto-mail-to-author is set):
;
;   (setq gnus-mail-send-method (function gnus-mail-send-with-note)
;         gnus-auto-mail-to-author-note "Text to use\n\n")
;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(require 'gnus)

(defun gnus-summary-save-in-directory (&optional dirname)
  "Save article in directory."
  (interactive)
  (gnus-summary-select-article
   (not (null gnus-save-all-headers)) gnus-save-all-headers)
  (gnus-eval-in-buffer-window gnus-article-buffer
    (save-excursion
      (save-restriction
        (widen)
        (let ((filename (concat dirname (int-to-string (gnus-header-number gnus-current-headers)))))
          (gnus-make-directory (file-name-directory filename))
          (gnus-output-to-file filename))))))

(defun gnus-summary-save-all-articles (&optional dir)
  "Save all articles in the current summary buffer which haven't been marked
in the specified directory."
  (interactive "FSave all articles in directory: ")
  (if (not (string-match "/$" dir))
      (setq dir (concat dir "/")))
  (goto-char (point-min))
  (gnus-execute "Subject" ".*" '(gnus-summary-save-in-directory dir) nil t))

(defun gnus-author-copy-save-in-nextmail (&optional file-name)
  "Mail yourself all articles which you post."
  (let ((newsgroups (gnus-fetch-field "newsgroups"))
        (artbuf (current-buffer))
        (tmpbuf (get-buffer-create " tmp-art")))
    (save-excursion
      (set-buffer tmpbuf)
      (erase-buffer)
      (insert-buffer-substring artbuf)
      (goto-char (point-min))
      (if newsgroups
          (cond
           ((re-search-forward "^\\(From: .*\\)(.*)\\(.*\\)$" nil t) 
            (replace-match (concat "\\1(To: " newsgroups ")\\2")))
           ((re-search-forward "^\\(From: .*\\)$" nil t)
            (replace-match (concat "\\1 (To: " newsgroups ")")))
           ((re-search-forward "\n\n")
            (replace-match (concat "\nFrom: " (user-login-name) " (To: " newsgroups ")\n\n")))))
      (call-process-region (point-min) (point-max) sendmail-program nil nil nil (user-login-name))
      (kill-buffer tmpbuf))))

(defun gnus-summary-save-in-nextmail ()
  "Mail yourself the current article."
  (interactive)
  (gnus-summary-select-article
   (not (null gnus-save-all-headers)) gnus-save-all-headers)
  (gnus-eval-in-buffer-window gnus-article-buffer
    (save-excursion
      (save-restriction
        (widen)
        (call-process-region (point-min) (point-max) sendmail-program nil nil nil (user-login-name))))))

(defun gnus-mail-reply-using-nextmail (&optional yank)
  "Reply to current article in Mail.app."
  (save-excursion (save-restriction
    (widen)
    (let* ((to (or (gnus-fetch-field "reply-to") (gnus-fetch-field "from") ""))
           (rsubject (or (gnus-fetch-field "subject") ""))
           (subject (if (string-match "^[Rr][Ee]:.+$" rsubject)
                        rsubject (concat "Re: " rsubject)))
           (text (save-excursion (goto-char (point-min))
                                 (search-forward "\n\n" nil 'move)
                                 (point)))
           (start (point-min))
           (end (if yank (point-max) text)))
      (call-process-region start end "nextmail" nil nil nil "-s" subject "-e" to))))
  (gnus-article-show-summary))

(defun gnus-mail-forward-using-nextmail (&optional yank)
  "Forward current article in Mail.app."
  (save-excursion (save-restriction
    (widen)
    (let* ((to (read-string "To: "))
           (rsubject (or (gnus-fetch-field "subject") ""))
           (subject (concat "[" gnus-newsgroup-name "] " rsubject))
           (start) (end) (artbuf) (tmpbuf))
      (setq artbuf (current-buffer)
            tmpbuf (get-buffer-create " tmp-art"))
      (set-buffer tmpbuf)
      (erase-buffer)
      (insert "------- Start of forwarded message -------\n")
      (insert-buffer-substring artbuf)
      (insert "------- End of forwarded message -------\n")
      (setq start (point-min)
            end (point-max))
      (call-process-region start end "nextmail" nil nil nil "-s" subject "-e" to)
      (kill-buffer tmpbuf))))
  (gnus-article-show-summary))

(defun gnus-mail-other-window-using-nextmail (&optional yank)
  "Send mail in Mail.app."
  (save-excursion (save-restriction
    (widen)
    (let* ((to (read-string "To: "))
           (subject (read-string "Subject: " (gnus-fetch-field "subject")))
           (text (save-excursion (goto-char (point-min))
                                 (search-forward "\n\n" nil 'move)
                                 (point)))
           (start (point-min))
           (end (if yank (point-max) text)))
      (call-process-region start end "nextmail" nil nil nil "-s" subject "-e" to))))
  (gnus-article-show-summary))

(defun gnus-insert-newsgroup-signature ()
  "Use in news-setup-hook to add newsgroup specific signature."
  (save-excursion
    (cond
     ((file-exists-p (concat "~/News/" gnus-newsgroup-name ".SIG"))
      (goto-char (point-max))
      (insert "--\n")
      (insert-file-contents (concat "~/News/" gnus-newsgroup-name ".SIG")))
     ((file-exists-p "~/News/SIG")
      (goto-char (point-max))
      (insert "--\n")
      (insert-file-contents "~/News/SIG")))))

(defvar gnus-auto-mail-to-author-note
  "[NOTE: This is an automatically generated copy of a posted article]\n\n"
  "Short string to prepend to mail sent to author if 'gnus-auto-mail-to-author'.")

(defun gnus-mail-send-with-note ()
  "Add note to mail sent to author if 'gnus-auto-mail-to-author'."
  (goto-char (point-min))
  (search-forward (concat "\n" mail-header-separator "\n"))
  (let ((p (point)))
    (if gnus-auto-mail-to-author-note
        (insert gnus-auto-mail-to-author-note))
    (funcall send-mail-function)
    (delete-region p (point))))

(define-key gnus-summary-mode-map "O" 'gnus-summary-save-all-articles)

