/*+++*
 *  title:	MailProxy.h
 *  abstract:	Interface of MailProxy for mailapp-utilities.
 *  author:	Tom Hageman <tom@basil.icce.rug.nl>
 *  created:	February 1998
 *  modified:	(see RCS Log at end)
 *  copyright:
 *
 *	Copyright (C) 1998  Tom R. Hageman, but otherwise perfect freeware.
 *
 *  description:
 *
 *      This module defines the following classes:
 *      - MailProxy
 *
 *	Abstract class.  Use factory methods
 *      (+classForMailer:... or +newForMailer:...)
 *      to get to (class, instance of) concrete subclass.
 *
 *---*/

#ifdef RCS_MailProxy_ID
static const char * const RCS_h_id = ((void)&RCS_h_id,
	"@(#)MailProxy.h,v 1.8 1998/12/04 19:55:09");
#endif

#import "compat.h"


@interface MailProxy : ROOT_OBJECT
{
   char *hostname;
   char *mailer;
   BOOL deliver;
   BOOL isConnected;
   BOOL isLocalHost;
}

// Implementation feature tests, may be overridden by subclasses.
+ (BOOL)canEditMessage;		// supports editing in compose window.
+ (BOOL)canDeliverMessage;	// supports direct delivery.
+ (BOOL)canIncorporateNewMail;	// supports incorporation in open mailbox.
+ (BOOL)canSendBcc;		// supports Bcc: recipients.
+ (BOOL)canSendArbitraryHeaders; // supports arbitrary headers.
   /* Default implementations return NO. */

+ (const char *)defaultMailer;
   /* Return name of default mailer; i.e., the value of `Mailer' in
      user defaults global domain, or "Mail" ("MailViewer" for Rhapsody)
      if that is unset. */

- initForMailer:(const char *)aMailer
   host:(const char *)aHostName
   forceLaunch:(BOOL)force
   directDelivery:(BOOL)direct;
      /* Designated initializer. */

#if NEXTSTEP
- (void)release;
- (void)dealloc;
#else
- free;
#endif

- (const char *)host;
- (const char *)mailer;
- (BOOL)isConnected;
- (BOOL)isLocalHost;

- (BOOL)connect:(BOOL)forceLaunch;
   /* Support method: if no connection is established yet,
      invoke -connectToMailer.  If that fails, and
      if `forceLaunch', invoke -launchMailer followed by -connectToMailer.
      If still no connection is established, finally invoke
      -retryConnectToMailer as a last resort. */

- (BOOL)connectToMailer;
   /* Abstract method; to be overridden by subclasses.
      -- do whatever is necessary to connect to mailer. */

- (BOOL)retryConnectToMailer;
   /* Abstract method: may be overridden by subclasses.
      -- do whatever is necessary to retry connection to mailer.
      -- default implementation is a no-op. */


- (BOOL)sendMailTo:(const char *)to
   subject:(const char *)subject
   body:(const char *)body
   cc:(const char *)cc
   bcc:(const char *)bcc;
   /* Abstract method; to be overridden by subclasses.
      OpenStep implementation just invokes -sendMailMessageTo:..., see below. */

- (BOOL)incorporateNewMail;
   /* Abstract method; may be overridden by subclasses.
      -- default implementation is a no-op. */

- (BOOL)launchMailer;
   /* Support: launch mailer application. */
- (BOOL)makeMailerActive;
   /* Support: make mailer the active application.
      -- to be used in subclass implementations of -sendMail(Message)To: */

@end // MailProxy


@interface MailProxy (Factory)

+ (Class)classForMailer:(const char *)aMailer
   directDelivery:(BOOL)direct
   incorporation:(BOOL)incorporate;
   /* Returns appropriate subclass for given mailer, delivery,
      and incorporation combination. */

+ newForMailer:(const char *)aMailer
   host:(const char *)aHostName
   forceLaunch:(BOOL)force
   directDelivery:(BOOL)direct
   incorporation:(BOOL)incorporate;
   /* Returns instance of appropriate subclass, capable of handling
      communication with requested mailer.
      NB. returns NON-autoreleased instance (OPENSTEP/RHAPSODY),
          for the sake of backward compatibility with NEXTSTEP. */

@end // MailProxy (Factory)


#if !NEXTSTEP

#  define MAILPROXY_TO		@"To"
#  define MAILPROXY_SUBJECT	@"Subject"
#  define MAILPROXY_CC		@"Cc"
#  define MAILPROXY_BCC		@"Bcc"

/* This category defines a more OPENSTEP-conformant API.
   These are mostly cover methods for the underlying NEXTSTEP-compatible
   base methods. */

@interface MailProxy (OpenStep)

+ classWithMailer:(NSString *)aMailer
   directDelivery:(BOOL)direct
   incorporation:(BOOL)incorporate;

+ mailProxyWithMailer:(NSString *)aMailer
   host:(NSString *)aHostName
   forceLaunch:(BOOL)force
   directDelivery:(BOOL)direct
   incorporation:(BOOL)incorporate;
   /* Similar to +newForMailer:..., but returns autoreleased instance. */

+ mailProxy;
   /* Simplified interface: default mailer, local host, force launch,
      open compose window (no direct delivery), no incorporation. */

- initWithMailer:(NSString *)aMailer
   host:(NSString *)aHostName
   forceLaunch:(BOOL)force
   directDelivery:(BOOL)direct;
   /* Similar to -initForMailer:... */

- (NSString *)hostName;
- (NSString *)mailerName;

- (BOOL)sendMailMessageTo:(NSString *)to
   subject:(NSString *)subject
   body:(id)body // NSString, NSAttributedString.
   cc:(NSString *)cc
   bcc:(NSString *)bcc;
   /* Similar to -sendMailTo:...  OpenStep subclasses should override
      _this_ method (or the next, if it supports arbitrary headers)
      instead of -sendMailTo:...
      The default implementation is to invoke -sendMailBody:headers:
      iff arbitrary headers are supported.
      Subclasses should be prepared to handle NSString and
      NSAttributedString body, at least. */

- (BOOL)sendMailBody:(id)body headers:(NSDictionary *)headers;
   /* Similar to -sendMailMessageTo:... but allows arbitrary headers
      to be specified.  OpenStep subclasses should override
      _this_ method when it desires to support arbitrary headers,
      instead of -sendMailTo:... or -sendMailMessageTo:.
      The default implementation is to invoke -sendMailMessage:...
      iff arbitrary headers are _not_ supported.
      Subclasses should be prepared to handle NSString and
      NSAttributedString body, at least. */

- (id)replacementObjectForBody:(id)body;
   /* Support method, can be used in -sendMailMessageTo:... implementations.
      Default implementation returns an NSString -- self for NSString body,
      [body string] for objects responding to `-string'
      (including NSAttributedString), and [body description] otherwise. */

@end // MailProxy (OpenStep)

#endif // !NEXTSTEP
