#import "TWCellDraw.h"
#ifdef WIN32
// I don't know where this thing is under windows
#define M_PI	3.14159265
#endif

/******************************************************************************
    TWCellDraw - by Jeff Martin
    
This file handles all of the drawing for the TWCell. Don't try to make sense of any of this.

Written by: Jeff Martin (jmartin@reportmill.com)
You may freely copy, distribute and reuse the code in this example.  
Don't even talk to me about warranties.
******************************************************************************/
#define EQUAL(a,b) (ABS((a)-(b))<0.00001)
#define NOTEQUAL(a,b) (ABS((a)-(b))>0.0001)
#define ISBETWEEN(x,a,b) (((x)>=(a))&&((x)<=(b)))
// A mod function for floating values
#define MOD(x,y) ((x) - (y)*(int)((float)(x)/(y)))
#define CLAMP_WITH_WRAP(a,x,y) \
( ((a) < (x)) ? ((y) - MOD(((x)-(a)),((y)-(x)))) : ( ((a) > (y)) ? ((x) + MOD(((a)-(y)),((y)-(x)))) : (a) ) )
#define EVEN(x) (!(((int)(x))%2))

@implementation TWCell(Draw)

- (void)drawInteriorWithFrame:(NSRect)frame inView:(NSView *)view
{
    NSRect insetFrame = NSInsetRect(frame, 2 , 2);	// Inset frame by 2 to allow for bezeled border
    float *pnts;                        // Used for user path of dashes
    char *ops;                            // Used for user path of dashes
    int pntCount, opCount;                // Used for user path of dashes
    float bbox[4] = { NSMinX(frame), NSMinY(frame), NSMaxX(frame), NSMaxY(frame) };
    
    // Draw the background
    if([self isLinear]) { NSDrawGrayBezel(frame , frame); [_color set]; NSRectFill(insetFrame); }
    
    // otherwise if([self isRadial])
    else {
        NSSize imageSize = {0,0};
        imageSize = [_image size];
        if(NOTEQUAL(imageSize.width,NSWidth(frame)) || NOTEQUAL(imageSize.height,NSHeight(frame)))
            [self generateThumbWheelBackgroundImageForRect:frame];
        [_image compositeToPoint:frame.origin operation:NSCompositeSourceOver];
    }
    
    // Get the userpath for the dashes
    [self getThumbWheelDashesForRect:insetFrame :&pnts :&pntCount :&ops :&opCount];

    // Draw dashes once for white part of groove
    if([self isHorizontal]) PStranslate(1,0); else PStranslate(0,-1);

    // Draw linear white dashes
    if([self isLinear]) {
        [[_color rgbColorScaledByFloatValue:1.5] set]; PSsetlinewidth(0);
        PSDoUserPath(pnts, pntCount, dps_float, ops, opCount, bbox, dps_ustroke);
    }
    
    // Break up radial white dashes to fade a little bit at ends
    else {
        int i = 0, j = 0, d = [self isHorizontal]? 0 : 1;
        float oneQuarterOfX = [self isHorizontal]? (NSMinX(insetFrame) + NSWidth(insetFrame)/4) :
            (NSMinY(insetFrame) + NSHeight(insetFrame)/4);
        float threeQuartersOfX =[self isHorizontal]? (NSMinX(insetFrame) + 3*NSWidth(insetFrame)/4) :
            (NSMinY(insetFrame) + 3*NSHeight(insetFrame)/4);

        [_color set]; PSsetlinewidth(0);
        while((i < pntCount) && (pnts[i+d] < oneQuarterOfX)) i += 4; j = i;
        if(i>0) PSDoUserPath(pnts, i, dps_float, ops, i/2, bbox, dps_ustroke);

        [[_color rgbColorScaledByFloatValue:1.5] set]; PSsetlinewidth(0);
        while((j < pntCount) && (pnts[j+d] < threeQuartersOfX)) j+=4;
        if(j>i) PSDoUserPath(&pnts[i], j-i, dps_float, &ops[i/2], j/2-i/2, bbox, dps_ustroke);

        [_color set]; PSsetlinewidth(0);
        if(pntCount>j) PSDoUserPath(&pnts[j], pntCount-j, dps_float, &ops[j/2], opCount-j/2, bbox, dps_ustroke);
    }
    
    if([self isHorizontal]) PStranslate(-1,0); else PStranslate(0,1);

    // Draw again for dark part of groove
    if([self isLinear]) [[_color rgbColorScaledByFloatValue:.5] set];
    else [[NSColor blackColor] set]; PSsetlinewidth(0);
    PSDoUserPath(pnts, pntCount, dps_float, ops, opCount, bbox, dps_ustroke);
    
    // If disabled then dim ThumbWheel out
    if(![self isEnabled]) {
        [[NSColor whiteColor] set]; PSsetalpha(.5);
        PScompositerect(NSMinX(insetFrame), NSMinY(insetFrame), NSWidth(insetFrame),
            NSHeight(insetFrame), NSCompositeSourceOver);
    }
    
    // Free user path variables, flush window
    free(pnts); free(ops);
}

- (void)getThumbWheelDashesForRect:(NSRect)rect :(float **)PNTS :(int *)PNTCOUNT :(char **)OPS :(int *)OPCOUNT
{
    // Get dashInterval (in pnts or degs depending on display mode) and shift
    float length = [self isVertical]? NSHeight(rect) : NSWidth(rect);
    int dashInt = [self isLinear]? _dashInterval : 360/(M_PI*length/[self dashInterval]);
    int shift = [self shift:rect];

    // Calculate how many dashes there will be and alloc space for pnts and ops
    int dashCount = 2 + ([self isLinear] ? length / dashInt : 180 / dashInt);
    float *pnts = malloc(sizeof(float)*dashCount*4); // (moveto+lineto)*(x+y)=4
    char *ops = malloc(sizeof(char)*dashCount*2);  // (moveto+lineto) = 2
    int i=0, j=0;
    
    // Calculate dash sizes
    int dashBase = [self isVertical] ? NSMinX(rect) : NSMinY(rect);
    int dashHeight = [self isVertical] ? NSWidth(rect) : NSHeight(rect);
    int dashMinTop = dashBase + dashHeight*.25, dashMajTop = dashBase + dashHeight*.5, dashTop = dashBase + dashHeight;

    float base = [self isVertical] ? NSMinY(rect) : NSMinX(rect);
    float width = [self isVertical]? NSHeight(rect) : NSWidth(rect), halfWidth = width/2;
    float mid = base + halfWidth, top = base + width;

    float mainDash;
    float x;
    
    // Calculate whether first dash is a major one. 
    BOOL isMajor = (shift>=0)? EVEN(shift/dashInt) : !EVEN(shift/dashInt);
        
    // Calculate Linear dashes
    if([self isLinear]) {

        // Set Main dash
        mainDash = base + shift;

        // Calculate starting point and set the dashes
        x = base+CLAMP_WITH_WRAP(shift,0,dashInt)%((shift>=0)? dashInt:999999);
        if([self isVertical]) while(x<top) {
            pnts[i++] = dashBase; pnts[i++] = x; pnts[i++] = isMajor ? dashMajTop : dashMinTop;
            if(EQUAL(x, mainDash)&&[self showMainDash]) pnts[i-1] = dashTop;
            pnts[i++] = x; x += dashInt; isMajor = !isMajor;
        }
        else while(x<top) {
            pnts[i++] = x; pnts[i++] = dashBase; pnts[i++] = x; pnts[i++] = isMajor ? dashMajTop : dashMinTop;
            if(EQUAL(x, mainDash)&&[self showMainDash]) pnts[i-1] = dashTop;
            x += dashInt; isMajor = !isMajor;
        }
    }

    // Calculate Radial Dashes
    else {

        // This is used to convert the degrees of a dash to a location(in pnts)
        float linDash = 0;
        
        // Inset dash size for beveled edges
        dashBase++; dashTop--;
        
        // Calc Main dash if we show it and it is in sight
        mainDash = mid - cos(shift*M_PI/180)*halfWidth;

        // Calculate the starting point and set the dashes
        x = CLAMP_WITH_WRAP(shift, 0, dashInt)%((shift>=0)? dashInt : 999999);
        if([self isVertical]) while(x<180) {
            linDash = mid - cos(x*M_PI/180)*halfWidth;
            pnts[i++] = dashBase; pnts[i++] = linDash; pnts[i++] = isMajor ? dashMajTop : dashMinTop;

            // Check to see if this is a valid main dash
            if(isMajor && EQUAL(linDash,mainDash) && [self showMainDash] && ISBETWEEN(shift, 0, 180)) pnts[i-1]=dashTop;
            pnts[i++] = linDash;
            x += dashInt; isMajor = !isMajor;
        }

        else while(x<180) {
            linDash = mid - cos(x*M_PI/180)*halfWidth;
            pnts[i++] = linDash; pnts[i++] = dashBase; pnts[i++] = linDash; pnts[i++] = isMajor ? dashMajTop : dashMinTop;

            // Check to see if this is a valid main dash
            if(isMajor && EQUAL(linDash,mainDash) && [self showMainDash] && ISBETWEEN(shift, 0, 180)) pnts[i-1]=dashTop;
            x += dashInt; isMajor = !isMajor;
        }
    }
        
    // fill the ops array with dps_moveto and dps_lineto
    while(j<i/2) { ops[j++] = dps_moveto; ops[j++] = dps_lineto; }
     
    // Set the passed in pointers to the arrays and the counts
    *PNTS = pnts; *OPS = ops; *PNTCOUNT = i; *OPCOUNT = j;
}

- (void)generateThumbWheelBackgroundImageForRect:(NSRect)rect
{
    NSRect topRect, middleRect = NSInsetRect(rect, 2, 2), bottomRect;
    NSColor *rgbColor = [_color colorUsingColorSpaceName:NSCalibratedRGBColorSpace];
    NSString *imageName;
    
    // Decrement last image's ref count (freeing it if ref count < 1)
    [_image release];
    
    // Generate imageName
    imageName = [NSString stringWithFormat:@"%@_%dx%d_Red%d_Green%d_Blue%d",
        [self isVertical]? @"Vertical" : @"Horizontal", (int)NSWidth(rect), (int)NSHeight(rect),
        (int)([rgbColor redComponent]*255), (int)([rgbColor greenComponent]*255), (int)([rgbColor blueComponent]*255)];
    
    // Try to find new image (returning if successful)
    if(_image = [NSImage imageNamed:imageName]) { [_image retain]; return; }
    
    // Allocate a new image, name it, size it & enter into the image hashtable
    _image = [[NSImage allocWithZone:[self zone]] initWithSize:rect.size]; [_image setName:imageName];
    
    // Lock Focus and draw the bezel around the image
    [_image lockFocus]; NSDrawGrayBezel(rect , rect);

    // Draw the top/left 2 pixels at 1.5 times given color
    NSDivideRect(middleRect , &topRect , &middleRect , 2, [self isVertical]? NSMinXEdge: NSMaxYEdge);
    [self drawThumbWheelGradationInRect:topRect color:[_color rgbColorScaledByFloatValue:1.5]];
    
    // Draw bottom/right two pixels at .5 given color    
    NSDivideRect(middleRect , &bottomRect , &middleRect , 2, [self isVertical]? NSMaxXEdge : NSMinYEdge);
    [self drawThumbWheelGradationInRect:bottomRect color:[_color rgbColorScaledByFloatValue:.5]];

    // Draw middle at given color
    [self drawThumbWheelGradationInRect:middleRect color:_color];
    
    // Unlock Focus
    [_image unlockFocus];
}

- (void)drawThumbWheelGradationInRect:(NSRect)rect color:(NSColor *)baseColor
{
    int thumbWheelLength = [self isHorizontal]? NSWidth(rect) : NSHeight(rect);
    NSColor *rgbBaseColor = [baseColor colorUsingColorSpaceName:NSCalibratedRGBColorSpace];
    int r = 255*[rgbBaseColor redComponent], g = 255*[rgbBaseColor greenComponent], b = 255*[rgbBaseColor blueComponent];
    float radius = (thumbWheelLength-1)/2.0, radiusSquared = radius*radius;

    int bmirWidth = [self isHorizontal]? NSWidth(rect) : 1;
    int bmirHeight = [self isHorizontal]? 1 : NSHeight(rect);
    int bmirRowBytes = [self isHorizontal]? bmirWidth*3 : 3;
    NSBitmapImageRep *bmir = [[NSBitmapImageRep alloc] initWithBitmapDataPlanes:NULL pixelsWide:bmirWidth
        pixelsHigh:bmirHeight bitsPerSample:8 samplesPerPixel:3 hasAlpha:NO isPlanar:NO
	colorSpaceName:NSDeviceRGBColorSpace bytesPerRow:bmirRowBytes bitsPerPixel:24];
    unsigned char *bmirData = [bmir bitmapData];

    // Fill bitmap image rep's data with color components for each point along the thumbWheelLength
    int i; for(i = 0; i < thumbWheelLength; i++) {

        // Calculate the height of the thumbwheel at current point
        float h = sqrt(radiusSquared - (radius-i)*(radius-i))/radius;
        
        // Get red, green and blue component of color (scaled for the height)
        bmirData[i*3] = r*h; bmirData[i*3+1] = g*h; bmirData[i*3+2] = b*h;
    }

    [bmir drawInRect:rect]; [bmir release];
}

- (NSString *)getInspectorClassName { return @"TWInspector"; }

@end

@implementation NSColor(Scale)

- (NSColor *)rgbColorScaledByFloatValue:(float)scale
{
    NSColor *rgbColor = [self colorUsingColorSpaceName:NSCalibratedRGBColorSpace];
    float newRed = [rgbColor redComponent]*scale, newGreen = [rgbColor greenComponent]*scale;
    float newBlue = [rgbColor blueComponent]*scale;
    if(newRed>1) newRed = 1; if(newGreen>1) newGreen = 1; if(newBlue>1) newBlue = 1;
    return [NSColor colorWithCalibratedRed:newRed green:newGreen blue:newBlue alpha:1];
}

@end
