// TEPipeCommand.m
// TextExtras - Yellow Box
//
// Copyright 1996-1999, Mike Ferris.
// All rights reserved.

#import "TEPipeCommand.h"

#define COMMAND_KEY @"Command"
#define INPUTSOURCE_KEY @"InputSource"
#define OUTPUTDEST_KEY @"OutputDestination"
#define KEYEQUIVALENT_KEY @"KeyEquivalent"

@implementation TEPipeCommand

- (id)initWithCommandString:(NSString *)command standardInputSource:(TEStandardInputSource)inSrc standardOutputDestination:(TEStandardOutputDestination)outDest keyEquivalent:(NSString *)keyEquiv {
    self = [super init];
    if (self) {
        commandString = [command copyWithZone:[self zone]];
        inputSource = inSrc;
        outputDestination = outDest;
        keyEquivalent = (keyEquiv ? [keyEquiv copyWithZone:[self zone]] : @"");
    }
    return self;
}

- (id)initWithDictionary:(NSDictionary *)dict {
    NSString *command = [dict objectForKey:COMMAND_KEY];
    NSString *inSrc = [dict objectForKey:INPUTSOURCE_KEY];
    NSString *outDst = [dict objectForKey:OUTPUTDEST_KEY];
    NSString *keyEquiv = [dict objectForKey:KEYEQUIVALENT_KEY];

    if (!command || !inSrc || !outDst) {
        [self dealloc];
        return nil;
    }
    return [self initWithCommandString:command standardInputSource:(TEStandardInputSource)[inSrc intValue] standardOutputDestination:(TEStandardOutputDestination)[outDst intValue] keyEquivalent:keyEquiv];
}

- (id)copyWithZone:(NSZone *)zone {
    return [[[self class] allocWithZone:[self zone]] initWithCommandString:commandString standardInputSource:inputSource standardOutputDestination:outputDestination  keyEquivalent:keyEquivalent];
}

- (void)dealloc {
    [commandString release];
    [keyEquivalent release];
    [super dealloc];
}

- (BOOL)isEqual:(id)otherObj {
    if (otherObj == self) {
        return YES;
    }
    if ([otherObj isKindOfClass:[TEPipeCommand class]] && (((TEPipeCommand *)otherObj)->inputSource == self->inputSource) && (((TEPipeCommand *)otherObj)->outputDestination == self->outputDestination) && ((((TEPipeCommand *)otherObj)->commandString == self->commandString) || ([((TEPipeCommand *)otherObj)->commandString isEqualToString:self->commandString]))) {
        return YES;
    }
    return NO;
}

- (NSDictionary *)dictionaryRepresentation {
    return [NSDictionary dictionaryWithObjectsAndKeys:commandString, COMMAND_KEY, [NSString stringWithFormat:@"%d", (int)inputSource], INPUTSOURCE_KEY, [NSString stringWithFormat:@"%d", (int)outputDestination], OUTPUTDEST_KEY, keyEquivalent, KEYEQUIVALENT_KEY, nil];
}

- (NSString *)description {
    char inChar, outChar;

    inChar = ((inputSource == NoInput) ? 'N' : ((inputSource == SelectionInput) ? 'S' : 'T'));
    outChar = ((outputDestination == OutputReplacesSelection) ? 'S' : ((outputDestination == OutputReplacesCompleteText) ? 'T' : ((outputDestination == OutputToSeparateWindow) ? 'W' : 'D')));
    return [NSString stringWithFormat:@"(%c-%c) %@", inChar, outChar, commandString];
}

// This method and the little notification method following implement synchronously running a task with input piped in from and string and output piped back out and returned as a string.   They require only a _stdoutData instance variable to function.
- (NSString *)executeBinary:(NSString *)executablePath inDirectory:(NSString *)currentDirPath withArguments:(NSArray *)args environment:(NSDictionary *)env inputString:(NSString *)input {
    NSTask *task;
    NSPipe *inputPipe;
    NSPipe *outputPipe;
    NSFileHandle *inputFileHandle;
    NSFileHandle *outputFileHandle;
    NSString *output = nil;
    
    task = [[NSTask allocWithZone:[self zone]] init];
    [task setLaunchPath:executablePath];
    if (currentDirPath) {
        [task setCurrentDirectoryPath:currentDirPath];
    }
    if (args) {
        [task setArguments:args];
    }
    if (env) {
        [task setEnvironment:env];
    }

    inputPipe = [NSPipe pipe];
    inputFileHandle = [inputPipe fileHandleForWriting];
    [task setStandardInput:inputPipe];
    outputPipe = [NSPipe pipe];
    outputFileHandle = [outputPipe fileHandleForReading];
    [task setStandardOutput:outputPipe];

    [task launch];

    [[NSNotificationCenter defaultCenter] addObserver:self selector:@selector(stdoutNowAvailable:) name:NSFileHandleReadToEndOfFileCompletionNotification object:outputFileHandle];
    [outputFileHandle readToEndOfFileInBackgroundAndNotifyForModes:[NSArray arrayWithObject:@"TE_SpecialPipeServiceRunLoopMode"]];

    if (input) {
        [inputFileHandle writeData:[input dataUsingEncoding:[NSString defaultCStringEncoding] allowLossyConversion:YES]];
    }
    [inputFileHandle closeFile];
    
    // Now loop the runloop in the special mode until we've processed the notification.
    _stdoutData = nil;
    while (_stdoutData == nil) {
        // Run the run loop, briefly, until we get the notification...
        [[NSRunLoop currentRunLoop] runMode:@"TE_SpecialPipeServiceRunLoopMode" beforeDate:[NSDate dateWithTimeIntervalSinceNow:1.0]];
    }
    [[NSNotificationCenter defaultCenter] removeObserver:self name:NSFileHandleReadToEndOfFileCompletionNotification object:outputFileHandle];

    output = [[NSString allocWithZone:[self zone]] initWithData:_stdoutData encoding:[NSString defaultCStringEncoding]];
    [_stdoutData release];
    _stdoutData = nil;
        
    return [output autorelease];
}

- (void)stdoutNowAvailable:(NSNotification *)notification {
    // This is the notification method that executeBinary:inDirectory:withArguments:environment:inputString: registers to get called when all the data has been read. It just grabs the data and stuffs it in an ivar.  The setting of this ivar signals the main method that the output is complete and available.
    NSData *outputData = [[notification userInfo] objectForKey:NSFileHandleNotificationDataItem];
    _stdoutData = (outputData ? [outputData retain] : [[NSData allocWithZone:[self zone]] init]);
}


- (NSString *)runWithInputString:(NSString *)input {
    // Takes a string which it parses as a command with optional arguments, finds the command, then calls executeBinary:inDirectory:withArguments:environment:inputString: to actually run the command and returns the output.
    NSCharacterSet *whitespaceSet = [NSCharacterSet whitespaceCharacterSet];
    NSScanner *commandScanner;
    NSString *executablePath;
    NSMutableArray *args = [NSMutableArray array];
    NSString *scanString = nil;
    NSString *output = nil;
    
    commandScanner = [[NSScanner allocWithZone:[self zone]] initWithString:commandString];
    [commandScanner scanUpToCharactersFromSet:whitespaceSet intoString:&scanString];
    executablePath = scanString;
    while (![commandScanner isAtEnd]) {
        [commandScanner scanUpToCharactersFromSet:whitespaceSet intoString:&scanString];
        [args addObject:scanString];
    }
    [commandScanner release];

    //NSLog(@"TextExtras: executeCommand: parsed executablePath: '%@' and arguments: %@", executablePath, args);

    if (executablePath && ![executablePath isEqualToString:@""]) {
        NSFileManager *fileManager = [NSFileManager defaultManager];
        BOOL foundExecutable = NO;
        
        if (![executablePath isAbsolutePath]) {
            // If the path is not absolute we'll try looking relative to all directories in the PATH environment variable.
            NSDictionary *env = [[NSProcessInfo processInfo] environment];
            NSString *pathEnv;
            NSArray *pathArray;
            unsigned i, c;
            NSString *tempPath;
#ifdef WIN32
            BOOL hasExecutableExtension = [[executablePath pathExtension] isEqualToString:@"exe"];
#endif

            pathEnv = [env objectForKey:@"PATH"];
            if (!pathEnv) {
                pathEnv = [env objectForKey:@"Path"];
            }
#ifdef WIN32
            pathArray = [pathEnv componentsSeparatedByString:@";"];
#else
            pathArray = [pathEnv componentsSeparatedByString:@":"];
#endif
            c = [pathArray count];
            for (i=0; i<c; i++) {
                tempPath = [[pathArray objectAtIndex:i] stringByAppendingPathComponent:executablePath];
                if ([fileManager isExecutableFileAtPath:tempPath]) {
                    executablePath = tempPath;
                    foundExecutable = YES;
                    break;
#ifdef WIN32
                } else if (!hasExecutableExtension) {
                    tempPath = [tempPath stringByAppendingPathExtension:@"exe"];
                    if ([fileManager isExecutableFileAtPath:tempPath]) {
                        executablePath = tempPath;
                        foundExecutable = YES;
                        break;
                    }
#endif
                }
            }
        } else {
#ifdef WIN32
            BOOL hasExecutableExtension = [[executablePath pathExtension] isEqualToString:@"exe"];
#endif
            if ([fileManager isExecutableFileAtPath:executablePath]) {
                foundExecutable = YES;
            }
#ifdef WIN32
            if (!hasExecutableExtension) {
                NSString *tempPath = [executablePath stringByAppendingPathExtension:@"exe"];
                if ([fileManager isExecutableFileAtPath:tempPath]) {
                    executablePath = tempPath;
                    foundExecutable = YES;
                }
            }
#endif
        }
        
        if (foundExecutable) {
            output = [self executeBinary:executablePath inDirectory:[executablePath stringByDeletingLastPathComponent] withArguments:args environment:nil inputString:input];
        } else {
            NSRunAlertPanel(NSLocalizedStringFromTableInBundle(@"Error", @"TextExtras", [NSBundle bundleForClass:[self class]], @"Title for error alerts"), NSLocalizedStringFromTableInBundle(@"The program you specified ('%@'), either does not exist or is not an executable program.", @"TextExtras", [NSBundle bundleForClass:[self class]], @"Error string"), NSLocalizedStringFromTableInBundle(@"OK", @"TextExtras", [NSBundle bundleForClass:[self class]], @"OK button title."), nil, nil, executablePath);
        }
    } else {
        NSRunAlertPanel(NSLocalizedStringFromTableInBundle(@"Error", @"TextExtras", [NSBundle bundleForClass:[self class]], @"Title for error alerts"), NSLocalizedStringFromTableInBundle(@"No program specified.  You must enter a command to execute.", @"TextExtras", [NSBundle bundleForClass:[self class]], @"Error string"), NSLocalizedStringFromTableInBundle(@"OK", @"TextExtras", [NSBundle bundleForClass:[self class]], @"OK button title."), nil, nil);
    }

    return output;
}

- (BOOL)runWithTextView:(NSTextView *)textView {
    NSString *input;
    NSString *output;

    if (inputSource == SelectionInput) {
        input = [[textView string] substringWithRange:[textView selectedRange]];
    } else if (inputSource == CompleteTextInput) {
        input = [textView string];
    } else {
        input = nil;
    }

    output = [self runWithInputString:input];

    if (output) {
        if ((outputDestination == OutputReplacesSelection) || (outputDestination == OutputReplacesCompleteText)) {
            NSRange charRange = [textView rangeForUserTextChange];
            if (charRange.location != NSNotFound) {
                if (outputDestination == OutputReplacesCompleteText) {
                    charRange = NSMakeRange(0, [[textView string] length]);
                }
                if (output && [textView shouldChangeTextInRange:charRange replacementString:output]) {
                    [[textView textStorage] replaceCharactersInRange:charRange withString:output];
                    [textView setSelectedRange:NSMakeRange(charRange.location, [output length])];
                    [textView didChangeText];
                }
            } else {
                NSBeep();
            }
        } else if (outputDestination == OutputToSeparateWindow) {
            NSRunAlertPanel(NSLocalizedStringFromTableInBundle(@"Command Output", @"TextExtras", [NSBundle bundleForClass:[self class]], @"Title for command output alert"), @"%@", NSLocalizedStringFromTableInBundle(@"OK", @"TextExtras", [NSBundle bundleForClass:[self class]], @"OK button title."), nil, nil, output);
        }

    }
    return ((output != nil) ? YES : NO);
}

@end
