// TEGotoPanelController.m
// TextExtras - Yellow Box
//
// Copyright 1996-1999, Mike Ferris.
// All rights reserved.

#import "TEGotoPanelController.h"
#import "TETextUtils.h"

enum {
    CharacterRanges = 0,
    LineRanges = 1,
};

#define GotoPanelAutoupdatesDefault @"GotoPanelAutoupdates"
#define GotoPanelShowsLineRangesDefault @"GotoPanelShowsLineRanges"

static BOOL GotoPanelAutoupdates() {
    static BOOL GotoPanelAutoupdates = NO;
    static BOOL ReadDefault = NO;
    if (!ReadDefault) {
        id val = [[NSUserDefaults standardUserDefaults] objectForKey:GotoPanelAutoupdatesDefault];
        if (val && ([val hasPrefix:@"y"] || [val hasPrefix:@"Y"])) {
            GotoPanelAutoupdates = YES;
        }
        ReadDefault = YES;
    }
    return GotoPanelAutoupdates;
}

static BOOL GotoPanelShowsLineRanges() {
    static BOOL GotoPanelShowsLineRanges = YES;
    static BOOL ReadDefault = NO;
    if (!ReadDefault) {
        id val = [[NSUserDefaults standardUserDefaults] objectForKey:GotoPanelShowsLineRangesDefault];
        if (!val || (([val hasPrefix:@"y"] || [val hasPrefix:@"Y"]))) {
            GotoPanelShowsLineRanges = YES;
        } else {
            GotoPanelShowsLineRanges = NO;
        }
        ReadDefault = YES;
    }
    return GotoPanelShowsLineRanges;
}

@implementation TEGotoPanelController

+ (id)sharedGotoPanelController {
    static TEGotoPanelController *sharedInstance = nil;
    if (!sharedInstance) {
        sharedInstance = [[TEGotoPanelController alloc] init];
    }
    return sharedInstance;
}

- (id)init {
    self = [super initWithWindowNibName:@"TEGotoPanel" owner:self];
    if (self) {
        [self setWindowFrameAutosaveName:@"TextExtrasGotoPanel"];
        autoupdateFlag = NO;
        autoupdateTextView = nil;
    }
    return self;
}

- (void)dealloc {
    if (autoupdateFlag) {
        [[NSNotificationCenter defaultCenter] removeObserver:self];
    }
    [super dealloc];
}

- (void)windowDidLoad {
    [super windowDidLoad];
    
    if (GotoPanelAutoupdates()) {
        autoupdateFlag = YES;
        [[NSNotificationCenter defaultCenter] addObserver:self selector:@selector(textViewSelectionChanged:) name:NSTextViewDidChangeSelectionNotification object:nil];
        [[NSNotificationCenter defaultCenter] addObserver:self selector:@selector(applicationWillUpdate:) name:NSApplicationWillUpdateNotification object:NSApp];
    } else {
        autoupdateFlag = NO;
    }
    [autoupdateCheckbox setState:autoupdateFlag];

    if (GotoPanelShowsLineRanges()) {
        [radioButtons selectCell:[radioButtons cellWithTag:LineRanges]];
    } else {
        [radioButtons selectCell:[radioButtons cellWithTag:CharacterRanges]];
    }

    if (autoupdateFlag) {
        // Get the ball rolling.
        [self applicationWillUpdate:nil];
    }
}

- (void)changeRangeTypeAction:(id)sender {
    // action for Character/Line radio buttons.
    id val;
    
    if (autoupdateTextView) {
        [self updateFieldFromTextView:autoupdateTextView];
    }
    val = (([[radioButtons selectedCell] tag] == CharacterRanges) ? @"NO" : @"YES");
    [[NSUserDefaults standardUserDefaults] setObject:val forKey:GotoPanelShowsLineRangesDefault];
}

- (void)autoupdateAction:(id)sender {
    BOOL newVal = ([autoupdateCheckbox state] ? YES : NO);
    if (newVal != autoupdateFlag) {
        autoupdateFlag = newVal;
        if (autoupdateFlag) {
            [[NSNotificationCenter defaultCenter] addObserver:self selector:@selector(textViewSelectionChanged:) name:NSTextViewDidChangeSelectionNotification object:nil];
            [[NSNotificationCenter defaultCenter] addObserver:self selector:@selector(applicationWillUpdate:) name:NSApplicationWillUpdateNotification object:NSApp];
           
        } else {
            [[NSNotificationCenter defaultCenter] removeObserver:self];
        }
        [[NSUserDefaults standardUserDefaults] setObject:(autoupdateFlag ? @"YES" : @"NO") forKey:GotoPanelAutoupdatesDefault];
    }
}

// A range specification is either a single integer or two colon-separated integers indicating first and last elements in a range.
// A range specification is assumed to be "1" based and inclusive.  So the range specification "1:3" will translate to units 1, 2, and 3, or range {0, 3}.  The range specification "5" will be returned as {4, 1}.

static NSString *rangeSpecificationFromRange(NSRange range) {
    // Given a range, produce a range specification.
    if (range.length < 2) {
        return [NSString stringWithFormat:@"%u", (range.location + 1)];
    } else {
        return [NSString stringWithFormat:@"%u:%u", (range.location + 1), NSMaxRange(range)];  // NSMaxRange(range) is already one bigger than it should be so we don't need to adjust it.
    }
}

static NSRange rangeFromRangeSpecification(NSString *rangeSpec) {
    // Given a range specification (a string containing either a single number or two numbers separated by a colon), return a range.
    NSScanner *scanner = [NSScanner localizedScannerWithString:rangeSpec];
    NSRange range;
    unsigned endLoc;

    if (![scanner scanInt:&(range.location)]) {
        return NSMakeRange(NSNotFound, 0);
    }
    if ([scanner isAtEnd]) {
        range.location--;
        range.length = 1;
        return range;
    }

    if (![scanner scanString:@":" intoString:NULL] || ![scanner scanInt:&(endLoc)]) {
        return NSMakeRange(NSNotFound, 0);
    }

    range.length = (endLoc + 1) - range.location;
    range.location--;
    return range;
}

#define UNICHAR_BUFF_SIZE 1024

- (NSRange)characterRangeForLineNumberRange:(NSRange)lineNumRange inString:(NSString *)string {
    unsigned stopLineNum = NSMaxRange(lineNumRange);
    unsigned curLineNum = 0;
    unsigned startCharIndex = NSNotFound;
    unichar buff[UNICHAR_BUFF_SIZE];
    unsigned i, buffCount;
    NSRange searchRange = NSMakeRange(0, [string length]);

    // Returned char range should start at beginning of line number lineNumRange.location and end at beginning of line number stopLineNum.
    if (lineNumRange.location == 0) {
        // Check for this case first since the loop won't.
        startCharIndex = 0;
    }
    while (searchRange.length > 0) {
        buffCount = ((searchRange.length > UNICHAR_BUFF_SIZE) ? UNICHAR_BUFF_SIZE : searchRange.length);
        [string getCharacters:buff range:NSMakeRange(searchRange.location, buffCount)];
        for (i=0; i<buffCount; i++) {
            // We're counting paragraph separators here.  We want to notice when we hit lineNumRange.location and remember where the starting char index is.  We also want to notice when we reach the stopLineNum and return the result.
            if (TE_IsHardLineBreakUnichar(buff[i], string, searchRange.location + i)) {
                curLineNum++;
                if (curLineNum == lineNumRange.location) {
                    // The next line is the first line we need.
                    startCharIndex = searchRange.location + i + 1;
                }
                if (curLineNum == stopLineNum) {
                    return NSMakeRange(startCharIndex, (searchRange.location + i + 1) - startCharIndex);
                }
            }
        }
        // Skip the search range past the part we just did.
        searchRange.location += buffCount;
        searchRange.length -= buffCount;
    }

    // If we're here, we didn't find the end of the line number range.
    // searchRange.location == [string length] at this point.
    if (startCharIndex == NSNotFound) {
        // We didn't find the start of the line number range either, so return {EOT, 0}.
       return NSMakeRange(searchRange.location, 0);
    } else {
        // We found the start, so return from there to the end of the text.
        return NSMakeRange(startCharIndex, searchRange.location - startCharIndex);
    }
}

- (NSRange)lineNumberRangeFromCharacterRange:(NSRange )charRange inString:(NSString *)string {
    unsigned stopCharIndex = NSMaxRange(charRange);
    unsigned curLineNum = 0;
    unsigned startLineNum = NSNotFound;
    unichar buff[UNICHAR_BUFF_SIZE];
    unsigned i, buffCount;
    NSRange searchRange = NSMakeRange(0, [string length]);

    while (searchRange.length > 0) {
        buffCount = ((searchRange.length > UNICHAR_BUFF_SIZE) ? UNICHAR_BUFF_SIZE : searchRange.length);
        [string getCharacters:buff range:NSMakeRange(searchRange.location, buffCount)];
        for (i=0; i<buffCount; i++) {
            // We're counting paragraph separators here.  We want to notice when we hit charRange.location and remember what the line number is.  We also want to notice when we reach the stopCharIndex and return the result.
            if (charRange.location == searchRange.location + i) {
                startLineNum = curLineNum;
                if (stopCharIndex == charRange.location) {
                    return NSMakeRange(startLineNum, 1);
                }
            }
            if (stopCharIndex == searchRange.location + i) {
                unsigned stopLineNum;
                if ((searchRange.location + i > 0) && TE_IsHardLineBreakUnichar([string characterAtIndex:searchRange.location + i - 1], string, searchRange.location + i - 1)) {
                    stopLineNum = curLineNum;
                } else {
                    stopLineNum = curLineNum + 1;
                }
                return NSMakeRange(startLineNum, stopLineNum - startLineNum);
            }
            if (TE_IsHardLineBreakUnichar(buff[i], string, searchRange.location + i)) {
                curLineNum++;
            }
        }
        // Skip the search range past the part we just did.
        searchRange.location += buffCount;
        searchRange.length -= buffCount;
    }

    // If we're here, we didn't find the end of the line number range.
    // curLineNum == number of last line at this point.
    if (startLineNum == NSNotFound) {
        // We didn't find the start of the line number range either, so return {EOT, 0}.
        return NSMakeRange(curLineNum, 0);
    } else {
        // We found the start, so return from there to the end of the text.
        unsigned stopLineNum;
        if ((searchRange.location > 0) && TE_IsHardLineBreakUnichar([string characterAtIndex:searchRange.location - 1], string, searchRange.location - 1)) {
            stopLineNum = curLineNum;
        } else {
            stopLineNum = curLineNum + 1;
        }
        return NSMakeRange(startLineNum, stopLineNum - startLineNum);
    }
}

- (void)gotoAction:(id)sender {
    id targetWindow = [NSApp mainWindow];
    id firstResponder = [targetWindow firstResponder];
    if ([firstResponder isKindOfClass:[NSTextView class]]) {
        NSRange desiredRange = rangeFromRangeSpecification([textField stringValue]);
        unsigned textLen = [[firstResponder textStorage] length];
        
        if (desiredRange.location == NSNotFound) {
            // Invalid string format in text field.
            NSBeep();
            return;
        }
        
        if ([[radioButtons selectedCell] tag] == LineRanges) {
            desiredRange = [self characterRangeForLineNumberRange:desiredRange inString:[firstResponder string]];
        }

        // Adjust the range within the text if necessary.
        if (NSMaxRange(desiredRange) > textLen) {
            if (desiredRange.location <= textLen) {
                desiredRange.length = textLen - desiredRange.location;
            } else {
                desiredRange = NSMakeRange(textLen, 0);
            }
        }

        [firstResponder setSelectedRange:desiredRange];
        [firstResponder scrollRangeToVisible:desiredRange];
        [targetWindow makeKeyWindow];
    } else {
        NSBeep();
    }
}

- (void)textFieldAction:(id)sender {
    // This is the action for the text field.  We want to click the button, but if the user hits return, we also want to dismiss the panel.
    [gotoButton performClick:self];
    [[self window] orderOut:self];
}

- (void)updateFieldFromTextView:(NSTextView *)textView {
    if ([self isWindowLoaded]) {
        BOOL byLine = [[radioButtons selectedCell] tag];
        NSRange range = [textView selectedRange];

        if (byLine) {
            range = [self lineNumberRangeFromCharacterRange:range inString:[textView string]];
        }

        [textField setStringValue:rangeSpecificationFromRange(range)];
    }
}

- (void)textViewSelectionChanged:(NSNotification *)notification {
    if ([notification object] == autoupdateTextView) {
        [self updateFieldFromTextView:autoupdateTextView];
    }
}

- (void)applicationWillUpdate:(NSNotification *)notification {
    NSWindow *targetWindow = [NSApp mainWindow];
    id target = [targetWindow firstResponder];
    if (target != autoupdateTextView) {
        if ([target isKindOfClass:[NSTextView class]] && ![target isFieldEditor]) {
            autoupdateTextView = [[target layoutManager] firstTextView];
            [self updateFieldFromTextView:autoupdateTextView];
        } else if (autoupdateTextView != nil) {
            autoupdateTextView = nil;
            [textField setStringValue:@""];
        }
    }
}

- (void)selectRangeText {
    [textField selectText:self];
}

@end
