/*
    File:       SillyBalls.m

    Contains:   View class for drawing lots of silly balls.

    Written by: Quinn "The Eskimo!"

    Created:    Thu 05-Jun-1997

    Copyright:  (c)1997 by Apple Computer, Inc., all rights reserved.

    Change History (most recent first):

    You may incorporate this sample code into your applications without
    restriction, though the sample code has been provided "AS IS" and the
    responsibility for its operation is 100% yours.  However, what you are
    not permitted to do is to redistribute the source as "DSC Sample Code"
    after having made changes. If you're going to re-distribute the source,
    we require that you make it clear in the source that the code was
    descended from Apple Sample Code, but that you've made changes.
*/

#import "SillyBallView.h"

@implementation SillyBallView

- (id)initWithFrame:(NSRect)frameRect
    // See comment in interface part.
{
    self = [super initWithFrame:frameRect];
    if (self != nil) {

        // Init some basic instance variables.
        
        running = NO;
        textString = @"Bo3b";
        textSize = [ textString sizeWithAttributes:nil ];

        // Initialise the other instance variables,
        // and start the drawintg timer.
        
        [self setTimerPeriod:3 source:nil];
    }
    return (self);
}

- (void)dealloc
    // See comment in interface part.
{
    [repeatTimer release];
    [super dealloc];
}

- (void)setTimerRunning:(BOOL)run source:(id)source
    // See comment in interface part.
{
    if ( !running && run ) {

        // If we're not running and we've been told to run, we'd
        // better start the timer.  Set it up so that it calls
        // our drawAnother: method periodically to draw a ball.
        // The timer period is based on the repeatPeriod instance
        // variable (as controlled by the user via the Ball Rate
        // slider, which we feed into a maths function (real_time_
        // period = 10 ** - ball_rate_slider) to get a pleasing
        // non-linearity.
        
        repeatTimer = [[NSTimer scheduledTimerWithTimeInterval:pow(10, -repeatPeriod)
                            target:self
                            selector:@selector(drawAnother:)
                            userInfo:nil
                            repeats:YES
            	      ] retain];
        running = YES;
    } else if (running && !run) {

        // We're running and we've been told not to, so let's stop.
        
        [repeatTimer invalidate];
        [repeatTimer release];
        repeatTimer = nil;
        running = NO;
    }

    // Now update menu item to reflect any changes of running status.
    
    if (startStopMenuItem != nil) {
        if (running) {
            [ startStopMenuItem setTitle:@"Stop"];
        } else {
            [ startStopMenuItem setTitle:@"Start"];
        }
    }

    // Now set the value of the user interface based on the current repeat
    // period.  Two things to note here.  Firstly, we conditionalise
    // this by a test whether the view is not nil because the first
    // time we run (called from initWithFrame:) the views have not
    // yet been set up properly because we're being dearchived out
    // of a NIB file.  While it doesn't hurt to call methods on
    // nil objects (it's defined as a no-op in Objective-C), I wanted
    // to explicitly highlight this edge case.
    //
    // The second thing to notice is that we don't set the view
    // if it was the user interface element that prompted the change
    // (presumably it's already changed to the appropriately value).
    // This doesn't have a big effect in this specific case, but you
    // could imagine it causing weird effects in other cases.

    if (sliderView != nil) {
        if (sliderView != source) {
            [sliderView setFloatValue:repeatPeriod];
        }
    }
    if (textView != nil) {
        if (textView != source) {
            [textView setFloatValue:repeatPeriod];
        }
    }
}

- (void)setTimerPeriod:(float)repPeriod source:(id)source
    // See comment in interface part.
{
    // Change the instance variable, then stop the repeat
    // timer (if it's running) and start it again with the
    // right period.
    
    repeatPeriod = repPeriod;
    if (running) {
        [self setTimerRunning:NO source:source];
    }
    [self setTimerRunning:YES source:source];
}

static float RandFloat(void)
    // Returns a random floating point number between
    // 0.0 and 1.0.
{
    return ((float) rand() / (float) RAND_MAX);
}

enum {
    kBallSize = 30
};

- (void)drawRandomBallInside:(const NSRect *)rect
    // See comment in interface part.
{
    float x, y;
    NSBezierPath *oval;

    // Calculate where the ball should go.
    
    x = rect->origin.x + rect->size.width * RandFloat();
    y = rect->origin.x + rect->size.height * RandFloat();

    // Set the current colour to a random RGB value.

    [[NSColor colorWithDeviceRed:RandFloat() green:RandFloat() blue:RandFloat() alpha:1.0] set];

    // Now construct a bezier path for an circle and draw it.
    
    oval = [NSBezierPath bezierPath];
    [oval appendBezierPathWithOvalInRect:NSMakeRect(x, y, kBallSize, kBallSize)];
    [oval fill];

    // Now set the current colour to black and draw
    // the text centred in the ball.
    
    [[NSColor blackColor] set];
    [textString drawAtPoint:NSMakePoint(x + kBallSize / 2 - textSize.width / 2,
                                        y) withAttributes:nil];
}

- (void)drawRect:(NSRect)rect
    // See comment in interface part.
{
    // Do nothing.  In a normal view, you would update the visual
    // representation of your view inside this method.  However, SillyBallView
    // does not remember which balls that it's drawn, so there is nothing it
    // can draw in this meathd.  Instead the drawing happens asynchronously
    // inside the drawAnother: method.  See the notes in ReadMe.rtf for details.
}

- (void)drawAnother:(id)timer
    // See comment in interface part.
{
    NSRect visRect;

    // Lock focus on ourselves.  We need to do this because we're drawing
    // outside of the context of NSView's drawRect: method.  This is relatively
    // unusual behaviour for a view.  See the discussion of this in ReadMe.rtf.
    
    [self lockFocus];

    // Now draw a random ball inside the view.
    visRect = [self visibleRect];
    [self drawRandomBallInside:&visRect];

    // And unlock the focus.
    [self unlockFocus];

    [[self window] flushWindow];
}

- (void)forceRedraw:(id)source
    // See comment in interface part.
{
    [self setNeedsDisplay:YES];
}

- (void)startOrStop:(id)source
    // See comment in interface part.
{
    [self setTimerRunning:!running source:source];
}

- (void)setPeriod:(id)source
    // See comment in interface part.
{
    [self setTimerPeriod:[source floatValue] source:source];
}

@end
