// algebra.h
//
// Some macros to ease calculation involving lines, vectors and points
// Vector and NSPoint are used in a rather confused manner here. They
// basicaly the same.
//
// created by Martin Wennerberg on Sun 12-Nov-1995
//
// when		who	modification

#include <Foundation/Foundation.h>

typedef NSPoint Vector;

static inline float vt_dotProd (Vector v1, Vector v2)
{ return v1.x * v2.x + v1.y * v2.y; }

static inline float vt_length (Vector v)
{ return sqrt(v.x * v.x + v.y * v.y);}

static inline float vt_angle (Vector v)
{ return atan2(v.y, v.x);}

static inline Vector vt_scale (Vector v, float d)
{ return NSMakePoint((v.x) * (d), (v.y) * (d));}

static inline Vector vt_perp (Vector v, float d)
{ return NSMakePoint(v.y, -v.x);}

static inline NSPoint pt_sum (NSPoint v1, NSPoint v2)
{ return NSMakePoint((v1.x) + (v2.x), (v1.y) + (v2.y));}

static inline NSPoint pt_sub (NSPoint v1, NSPoint v2)
{ return NSMakePoint((v1.x) - (v2.x), (v1.y) - (v2.y));}

static inline NSPoint pt_scale (NSPoint p, float sx, float sy)
{ return NSMakePoint((p.x) * (sx), (p.y) * (sy));}

static inline float pt_dist2 (NSPoint p1, NSPoint p2)
{ return (p1.x - p2.x) * (p1.x - p2.x) + (p1.y - p2.y) * (p1.y - p2.y);}

static inline float pt_dist (NSPoint p1, NSPoint p2)
{ return sqrt (pt_dist2 (p1, p2));}

static inline float line_angle (NSPoint p1, NSPoint p2)
{
    return  vt_angle(pt_sub(p2, p1));
}

static inline NSPoint pt_wmean (NSPoint p1, NSPoint p2, float d)
{ return  NSMakePoint (p1.x + d * (p2.x - p1.x), p1.y + d * (p2.y - p1.y));}

static inline NSRect rectContainingPoints (NSPoint p1, NSPoint p2)
{ return NSMakeRect(MIN(p1.x, p2.x), MIN(p1.y, p2.y), MAX(p1.x, p2.x) - MIN(p1.x, p2.x),MAX(p1.y, p2.y) - MIN(p1.y, p2.y));}

static inline float line_perp_dist (NSPoint p1, NSPoint p2, NSPoint p)
{return vt_length( pt_sub( p,
                         pt_sum( p1,
                                 vt_scale( pt_sub( p2,p1),
                                           vt_dotProd( pt_sub(p,p1), pt_sub(p2,p1))
                                           / vt_dotProd (pt_sub(p2,p1), pt_sub(p2,p1))))));}

static inline float line_perp_dist2 (NSPoint p1, NSPoint p2, NSPoint p)
{
    NSPoint pt = ( pt_sub( p,
                         pt_sum( p1,
                                 vt_scale( pt_sub( p2,p1),
                                           vt_dotProd( pt_sub(p,p1), pt_sub(p2,p1))
                                           / vt_dotProd (pt_sub(p2,p1), pt_sub(p2,p1))))));
    return pt.x * pt.x + pt.y * pt.y;
}

// Returns the perpendicular distance from a point p and a line that goes throught p1 and p2. Neads cleaning.

static inline float line_pos_along (NSPoint p1, NSPoint p2, NSPoint p)
{ return vt_dotProd (pt_sub (p, p1), pt_sub (p2, p1))
         / pt_dist2 (p1, p2);}

static inline float line_dist_to_point (NSPoint p1, NSPoint p2, NSPoint p)
{
  float posAlong = line_pos_along (p1, p2, p);

  if (posAlong > 1)
    return pt_dist (p2, p);
  else if (posAlong < 0)	
    return pt_dist (p1, p);	
  else
    return line_perp_dist (p1, p2, p);
}

static inline float line_dist2_to_point (NSPoint p1, NSPoint p2, NSPoint p)
{
    float posAlong = line_pos_along (p1, p2, p);
    
  if (posAlong > 1)
    return pt_dist2 (p2, p);
  else if (posAlong < 0)	
    return pt_dist2 (p1, p);	
  else
      return line_perp_dist2 (p1, p2, p);
}

/**********************************************************
 *  Old version
 **********************************************************/

#define PI	3.147158
#define PT_DOTPROD(p1,p2)	(p1.x * p2.x + p1.y * p2.y)
#define PT_LENGTH(p)	sqrt(p.x * p.x + p.y * p.y)
#define PT_ANGLE(p)	(atan2(p.y,p.x))
/*
extern NSPoint	PT_sum (NSPoint p1, NSPoint p2);
extern NSPoint	PT_sub (NSPoint p1, NSPoint p2);
extern NSPoint	PT_scale (float k, NSPoint p);
extern NSPoint	PT_perp (NSPoint p);
*/