/* indent:4  tabsize:8  font:fixed-width */

#import "Solitaire.h"
#import "ModuleList.h"
#import "localstrings.h"
#import "NSString+whitespace.h"

#define GAMEDIRECTORY @"Solitaire"

@implementation Solitaire(More)

/*----------------------------------------------------------------------------
|
|    - getGameType
|
|-----------------------------------------------------------------------------
|
|    Go find dynamically loadable Solitaire games in ~/Library/Solitaire,
|    Solitaire.app, and /LocalLibrary/Solitaire and any directory in 
|	 the ther user's defaults (SolitaireBundlePaths).  Load their descriptions
|    into the game-selection browser.
|			
\----------------------------------------------------------------------------*/

- (void) getGameType
{
    NSMatrix* 		theMatrix;
    NSString*		unseparatedBundlePaths;
    NSArray*		bundlePaths = nil;
    NSEnumerator*	bundlePathsEnumerator = nil;
    NSString*		aBundlePath;
    NSString *bundleLocationInLibraries=GAMEDIRECTORY;
    NSArray *librariesArray;
    NSArray *gamePaths;
    NSEnumerator *gamePathEnumerator;
    id eachGamePath;
    
    NSSearchPathDomainMask searchMask=NSUserDomainMask|NSLocalDomainMask|NSNetworkDomainMask;
    librariesArray=NSSearchPathForDirectoriesInDomains(NSLibraryDirectory,searchMask,YES);
    gamePaths=[bundleLocationInLibraries stringsByAppendingComponentToPaths:librariesArray];

    
    // Get any paths from the user's defaults.
    unseparatedBundlePaths = [[NSUserDefaults standardUserDefaults] stringForKey:@"SolitaireBundlePaths"];
    if (unseparatedBundlePaths != nil && ![unseparatedBundlePaths isEqualToString:@""]) {
        bundlePaths = [unseparatedBundlePaths componentsSeparatedByString:@";"];
    }
    
    if (!moduleList)
    {
        moduleList = [[ModuleList allocWithZone:[self zone]] init];
    }

    // Do the obvious places first (on Mach anyway).

    gamePathEnumerator=[gamePaths objectEnumerator];
    while (eachGamePath = [gamePathEnumerator nextObject])
      {
        [self loadGamesFrom:eachGamePath];
      }
    
    [self loadGamesFrom:[self appDirectory]];

    // Now do any paths found in the defaults.
    bundlePathsEnumerator = [bundlePaths objectEnumerator];
    while ((aBundlePath = [bundlePathsEnumerator nextObject]) != nil) {
        if (![aBundlePath isEqualToString:@""]) {
            [self loadGamesFrom:aBundlePath];
        }
    }
    
    if ([moduleList count] == 0)
    {
        NSRunAlertPanel([[NSProcessInfo processInfo] processName],
                        LOCALIZED_NOGAMES, LOCALIZED_TERMINATE, nil, nil);
	[NSApp terminate:self];
    }
    
    [moduleList sort];
    browserValid = NO;
    [gameSelectionBrowser loadColumnZero];
    theMatrix = [gameSelectionBrowser matrixInColumn:0];
    [theMatrix selectCellAtRow:realGameIndex column:0];
    [theMatrix scrollCellToVisibleAtRow:realGameIndex column:0]; 
}


/*----------------------------------------------------------------------------
|
|    - setGameType:(NSString *)filename dir:(NSString *)dirname
|
|-----------------------------------------------------------------------------
|
|    Load the description of the specified .solitaire bundle into the 
|    browser.
|			
\----------------------------------------------------------------------------*/

- (void) setGameType:(NSString *)filename dir:(NSString *)dirname
{
    NSMatrix* theMatrix;

    if (!moduleList)
    {
        moduleList = [[ModuleList allocWithZone:[self zone]] init];
    }

    browserValid = NO;
    [self addGameToList:filename dir:dirname];
    [moduleList sort];
    [gameSelectionBrowser loadColumnZero];
    theMatrix = [gameSelectionBrowser matrixInColumn:0];
    [theMatrix selectCellAtRow:realGameIndex column:0];
    [theMatrix scrollCellToVisibleAtRow:realGameIndex column:0]; 
}

/*----------------------------------------------------------------------------
|
|    - loadGamesFrom:(NSString *)dirname
|
|-----------------------------------------------------------------------------
|
|    Find all .solitaire modules in the specified directory.
|    If we find a module in several places, we keep only the first one found.
|			
\----------------------------------------------------------------------------*/

- (void) loadGamesFrom:(NSString *)dirname
{
    NSFileManager *fm = [NSFileManager defaultManager];
    NSArray *files = [fm directoryContentsAtPath:dirname];
    int i, count = [files count];

    for (i = 0; i < count; i++) {
        NSString *file = [files objectAtIndex:i];
        if ([[file pathExtension] isEqualToString:@"solitaire"]) {
            [self addGameToList:file dir:dirname];
        }
    }
}

/*----------------------------------------------------------------------------
|
|    - addGameToList:(NSString *)filename dir:(NSString *)dirname
|
|-----------------------------------------------------------------------------
|
|    Add the specified game to the list of games.
|
\----------------------------------------------------------------------------*/

- (void) addGameToList:(NSString *)filename dir:(NSString *)dirname
{
    ModuleInfo *m;
    int i, numstrings;
    NSString* gameName = nil;
    BOOL valid = YES;
   
    numstrings = [moduleList count];
    // Game.solitaire -> Game
    gameName = [[filename stringByDeletingPathExtension] lastPathComponent];

    // check for duplicate modules; keep only the first one found
    for (i=0; (i < numstrings) && (valid == YES); i++)
    {
	if ([gameName isEqualToString:[moduleList nameAtIndex:i]])
	{
	    // already have a module with this name
	    valid = NO;
	}
    }
    
    if (valid) 
    {
	NSString* path;

	// add a new entry for this module
	path = [dirname stringByAppendingPathComponent:filename];

	m = [[ModuleInfo allocWithZone:[self zone]] 
		initWithController:NULL name:gameName path:path];
	if (m)
	    [moduleList addObject:m];
    } 
}


/*----------------------------------------------------------------------------
|
|    - (NSString *)appDirectory
|
|    returns:  (NSString *) full path of main bundle
|
|-----------------------------------------------------------------------------
|
|    Returns the application's main bundle directory.
|			
\----------------------------------------------------------------------------*/

- (NSString *)appDirectory
{
    return [[NSBundle mainBundle] bundlePath];
}


@end
