/* indent:4  tabsize:8  font:fixed-width */

#import <Solitaire/GameModule.h>
#import <Solitaire/GamePref.h>
#import "localstrings.h"
#import <Solitaire/Card.h>
#import <Solitaire/GameModel.h>


@implementation GameModule

/*"
	GameModule is an abstract superclass providing a generic framework for 
	#{.solitaire} game modules.  It contains a number of methods which can be 
	overridden to provide game-specific behavior.

	The methods #startGame:, #restartGame:, and #checkForWin: should always be 
	overridden.
"*/

- initFromBundle:(NSBundle*)aBundle withName:(NSString*)name
/*"
	Designated initializer for the GameModule class.  If this method is 
	overridden in a subclass, always pass message back to this class via super.
"*/
{
    [super init];
    gameName = [name copy];
    bundle = [aBundle retain];

    [NSBundle loadNibNamed:@"Engine" owner:self];
    [prefs setGameName:name];

    [[NSNotificationCenter defaultCenter] addObserver:self
                                             selector:@selector(statisticChanged:)
                                                 name:STATISTIC_CHANGED
                                               object:nil];

    return self;
}

- (void) dealloc
{
    [[NSNotificationCenter defaultCenter] removeObserver:self];
    [inspector release];
    [bundle release];
    [prefs release];
    [gameName release];
    [super release];
}

- (NSView*) inspector:sender
/*"
	Returns the module's inspector view.
"*/
{
    return inspector;
}


- (void) inspectorWillBeRemoved
/*"	
	Sent just before the module's inspector is removed; i.e. before switching 
	to a new game. The default implementation does nothing.
"*/
{
     
}


- (void) inspectorInstalled
/*"
	Sent immediately after this module's inspector view has been installed in
	the Game Selection Panel.  Override as needed. The default implementation
	does nothing.
"*/
{
    [self statisticChanged:nil];
}


- (void) newGame:(id)sender
/*"
	Start a new game; i.e. shuffle the deck and deal.  If overridden, always 
	send the message back to the GameModule class via super.
"*/
{
    // subclass responsability
}

- (void) win
/*"
        Called when the game has been won.  By default, invokes the generic "win"
        routine.  Override to create custom "game won" behaviour.
"*/
{
    [[NSApp delegate] win];
}

- (void) showRules:(id)sender
/*"
        Displays the rules panel.
"*/
{
    if (!rulesWindow)
    {
        NSString* path;
        [NSBundle loadNibNamed:@"Rules" owner:self];
        path = [NSString stringWithFormat:@"%@Rules",gameName];
        [rulesWindow setFrameAutosaveName:path];
    }
    [rulesWindow makeKeyAndOrderFront:sender];
}

- gamePrefWithName:pref
    /*"
            Return a preferences string by appending the pref string given with the game name.
    "*/
{
    Class pc = [self class];
    return [NSString stringWithFormat:@"%@-%@", pc, pref];
}

- (void)statisticChanged:(NSNotification*)notif
    /*"
            Displays the rules panel.
    "*/
{
    NSUserDefaults *d = [NSUserDefaults standardUserDefaults];
    int won = [d integerForKey:[self gamePrefWithName:@"LifetimeWins"]];
    int lost = [d integerForKey:[self gamePrefWithName:@"LifetimePlayed"]];
    [totalWinTextField setIntValue:won];
    [totalPlayedTextField setIntValue:lost];
}

@end
