/* indent:4  tabsize:8  font:fixed-width */

#import "Card.h"
#import <Foundation/Foundation.h>

/*---------------------------------------------------------------------------
|
|    Card Globals
|
|    The arrays _CardPipPositions and _CardCardPips provide information about
|    the positioning of pips on cards.  There are 17 possible positions,
|    described in pipPositions, which are selectively refrerred to for each 
|    of 13 possible card values by a character string.
|
|    The letters A-Q are used to represent pips present in the given card.
|
|    The id's _CardFront, _CardBack etc... are images shared by all instances
|    of Card
|
\----------------------------------------------------------------------------*/

static NSString *cardValueStrings[] = {    
    @"Ace", @"2", @"3", @"4", @"5", @"6", @"7", @"8", @"9", @"10",
    @"Jack", @"Queen", @"King", nil
};

static NSString *cardSuitStrings[] = {
    @"Clubs", @"Spades", @"Diamonds", @"Hearts", nil
};

static NSString *cardColorStrings[] = {
    @"Black", @"Red", nil
};


@implementation Card

/*"
   Instances of this class provide low-level functionality for card games.
   Cards only have a suit and a value.  These values can be
   initialized, modified and determined by clients of the class.

   Cards do not know how to render themselves on screen.
"*/

//-------------------------------------------------------------------
//	Initializing the class
//-------------------------------------------------------------------

+ (NSString *)stringForCardSuit:(CardSuit)aSuit
/*"
**	Returns a string that represents aSuit. This will be one of Spades, Clubs,
**	Diamonds or Hearts. The current way of doing this is not localized.
"*/
{
    if (aSuit > CS_HIGHSUIT)
        [NSException raise:NSRangeException format:
            @"***%s: index (%d) beyond bounds (%d)",
            sel_getName(_cmd), aSuit, CS_HIGHSUIT];

    return cardSuitStrings[aSuit];
}

+ (NSString *)stringForCardValue:(CardValue)aValue
/*"
**	Returns a string that represents aValue. This will be one of Ace, Jack, 
**	Queen, King, or 1 through 10. This current implementation does not lend 
**	itself to localization.
"*/
{
    if (aValue > CS_HIGHVALUE)
        [NSException raise:NSRangeException format:
            @"***%s: index (%d) beyond bounds (%d)",
            sel_getName(_cmd), aValue, CS_HIGHVALUE];

    return cardValueStrings[aValue];
}

+ (NSString *)stringForCardColor:(CardColor)aColor
/*"
**	Returns a string that represents aColor. It will either return Red or 
**	Black. 
"*/
{
    if (aColor > CS_HIGHCOLOR)
        [NSException raise:NSRangeException format:
            @"***%s: index (%d) beyond bounds (%d)",
            sel_getName(_cmd), aColor, CS_HIGHCOLOR];

    return cardColorStrings[aColor];
}

+ (NSArray*)fullSet
/*" SHOULD Return an array with every card in it."*/
{
    static NSArray* fullSet = nil;
    if(fullSet == nil) {
        
    }
    return fullSet;
}


//-------------------------------------------------------------------
//	Creating and freeing
//-------------------------------------------------------------------

- init
/*"
   Initialize a new Card object.  This method invokes the
   #initSuit:value:faceUp: method.  The suit is set to CS_CLUBS, the value
   is set to CS_ACE, and the card is face down.  Returns self.
"*/
{
    return [self initSuit:CS_ACE value:CS_CLUBS];
}


- initSuit:(CardSuit)aSuit value:(CardValue)aValue
/*"
   Initializes a card with suit aSuit, value aValue, and orientation anOrientation.
   This is the designated initializer for Card object instances. Returns self.
"*/
{
    [super init];
    cardSuit = aSuit;
    cardValue = aValue;
    return self;
}


- copyWithZone:(NSZone *)zone
/*"
   Returns a new Card object, allocated from the specified zone, which is an
   exact duplicate of the sender.
"*/
{
    return [[[self class] allocWithZone:zone] initSuit:cardSuit value:cardValue];
}

- (NSString *)description
{
    NSString *theString;

    theString = [NSString stringWithFormat:@"%@ : %@ of %@",
        [super description], [Card stringForCardValue:cardValue],
        [Card stringForCardSuit:cardSuit]];
    return theString;
}


- (void) setCardSuit:(CardSuit)aSuit
/*"
   Sets the card's suit to aSuit. Valid suit values are:

   _{#{Constant Suit}}
   _{CS_LOWSUIT	Clubs}
   _{CS_CLUBS		Clubs}
   _{CS_SPADES		Spades}
   _{CS_DIAMONDS	Diamonds}
   _{CS_HEARTS		Hearts}
   _{CS_HIGHSUIT	Hearts}
"*/
{
    cardSuit = aSuit; 
}


- (void) setCardValue:(CardValue)aValue
/*"
   Sets the card's value to aValue.  Valid card values are:

   _{#{Constant		Card Value}}
   _{CS_LOWVALUE	Ace}
   _{CS_ACE			Ace}
   _{CS_TWO			Two}
   _{CS_THREE		Three}
   _{CS_FOUR		Four}
   _{CS_FIVE		Five}
   _{CS_SIX			Six}
   _{CS_SEVEN		Seven}
   _{CS_EIGHT		Eight}
   _{CS_NINE		Nine}
   _{CS_TEN			Ten}
   _{CS_JACK		Jack}
   _{CS_QUEEN		Queen}
   _{CS_KING		King}
   _{CS_HIGHVALUE	King}
"*/
{
    cardValue = aValue; 
}

- (CardSuit) cardSuit
/*"
   Returns the receiver's suit.  The returned value will be one of:

   _{#{Constant	Suit}}
   _{CS_LOWSUIT	Clubs}
   _{CS_CLUBS	Clubs}
   _{CS_SPADES	Spades}
   _{CS_DIAMONDS Diamonds}
   _{CS_HEARTS	Hearts}
   _{CS_HIGHSUIT Hearts}
"*/
{
    return cardSuit;
}


- (CardValue) cardValue
/*"
   Returns the receiver's value.  The returned value will be one of:

   _{#{Constant		Card Value}}
   _{CS_LOWVALUE	Ace}
   _{CS_ACE			Ace}
   _{CS_TWO			Two}
   _{CS_THREE		Three}
   _{CS_FOUR		Four}
   _{CS_FIVE		Five}
   _{CS_SIX			Six}
   _{CS_SEVEN		Seven}
   _{CS_EIGHT		Eight}
   _{CS_NINE		Nine}
   _{CS_TEN			Ten}
   _{CS_JACK		Jack}
   _{CS_QUEEN		Queen}
   _{CS_KING		King}
   _{CS_HIGHVALUE	King}
"*/
{
    return cardValue;
}

+ (CardColor) colorInverseOf:(CardColor)co
{
    if (co == CS_BLACK)
    {
        return CS_RED;
    }
    return CS_BLACK;
}

+ (CardColor) colofOfSuit:(CardSuit)sui
    /*"
       Returns the color for suit.  The returned value is equal to one of these constants:

       _{#{Constant	Color}}
       _{CS_BLACK	Card's suit is either spades or clubs.}
       _{CS_RED		Card's suit is either hearts or diamonds.}
    "*/
{
    if ((sui == CS_SPADES) || (sui == CS_CLUBS))
    {
        return CS_BLACK;
    }
    return CS_RED;
}

- (CardColor) cardColor
    /*"
       Returns the receiver's color.  The returned value is equal to one of these constants:

       _{#{Constant	Color}}
       _{CS_BLACK	Card's suit is either spades or clubs.}
       _{CS_RED		Card's suit is either hearts or diamonds.}
    "*/
{
    return [[self class] colofOfSuit:cardSuit];
}


-(BOOL) isEqual:(Card*)aCard
/*"
   Returns YES if the receiver and aCard have the same suit and value.
"*/
{
	if (self == aCard)
		return YES;
	
	if ((cardValue == [aCard cardValue]) &&
		(cardSuit == [aCard cardSuit]))
		return YES;
	
	return NO;
}


-(BOOL) isOneLargerThan:(Card*)aCard wrap:(BOOL)flag
/*"
   Returns YES if the receiver's value is exactly one larger than aCard.
   The cards can either be in the same suit or different ones. If flag
   is YES then the case of the receiver being an ace and aCard being
   a king would return YES.
"*/
{
	if (cardValue == [aCard cardValue] + 1)
		return YES;
	
	if (flag && ((cardValue == CS_LOWVALUE) &&
				 ([aCard cardValue] == CS_HIGHVALUE)))
		return YES;
	
	return NO;
}


-(BOOL) isOneSmallerThan:(Card*)aCard wrap:(BOOL)flag
/*"
   Returns YES if the receiver's value is exactly one smaller than aCard.
   The cards can either be in the same suit or different ones. If flag
   is YES then the case of the receiver being a king and aCard being
   an ace would return YES.
"*/
{
	if (cardValue == [aCard cardValue] - 1)
		return YES;
	
	if (flag && ((cardValue == CS_HIGHVALUE) &&
				 ([aCard cardValue] == CS_LOWVALUE)))
		return YES;
	
	return NO;
}

- (id) initWithCoder:(NSCoder *)aDecoder
/*"
   Unarchives an instance of Card. Returns self.
"*/
{
    [aDecoder decodeValuesOfObjCTypes:"ii", &cardSuit, &cardValue];
    return self;
}

- (void) encodeWithCoder:(NSCoder *)aCoder
/*"
   Writes the Card object to aCoder. This message is sent is response to archiving;
   you never send this message directly.
"*/
{
    [aCoder encodeValuesOfObjCTypes:"ii", &cardSuit, &cardValue];
}

@end

