/* indent:4  tabsize:8  font:fixed-width */

#import <Solitaire/CardSet.h>
#import "CardPileView.h"
#import "ListHolder.h"
#import <Foundation/Foundation.h>
#import "CardPileController.h"

/*--------------------------------------------------------------------------
|
|    CardPileView Globals
|
|    CardPilePBoardType is a private pasteboard type used by the
|    CardPileView class.
|
\---------------------------------------------------------------------------*/

NSString* CardPilePBoardType = @"CardPilePBoardType";

// The default for our maxVisibleCards method.
#define CS_SHOWALL	2147483647

@implementation CardPileView

/*"
Instances of this class provide a visual representation of a pile of cards.
	CardPileView uses CardPile as the underlying representation, but adds a
	visual aspect.  User interaction is done by the controller object typicaly a CardPileController.

 We do not exactly define how a card is rendered, Instead we specify a layout and ask a
 cardImager to do the actual rendering.  The CardPileView is responsible of the display only.
 It wait changed notification from it's model to redisplay itself.
 "*/

- initWithFrame:(NSRect)theFrame
    /*"
    Creates an empty CardPile object to be managed by the CardPileView.  If
     theFrame is wider and/or taller than the size required by a single Card
     object, then the pile will automatically be drawn with an offset in that
     direction.  Returns self.
     "*/
{
    /* Default background color is felt green or light gray */

    [super initWithFrame:theFrame];
    cardImager = (CardImager*)[CardImager class];
    controller = [[CardPileController alloc] init];
    [controller setView:self];
    [self setDrawOutline:YES];
    [self registerForDraggedTypes:[NSArray arrayWithObject:CardPilePBoardType]];
    [self setMaxVisibleCards:CS_SHOWALL];     // Show all the cards we have by default.

    /*----------------------------------------------------------------------
        |    If the view is large enough to allow cards to be
        |    stacked vertically or horizontally, initialize the
        |    offsets appropriately to give an apparent "height"
        |    to the pile
        \----------------------------------------------------------------------*/

    if (theFrame.size.width > [cardImager cardWidth])
    {
        xOffset = 0.2;
    }

    if (theFrame.size.height > [cardImager cardHeight])
    {
        yOffset = 0.2;
    }

    return self;
}
- (void) dealloc
    /*"
    Frees our resources.
     "*/
{
    [[NSNotificationCenter defaultCenter] removeObserver:self];
    [model release];
//    [cardImager release];
    [controller release];
    [super dealloc];
}

- (void)pileChanged:(NSNotification *)notification
{
    [self display];
}

- (void) setMaxVisibleCards:(int)cardsVisible
    /*"
    Sets the maximum number of cards visible at any one time. An example of
     this would be when you are playing 3 card Klondike and you can see the
     edges of three cards at one time. The default is CS_SHOWALL.
     "*/
{
    cardsVisible = (cardsVisible < 0) ? 0 : cardsVisible;
    maxVisibleCards = cardsVisible;	
}


-(int) maxVisibleCards
    /*" Returns the maximum number of visible cards at any one time. The default
    is CS_SHOWALL.
    "*/
{
    return maxVisibleCards;
}

- (BOOL) willDrawOutline
    /*"  Returns YES if the CardPileView will draw a border, NO otherwise.  "*/
{
    return drawOutline;
}


- (void) setDrawOutline:(BOOL)aFlag
    /*"  Sets whether we'll draw a border around ourselves. The default is YES.  "*/
{
    drawOutline = aFlag;
}


- (ListHolder*) model
    /*"
    Returns our CardPile instance that holds all the cards we're displaying.
     "*/
{
    return model;
}

- (void)setModel:(CardPileHolder*)p
{
    if(model != p) {
        [[NSNotificationCenter defaultCenter] removeObserver:self name:nil object:model];
        [model release];
        model = [p retain];
        [controller setModel:p];
        [[NSNotificationCenter defaultCenter] addObserver:self
                                                 selector:@selector(pileChanged:)
                                                     name:LINK_LIST_CHANGED object:model];
    }
}

- cardImager
{
    return cardImager;
}

- (void)setOffset:(float)xOf :(float)yOf
{
    xOffset = xOf;
    yOffset = yOf;
}

- (BOOL)isOpaque
{
    return NO;
}

- (void) drawRect:(NSRect)theRect
    /*"
    Draws a visual representation of our CardPile.  Use the #display message,
     rather than calling this method directly.  You must call #display (or
                                                                        preferably #displayIfNeeded ) after any change to a CardPileView.
     "*/
{
    NSRect	cardRect = {{0, [self bounds].size.height - [cardImager cardHeight]},
			     {[cardImager cardWidth], [cardImager cardHeight]}};
    NSRect	bezelRect;
    NSPoint	lastOrigin = {-1000, -1000};
    CardPile* 	drawLastCard = (CardPile*)[self model];
    
    // only display the last maxVisibleCards.
    if(maxVisibleCards < [drawLastCard count]) {
        int idx = [drawLastCard count] - maxVisibleCards;
        drawLastCard = [drawLastCard objectAtIndex:idx];
    }

    if (drawOutline)
    {
        PSsetgray(NSBlack);
        bezelRect = cardRect;
        bezelRect.size.width -= 1.0;
        bezelRect.size.height -= 1.0;
        NSFrameRect(bezelRect);
        bezelRect.origin.x += 1.0;
        bezelRect.origin.y += 1.0;
        PSsetgray(NSWhite);
        NSFrameRect(bezelRect);
    }

    /*----------------------------------------------------------------------
	  |    Draw each card in turn
	  \---------------------------------------------------------------------*/

    while ([drawLastCard nextObject] != nil)
    {
        drawLastCard = [drawLastCard nextObject];

        /*--------------------------------------------------------------
        |    If the cards are widely spread draw the whole card
        \-------------------------------------------------------------*/

        if ( (yOffset > 2) || (xOffset > 2) || ([drawLastCard nextObject] == nil) )
        {
            [cardImager drawCard:drawLastCard at:cardRect.origin];
        }
        else
        {
            /*-------------------------------------------------------------
            |    Since the cards are closely spaced, don't draw
            |    an outline unless it is at least 2 pixels from the
            |    last card drawn.  Regardless, keep track of the
            |    most recent card so we can draw the image after
            |    the stack
            \--------------------------------------------------------------*/
            if ((cardRect.origin.x - lastOrigin.x >= 2.0) ||
                (lastOrigin.y - cardRect.origin.y >= 2.0))
            {
                [cardImager drawOutlineOfCard:drawLastCard at:cardRect.origin];
                lastOrigin = cardRect.origin;
            }
        }

        /*--------------------------------------------------------------
            |    Reset the origin for the next card
            \--------------------------------------------------------------*/
        cardRect.origin.x += xOffset;
        cardRect.origin.y -= yOffset;
    }
}

- (CardPile*) findPileAtPoint:(NSPoint)thePoint
    /*"  Returns the Pile object at thePoint, or nil if there is no card at that location.  "*/
{
    /*----------------------------------------------------------------------
    |    Search for a card hit from the top of the stack down
    \---------------------------------------------------------------------*/
    int cardIndex = ([self maxVisibleCards] > [[self model] count])
    ? [[self model] count] :
    [self maxVisibleCards];
    NSRect	cardRect = {{0, 0}, {[cardImager cardWidth], [cardImager cardHeight]}};
    CardPile*  	thePile = [[self model] lastObject];

    while( cardIndex > 0 )
    {
        cardRect.origin.x = (cardIndex -1) * xOffset;
        cardRect.origin.y = ([self bounds].size.height - [cardImager cardHeight])
            - ((cardIndex -1) * yOffset);
        if (NSMouseInRect(thePoint , cardRect , NO))
        {
            return thePile ;
        }

        cardIndex --;
        thePile = [thePile holder];
    }
    // do not return the list holder if did not clicked on a card, return nil instead.
    return ( model == (ListHolder*)thePile ) ? nil : thePile;
}

- (NSRect) getRectForPile:thePile
    /*"Return the rect covered by all the pile, in a spread form."*/
{
    NSRect 	cardRect = NSMakeRect(0.0, 0.0, 0.0, 0.0);
    NS_DURING
        int	idx	= [[self model] indexOfObject:thePile];
        int	count	= [thePile count];
	    int hiden = ([model count] - [self maxVisibleCards]);
	if(hiden < 0) hiden = 0;
        
        // Take into the cards that aren't visible
      	// idx = ((idx-1) % [self maxVisibleCards]) +1;
	idx = idx - hiden;

        /*----------------------------------------------------------------------
            |    Calculate and supply the bounding rectangle
            \---------------------------------------------------------------------*/
        cardRect.origin.x = xOffset * (idx -1);
        cardRect.origin.y = ([self bounds].size.height - [cardImager cardHeight]) - (yOffset * (idx + count -2));
        cardRect.size.width = [cardImager cardWidth] + (xOffset * ( count - 1));
        cardRect.size.height = [cardImager cardHeight] + (yOffset *( count - 1));

    NS_HANDLER
        /*----------------------------------------------------------------------
        |  int 	idx	= [[self model] indexOfObject:thePile];
        |
        |    will raise an exception if can not find the object in the list.
        |    Return immediately if the card isn't in the cardPile
        \---------------------------------------------------------------------*/
        NS_ENDHANDLER
        return cardRect;
}

- (void) drawDragCard:sender
    /*"
    Draw the pile of cards being dragged.  Do not call directly.
     "*/
{
    NSPoint        tempPoint = { 0, 0 };
    CardPile *thePile = [[controller draggedHolder] firstObject];

    tempPoint.y += ([thePile count] - 1) * yOffset;

    do {
        [cardImager drawCard:thePile at:tempPoint];
        tempPoint.x += xOffset;
        tempPoint.y -= yOffset;
    } while((thePile = [thePile nextObject]) != nil);
}

- controller
{
    return controller;
}

- (void) setTag:(int)theTag
    /*"	Sets our tag.   "*/
{
    tag = theTag;
}

- (int) tag
    /*"	Returns our tag.   "*/
{
    return tag;
}

/*" User interaction "*/

- (void) superMouseDragged:(NSEvent *)theEvent
/*" This method use default mouseDragged: method that the NSDragging mechanism expect from a view."*/
{
    [[self nextResponder] mouseDragged:theEvent];
}

- (void) mouseDragged:(NSEvent *)theEvent
    /*" user interaction behavior is done by the controller."*/
{
    [controller mouseDragged:theEvent];
}

- (void) superMouseUp:(NSEvent*)ev
{
    [[self nextResponder] mouseUp:ev];
}

- (void) mouseUp:(NSEvent*)thisEvent
    /*" user interaction behavior is done by the controller."*/
{
    [controller mouseUp:thisEvent];
}

- (void) superMouseDown:(NSEvent*)ev
{
    [[self nextResponder] mouseDown:ev];
}

- (void) mouseDown:(NSEvent *)thisEvent
    /*" user interaction behavior is done by the controller."*/
{
    [controller mouseDown:thisEvent];
}

- (unsigned int) draggingEntered:sender
    /*" user interaction behavior is done by the controller."*/
{
    return [controller draggingEntered:sender];
}

- (BOOL) prepareForDragOperation:sender
    /*" user interaction behavior is done by the controller."*/
{
    return [controller prepareForDragOperation:sender];
}

- (BOOL)performDragOperation:sender
    /*" user interaction behavior is done by the controller."*/
{
    return [controller performDragOperation:sender];
}



@end

