objc:declare STR tcl_tmpnam STR

if {$tcl_platform(platform) == "windows"} {
    set openDirectory $env(NEXT_ROOT)/LocalLibrary/Services
} else {
    set openDirectory ~/Library/Services
}

objc:teach $self {

- TclObj nextUntitled: {TclObj list} {
    set max -1
    foreach name $list {
        switch -regexp [file tail $name] {
            {^UNTITLED$} {
                if {$max < 0} {set max 0}
            }
            {^UNTITLED[0-9]+$} {
                scan [file tail $name] UNTITLED%d n
                if {$max < $n} {set max $n}
            }
        }
    }
    if {$max == -1} {
        return UNTITLED
    }
    return UNTITLED[expr {$max + 1}]
}

- void registerDocument: document {
    global documentPath isUntitled documents
    if {[info exists documentPath]} {
        set isUntitled [NO]
        set name $documentPath
        unset documentPath
    } else {
        set isUntitled [YES]
        set name [file join [$self openDirectory] \
                    [$self nextUntitled: [array names documents \
                                            [file join * UNTITLED*]]]]
    }
    if {[$document openWithName: $name untitled: $isUntitled]} {
        set documents($name) $document
    } else {
        $document autorelease
        [$document window] autorelease
    }
}

- void unregisterDocument: document {
    global documents
    unset documents([$document name])
    $document autorelease
    [$document window] autorelease
}

- void document: document didChangeNameFrom: {TclObj oldName} {
    global documents
    unset documents($oldName)
    set documents([$document name]) $document
}

- TclObj openDirectory {
    global NSApp openDirectory
    if {[set mainWindow [$NSApp mainWindow]] == "nil"} {
        return $openDirectory
    }
    file dirname [* [$mainWindow representedFilename]]
}

- BOOL application: sender openFile: file {
    global documentPath documents
    set name [* [$file stringByStandardizingPath]]
    if {[info exists documents($name)]} {
        [$documents($name) window] makeKeyAndOrderFront: $sender
        return [YES]
    }
    set documentPath $name
    NSBundle loadNibNamed: [@ ServiceDocument] owner: $self
    if {[info exists documents($name)]} {YES} {NO}
}

- BOOL applicationOpenUntitledFile: sender {
    NSBundle loadNibNamed: [@ ServiceDocument] owner: $self
    return [YES]
}

- BOOL applicationShouldTerminate: sender {
    global documents
    foreach name [array names documents] {
        if {[[$documents($name) window] isDocumentEdited]} {
            switch -- [NSRunAlertPanel [@ Quit] \
                        [@ "There are edited windows."] \
                        [@ "Review Unsaved"] [@ "Quit Anyway"] [@ Cancel]] \
                [NSAlertDefaultReturn] {
                    foreach name [array names documents] {
                        if {[[$documents($name) window] isDocumentEdited]} {
                            switch -- [NSRunAlertPanel [@ "Review Unsaved"] \
                                    [@ "Save changes to [file tail $name]?"] \
                                    [@ Save] [@ "Don't Save"] [@ Cancel]] \
                                [NSAlertDefaultReturn] {
                                    $documents($name) save: $sender
                                } \
                                [NSAlertOtherReturn] {return [NO]}
                        }
                    }
                    return [YES]
                } \
                [NSAlertAlternateReturn] {return [YES]} \
                [NSAlertOtherReturn] {return [NO]}
        }
    }
    return [YES]
}

- BOOL validateMenuItem: item {
    global documents
    if {[$item action] == "saveAll:" && [array names documents] == ""} {
        return [NO]
    }
    return [YES]
}

- void info: sender {
    global mainInterp
    $mainInterp globEval: {
        if {![info exists infoPanel]} {
            NSBundle loadNibNamed: [@ InfoPanel] owner: $self
            $infoPanel center
        }
        $infoPanel makeKeyAndOrderFront: $self
    }
}

- void open: sender {
    global openDirectory
    set openPanel [NSOpenPanel openPanel]
    $openPanel setAllowsMultipleSelection: [YES]
    if {[$openPanel runModalForDirectory: [@ [$self openDirectory]] file: nil \
      types: [NSArray arrayWithObject: [@ service]]] == [NSCancelButton]} {
        return
    }
    set openDirectory [* [$openPanel directory]]
    objc:foreach file [$openPanel filenames] {
        $self application: $sender openFile: $file
    }
}

- void new: sender {
    $self applicationOpenUntitledFile: $sender
}

- void saveAll: sender {
    global documents
    foreach name [array names documents] {
        $documents($name) save: $sender
    }
}

- void rescanServices: sender {
    [$self serverProxy] performSelector: rescanServices
    catch {exec make_services &}
}

# returns the Workspace's autolaunch paths -
# an NSMutableArray on Rhapsody, an NSString on Next
- launchPaths {
    global tcl_platform
    if {$tcl_platform(os) == "Rhapsody"} {
        if {[catch {set launchPaths [@ [exec defaults read Workspace AutoLaunchedApplication]]}]} {
            return [NSMutableArray array]
        }
        set tmpFile [@ [tcl_tmpnam NULL]]
        $launchPaths writeToFile: $tmpFile atomically: [NO]
        return [NSMutableArray arrayWithContentsOfFile: $tmpFile]
    } else {
        if {[catch {set launchPaths [lindex [exec dread Workspace LaunchPaths] 2]}]} {
            return [@ ""]
        }
        return [@ $launchPaths]
    }
}

- void setLaunchPaths: newLaunchPaths {
    global tcl_platform
    if {$tcl_platform(os) == "Rhapsody"} {
        exec defaults write Workspace AutoLaunchedApplication [* [$newLaunchPaths description]]
        return
    }
    if {[* $newLaunchPaths] == ""} {
        exec dremove Workspace LaunchPaths
        return
    }
    exec dwrite Workspace LaunchPaths [* $newLaunchPaths]
}

- void addToLaunchPaths: path {
    global tcl_platform
    if {$tcl_platform(os) == "Rhapsody"} {
        set launchPaths [$self launchPaths]
        $launchPaths addObject: $path
        $self setLaunchPaths: $launchPaths
        return
    }
    set launchPaths [* [$self launchPaths]]
    if {$launchPaths == ""} {
        $self setLaunchPaths: $path
    } else {
        $self setLaunchPaths: [@ "$launchPaths;[* $path]"]
    }
}

- void removeFromLaunchPaths: path {
    global tcl_platform
    if {$tcl_platform(os) == "Rhapsody"} {
        set launchPaths [$self launchPaths]
        $launchPaths removeObject: $path
        $self setLaunchPaths: $launchPaths
        return
    }
    set launchPaths [split [* [$self launchPaths]] ";"]
    set index [lsearch $launchPaths [* $path]]
    $self setLaunchPaths: [@ [join [lreplace $launchPaths $index $index] ";"]]
}

- BOOL serverIsInstalled {
    global serverIsInstalled tcl_platform
    if {![info exists serverIsInstalled]} {
        if {$tcl_platform(os) == "Rhapsody"} {
            set serverIsInstalled [[$self launchPaths] containsObject: [$self serverPath]]
        } else {
            if {[string first [* [$self serverPath]] [* [$self launchPaths]]] == -1} {
                set serverIsInstalled [NO]
            } else {
                set serverIsInstalled [YES]
            }
        }
    }
    return $serverIsInstalled
}

- serverPath {
    global tcl_platform
    if {$tcl_platform(platform) == "windows"} {
        set type exe
    } else {
        set type daemon
    }
    return [[NSBundle mainBundle] pathForResource: [@ JoyServer] ofType: [@ $type]]
}

- void installServer: sender {
    global serverIsInstalled
    $self addToLaunchPaths: [$self serverPath]
    set serverIsInstalled [YES]
}

- void uninstallServer: sender {
    global serverIsInstalled
    $self removeFromLaunchPaths: [$self serverPath]
    set serverIsInstalled [NO]
}

- serverProxy {
    global serverProxy
    if {![info exists serverProxy] || $serverProxy == "nil"} {
        set serverProxy [[NSConnection rootProxyForConnectionWithRegisteredName: [@ JoyServicesDaemon] host: nil] retain]
        catch {
            objc:newProtocol JoyServiceProtocol {
                - void rescanServices
                - void terminate: sender
            }
        }
        $serverProxy setProtocolForProxy: [objc:protocol JoyServiceProtocol]
    }
    return $serverProxy
}

- void startServer: sender {
    global serverProxy
    exec [* [$self serverPath]] &
    unset serverProxy
    while {[$self serverProxy] == "nil"} {
    }
}

- void stopServer: sender {
    global serverProxy
    catch {
        [$self serverProxy] performSelector: terminate: withObject: $sender
    }
    unset serverProxy
}

- BOOL validateMenuItem: item {
    global tcl_platform
    switch [$item tag] {
        1 - 3 { # Server/Update, Server/Stop
            if {[$self serverProxy] == "nil"} {
                return [NO]
            }
            return [YES]
        }
        2 { # Server/Start
            if {[$self serverProxy] == "nil"} {
                return [YES]
            }
            return [NO]
        }
        4 { # Server/Install
            if {$tcl_platform(platform) == "windows" || [$self serverIsInstalled]} {
                return [NO]
            }
            return [YES]
        }
        5 { # Server/Deinstall
            if {$tcl_platform(platform) == "windows" || ![$self serverIsInstalled]} {
                return [NO]
            }
            return [YES]
        }
        default {
            former validateMenuItem: $item
        }
    }
}

}

objc:teach NSTableView {

- int numberOfRowsInTableView: sender {
    llength [iset list]
}

- tableView: sender objectValueForTableColumn: column row: {int row} {
    @ [lindex [iset list] $row]
}

- void tableView: sender setObjectValue: new
  forTableColumn: column row: {int row} {
    global defaultCenter
    set list [iset list]
    set new [* $new]
    if {[lsearch -exact $list $new] != -1} {
        return
    }
    iset list [lreplace $list $row $row $new]
    $defaultCenter \
        postNotificationName: [@ NSTableViewListDidChangeNotification] \
                      object: $self
}

- void setList: {TclObj list} {
    iset list $list
    $self setDataSource: $self
    $self reloadData
}

- TclObj list {
    if {[catch {iset list} result]} {
        iset list ""
    } else {
        return $result
    }
}

- TclObj selection {
    if {[$self allowsMultipleSelection]} {
        set sel ""
        set enumerator [$self selectedRowEnumerator]
        while {[set row [$enumerator nextObject]] != "nil"} {
            lappend sel [lindex [iset list] [$row intValue]]
        }
        return $sel
    }
    if {[set row [$self selectedRow]] == -1} {
        return ""
    }
    lindex [iset list] $row
}

- void setSelection: {TclObj selection} {
    if {$selection == ""} {
        # This will only work correctly if allowsEmptySelection is YES.
        $self deselectAll: nil
        return
    }
    if {[$self allowsMultipleSelection]} {
        set i 0
        foreach item [iset list] {
            set index($item) $i
            incr i
        }
        set flag [NO]
        foreach sel $selection {
            $self selectRow: $index($sel) byExtendingSelection: $flag
            set flag [YES]
        }
        $self scrollRowToVisible: $index([lindex $selection 0])
    } else {
        $self selectRow: [lsearch -exact [iset list] $selection] \
              byExtendingSelection: [NO]
    }
}

}

$self eval: {
    if {$tcl_platform(platform) != "windows" && ![$self serverIsInstalled]} {
        if {[NSRunAlertPanel [@ "Server not installed"] [@ "The Joy Services Server is not installed for this user account."] [@ "Install & Run"] [@ Cancel] nil] == [NSAlertDefaultReturn]} {
            $self installServer: $self
            if {[$self serverProxy] == "nil"} {
                $self startServer: $self
            }
        }
    } elseif {[$self serverProxy] == "nil" && [NSRunAlertPanel [@ "Server not running"] [@ "The Joy Services Server is not running on this machine."] [@ Run] [@ Cancel] nil] == [NSAlertDefaultReturn]} {
        $self startServer: $self
    }
} after: 1
