/*  
 *  MacOSXAmp - graphically mp3 player for MaxOS X Server
 *  Copyright (C) 1999  Scott P. Bender (sbender@harmony-ds.com)
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; see the file COPYING if not, write to 
 *  the Free Software Foundation, Inc., 59 Temple Place - Suite 330, 
 *  Boston, MA 02111-1307, USA.
*/
#import "Plugins.h"
#import "Visualization.h"
#import "Config.h"
#import "MainView.h"
#import "PlaylistEntry.h"
#import "AppDelegate.h"


static NSMutableArray *input_plugins = nil;
static NSMutableArray *output_plugins = nil;
static NSMutableArray *effect_plugins = nil;
static BOOL input_playing = NO;
static BOOL input_paused = NO;
static Input *current_input_plugin = nil;
static Output *selectedOutputPlugin = nil;
static NSMutableArray *vis_list = nil;
static NSLock *vis_list_lock = nil;
static NSString *playingFile = nil;
NSString *SongInfoChangedNotification = @"SongInfoChangedNotification";
NSString *SongEndedNotification = @"SongEndedNotification";

@implementation Input


+ (void)registerPlugin:(Plugin *)op
{
  if ( input_plugins == nil )
    input_plugins = [[NSMutableArray array] retain];
  [input_plugins addObject:op];
}

+ (NSArray *)plugins
{
  return input_plugins;
}

+ (void)setInfoTitle:(NSString *)title 
	      length:(int)length
		rate:(int)rate
	   frequency:(int)frequency
	 numChannels:(int)numChannels;
{
  NSMutableDictionary *info = [NSMutableDictionary dictionary];

  if ( title != nil )
    [info setObject:title
	     forKey:@"title"];
  if ( frequency != -1 ) {
    [info setObject:[NSNumber numberWithInt:length]
	     forKey:@"length"];
    [info setObject:[NSNumber numberWithInt:frequency]
	     forKey:@"frequency"];
    [info setObject:[NSNumber numberWithInt:numChannels]
       forKey:@"numChannels"];
  }

  [info setObject:[NSNumber numberWithInt:rate]
	   forKey:@"rate"];


  [[NSNotificationCenter defaultCenter]
     postNotificationName:SongInfoChangedNotification
		   object:nil
		 userInfo:info];
}


+ (void)setInfoText:(NSString *)text
{
#if 1
  NSMutableDictionary *info = [NSMutableDictionary dictionary];

  if ( text != nil ) {
//    NSLog(text);
    [info setObject:text forKey:@"title"];
  }

  [[NSNotificationCenter defaultCenter]
     postNotificationName:SongInfoChangedNotification
		   object:nil
		 userInfo:info];
#endif
}

+ (void)lockInfoText:(NSString *)text
{
  [[NSApp delegate] lockMainInfoText:text];
}

+ (void)unlockInfoText
{
  [[NSApp delegate] unlockMainInfoText];
}

- (void)setInfoText:(NSString *)text
{
  [[self class] setInfoText:text];
}

- (void)lockInfoText:(NSString *)text
{
  [[self class] lockInfoText:text];
}

- (void)unlockInfoText
{
  [[self class] unlockInfoText];
}

+ (void)setEq
{
  if ( input_playing && current_input_plugin != nil )
    [current_input_plugin setEq:[cfg equalizer_active]
			 preamp:[cfg equalizer_preamp]
			  bands:[cfg eq_bands]];
}

+ (NSString *)playingFile
{
  return playingFile;
}

+ (Input *)epluginForFile:(NSString *)filename
{
  NSArray *inputps;
  Input *ip;
  int i;

  inputps = [Input plugins];
    
  for ( i = 0; i < [inputps count]; i++ ) {
    ip = [inputps objectAtIndex:i];
    if ( [ip enabled] && [ip isOurFile:filename] ) {
      return ip;
    }
  }
  return nil;
}

+ (Input *)pluginForFile:(NSString *)filename
{
  Input *ip;

  ip = [self epluginForFile:filename];
  if ( ip == nil && [[cfg default_extension] length] > 0 ) {
    ip = [self epluginForFile:[filename stringByAppendingFormat:@".%@",
				[cfg default_extension]]];
  }
  return ip;
}

+ (void)playFile:(NSString *)filename
{
  Input *ip;

  if ( vis_list_lock == nil )
    vis_list_lock = [[NSLock alloc] init];
  
  if ( input_playing )
    [Input stop];

  ip = [self pluginForFile:filename];

  if ( ip != nil ) {
    [ip playFile:filename];

//    mysleep(30000); // let thigs get started

    [ip setEq:[cfg equalizer_active]
       preamp:[cfg equalizer_preamp]
	bands:[cfg eq_bands]];
    input_playing = YES;
    current_input_plugin = ip;
    playingFile = [filename retain];
  } else {
    NSRunAlertPanel(PACKAGE, @"No input plugin found for file: %@", @"OK", 
		    nil, nil, filename);    
  }
}

+ (void)updateVolume
{
  int b, v = (int)rint(([cfg volume]/51.0)*100);
  if ( [cfg balance] < 12 ) {
    b = ([cfg balance]*100)/12;
    [Input setVolumeLeft:v right:(v*b)/100];
  } else if ( [cfg balance] > 12) {
    b = 100-((([cfg balance]-12)*100)/12);
    [Input setVolumeLeft:(v*b)/100 right:v];
  } else {
    [Input setVolumeLeft:v right:v];
  }
}

+ (BOOL)isPlaying
{
  return input_playing;
}

  
+ (BOOL)isPaused
{
  return input_paused;
}

+ (void)donePlaying
{
  if ( input_playing ) {
    input_playing = NO;
    [playingFile release];
    playingFile = nil;
    [[NSNotificationCenter defaultCenter]
      postNotificationName:SongEndedNotification
		    object:nil
		  userInfo:nil];
  }
//  [[[NSApp delegate] mainView] setRate:0 freq:0 numChannels:0];
}

- (void)donePlaying
{
  [[self class] donePlaying];
}

+ (int)getTime
{
  return [current_input_plugin getTime];
}

+ (void)stop
{
  if ( input_playing ) {
    input_playing = NO;
    [playingFile release];
    playingFile = nil;
    [current_input_plugin stop];
    [[[NSApp delegate] mainView] setRate:0 freq:0 numChannels:0];
  }
}

+ (void)pause
{
  if  ( input_playing ) {
    input_paused = !input_paused;
    [current_input_plugin pause:input_paused];
    [[[NSApp delegate] mainView] 
      setPlayStatus:input_paused ? STATUS_PAUSE : STATUS_PLAY];
  }
}

+ (void)getSongInfo:(NSString *)fileName entry:(PlaylistEntry *)entry
{
  Input *ip;
  NSDictionary *res = nil;

  ip = [self pluginForFile:fileName];
  if ( ip != nil ) {
    res = [ip getSongInfoForFile:fileName];
    if ( res != 0 ) {
      [entry setTitle:[res objectForKey:@"title"]];
      [entry setLength:[[res objectForKey:@"length"] intValue]];
      [entry setAlbumName:[res objectForKey:@"albumName"]];
      [entry setArtistName:[res objectForKey:@"artistName"]];      
      [entry setSongName:[res objectForKey:@"songName"]];
    }
  }
}

+ (unsigned char *)getVis:(int)time
{
  VisEntry *vis = nil, *next_vis;
  unsigned char *ret=NULL;
  BOOL found;
  int i;

  [vis_list_lock lock];

  found=FALSE;
  for ( i = 0; i < [vis_list count] && !found; i++ ) {
    vis = [vis_list objectAtIndex:i];
    next_vis = nil;
    if ( i < [vis_list count]-1 )
      next_vis = [vis_list objectAtIndex:i+1];
    if ( (next_vis == nil || next_vis->time >= time) && vis->time < time) {
      if((vis->type==INPUT_VIS_ANALYZER
	  && [cfg vis_type] == VIS_ANALYZER
	  && (![cfg player_shaded] || ![cfg player_visible]))
	 || (vis->type == INPUT_VIS_VU && [cfg vis_type] == VIS_ANALYZER
	     && [cfg player_shaded] && [cfg player_visible])
	 || (vis->type == INPUT_VIS_SCOPE && [cfg vis_type] == VIS_SCOPE))
	{
	  ret = malloc(75);
	  memcpy(ret, vis->vis, 75);
	}
      found=TRUE;
    }
  }
  if ( found ) {
    BOOL done = FALSE;
    while (!done) {
      next_vis = [vis_list objectAtIndex:0];
      if ( next_vis == vis )
	done = TRUE;
      [vis_list removeObject:next_vis];
    }
  }
  [vis_list_lock unlock];
  return ret;
}

+ (void)getVolumeLeft:(int *)l right:(int*)r
{
  if ( input_playing ) {
    if ( [current_input_plugin getVolumeLeft:l right:r] )
      return;
    [selectedOutputPlugin getVolumeLeft:l right:r];
  }
}

+ (void)setVolumeLeft:(int)l right:(int)r
{
  if ( input_playing ) {
    if ( [current_input_plugin setVolumeLeft:l right:r] )
      return;
    [selectedOutputPlugin setVolumeLeft:l right:r];
  }
}

+ (void)fileInfoBox:(NSString *)filename
{
  Input *ip;

  ip = [self pluginForFile:filename];
  if ( ip != nil )
    [ip fileInfoBox:filename];
}

- (BOOL)enabledByDefault
{
  return YES;
}

- (NSArray *)scanDir:(NSString *)dirname
{
  return nil;
}

- (void)stop
{
}

- (void)pause:(BOOL)paused
{
}

- (void)seek:(int)time
{
}

- (BOOL)getVolumeLeft:(int *)l right:(int*)r
{
  return NO;
}

- (BOOL)setVolumeLeft:(int)l right:(int)r
{
  return NO;
}


- (NSDictionary *)getSongInfoForFile:(NSString *)filename
{
  return nil;
}

- (void)fileInfoBox:(NSString *)filename
{
}

- (void)setEq:(BOOL)on preamp:(float)preamp bands:(float *)bands
{
}

- (BOOL)isOurFile:(NSString *)filename
{
  return NO;
}

- (void)playFile:(NSString *)filename
{
}

- (int)getTime
{
  return -1;
}



+ (void)addVisTime:(int)time data:(const unsigned char *)s
			     type:(InputVisType)type

{
  VisEntry *vis;

  if([Input getVisType]==INPUT_VIS_OFF)
    return;

  vis = [[[VisEntry alloc] init] autorelease];
  vis->time = time;
  vis->type = type;
  memcpy(vis->vis, s, 75);
  [vis_list_lock lock];
  if ( vis_list == nil )
    vis_list = [[NSMutableArray array] retain];
  [vis_list addObject:vis];
  [vis_list_lock unlock];
}

#define GUINT16_SWAP_LE_BE(val)        ((unsigned short) ( \
    (((unsigned short) (val) & (unsigned short) 0x00ffU) << 8) | \
    (((unsigned short) (val) & (unsigned short) 0xff00U) >> 8)))

#define GUINT16_TO_LE(val)  ((unsigned short) (val))
#define GUINT16_TO_BE(val)  ((unsigned short) GUINT16_SWAP_LE_BE (val))
#define GINT16_TO_LE(val)  ((short) (val))
#define GINT16_TO_BE(val)  ((short) GUINT16_SWAP_LE_BE (val))
#define GUINT16_FROM_LE(val)    (GUINT16_TO_LE (val))
#define GUINT16_FROM_BE(val)    (GUINT16_TO_BE (val))
#define GINT16_FROM_LE(val)    (GINT16_TO_LE (val))
#define GINT16_FROM_BE(val)    (GINT16_TO_BE (val))

#define fixed short
#define N_WAVE          1024    /* dimension of Sinewave[] */
#define LOG2_N_WAVE     10      /* log2(N_WAVE) */
#define N_LOUD          100     /* dimension of Loudampl[] */
#define FIX_MPY(DEST,A,B)       DEST = ((long)(A) * (long)(B))>>15

extern fixed Sinewave[N_WAVE]; /* placed at end of this file for clarity */
extern  fixed Loudampl[N_LOUD];

/*
        fix_mpy() - fixed-point multiplication
*/
static fixed fix_mpy(fixed a, fixed b)
{
        FIX_MPY(a,a,b);
        return a;
}

static int db_from_ampl(fixed re, fixed im)
{
        static long loud2[N_LOUD] = {0};
        long v;
        int i;

        if(loud2[0] == 0) {
                loud2[0] = (long)Loudampl[0] * (long)Loudampl[0];
                for(i=1; i<N_LOUD; ++i) {
                        v = (long)Loudampl[i] * (long)Loudampl[i];
                        loud2[i] = v;
                        loud2[i-1] = (loud2[i-1]+v) / 2;
                }
        }

        v = (long)re * (long)re + (long)im * (long)im;

        for(i=0; i<N_LOUD; ++i)
                if(loud2[i] <= v)
                        break;

        return (-i);
}

/*
        fix_fft() - perform fast Fourier transform.

        if n>0 FFT is done, if n<0 inverse FFT is done
        fr[n],fi[n] are real,imaginary arrays, INPUT AND RESULT.
        size of data = 2**m
        set inverse to 0=dft, 1=idft
*/
static int fix_fft(fixed fr[], fixed fi[], int m, int inverse)
{
        int mr,nn,i,j,l,k,istep, n, scale, shift;
        fixed qr,qi,tr,ti,wr,wi;

                n = 1<<m;

        if(n > N_WAVE)
                return -1;

        mr = 0;
        nn = n - 1;
        scale = 0;

        /* decimation in time - re-order data */
        for(m=1; m<=nn; ++m) {
                l = n;
                do {
                        l >>= 1;
                } while(mr+l > nn);
                mr = (mr & (l-1)) + l;

                if(mr <= m) continue;
                tr = fr[m];
                fr[m] = fr[mr];
                fr[mr] = tr;
                ti = fi[m];
                fi[m] = fi[mr];
                fi[mr] = ti;
        }

        l = 1;
        k = LOG2_N_WAVE-1;
        while(l < n) {
                if(inverse) {
                        /* variable scaling, depending upon data */
                        shift = 0;
                        for(i=0; i<n; ++i) {
                                j = fr[i];
                                if(j < 0)
                                        j = -j;
                                m = fi[i];
                                if(m < 0)
                                        m = -m;
                                if(j > 16383 || m > 16383) {
                                        shift = 1;
                                        break;
                                }
                        }
                        if(shift)
                                ++scale;
                } else {
                        /* fixed scaling, for proper normalization -
                           there will be log2(n) passes, so this
                           results in an overall factor of 1/n,
                           distributed to maximize arithmetic accuracy. */
                        shift = 1;
                }
                /* it may not be obvious, but the shift will be performed
                   on each data point exactly once, during this pass. */
                istep = l << 1;
                for(m=0; m<l; ++m) {
                        j = m << k;
                        /* 0 <= j < N_WAVE/2 */
                        wr =  Sinewave[j+N_WAVE/4];
                        wi = -Sinewave[j];
                        if(inverse)
                                wi = -wi;
                        if(shift) {
                                wr >>= 1;
                                wi >>= 1;
                        }
                        for(i=m; i<n; i+=istep) {
                                j = i + l;
				tr = fix_mpy(wr,fr[j]) -
				  fix_mpy(wi,fi[j]);
				ti = fix_mpy(wr,fi[j]) +
				  fix_mpy(wi,fr[j]);
                                qr = fr[i];
                                qi = fi[i];
                                if(shift) {
                                        qr >>= 1;
                                        qi >>= 1;
                                }
                                fr[j] = qr - tr;
                                fi[j] = qi - ti;
                                fr[i] = qr + tr;
                                fi[i] = qi + ti;
                        }
                }
                --k;
                l = istep;
        }

        return scale;
}


/*      window() - apply a Hanning window       */
static void window(fixed fr[], int n)
{
        int i,j,k;

        j = N_WAVE/n;
        n >>= 1;
        for(i=0,k=N_WAVE/4; i<n; ++i,k+=j)
                FIX_MPY(fr[i],fr[i],16384-(Sinewave[k]>>1));
        n <<= 1;
        for(k-=j; i<n; ++i,k-=j)
                FIX_MPY(fr[i],fr[i],16384-(Sinewave[k]>>1));
}

/*      fix_loud() - compute loudness of freq-vis components.
        n should be ntot/2, where ntot was passed to fix_fft();
        6 dB is added to account for the omitted alias components.
        scale_shift should be the result of fix_fft(), if the time-series
        was obtained from an inverse FFT, 0 otherwise.
        loud[] is the loudness, in dB wrt 32767; will be +10 to -N_LOUD.
*/
static void fix_loud(fixed loud[], fixed fr[], fixed fi[], int n, int scale_shift)
{
        int i, max;

        max = 0;
        if(scale_shift > 0)
                max = 10;
        scale_shift = (scale_shift+1) * 6;

        for(i=0; i<n; ++i) {
                loud[i] = db_from_ampl(fr[i],fi[i]) + scale_shift;
                if(loud[i] > max)
                        loud[i] = max;
        }
}

+ (void)addVisPcmTime:(int)time 
	       format:(AFormat)fmt
	  numChannels:(int)nch
	      length:(int)length
		data:(void *)ptr
{
  short *left,*right;
  unsigned short *ptru16,vu,val;
  unsigned char *ptru8;
  int i,max,step,pos,spec_len,spec_base;
  char vis[75];
  short *re,*im,*loud;
  InputVisType type;
	
  if([Input getVisType]==INPUT_VIS_OFF)
    return;
	
  max=length/nch;
  if(fmt==FMT_U16_LE||fmt==FMT_U16_BE||fmt==FMT_U16_NE||fmt==FMT_S16_LE||fmt==FMT_S16_BE||fmt==FMT_S16_NE)
    max/=2;
  left=malloc(max*sizeof(short));
  right=malloc(max*sizeof(short));
	
	
  switch(fmt) {
  case	FMT_U8:
    ptru8=ptr;
    if(nch==1)
      for(i=0;i<max;i++)
	left[i]=((*ptru8++)^128)<<8;
    else
      for(i=0;i<max;i++)
	{
	  left[i]=((*ptru8++)^128)<<8;
	  right[i]=((*ptru8++)^128)<<8;
	}
    break;
  case	FMT_S8:
    ptru8=ptr;
    if(nch==1)
      for(i=0;i<max;i++)
	left[i]=(*ptru8++)<<8;
    else
      for(i=0;i<max;i++)
	{
	  left[i]=(*ptru8++)<<8;
	  right[i]=(*ptru8++)<<8;
	}
    break;
  case	FMT_U16_LE:
    ptru16=ptr;
    if(nch==1)
      for(i=0;i<max;i++)
	left[i]=GUINT16_FROM_LE(*ptru16++)^32768;
    else
      for(i=0;i<max;i++)
	{
	  left[i]=GUINT16_FROM_LE(*ptru16++)^32768;
	  right[i]=GUINT16_FROM_LE(*ptru16++)^32768;
	}
    break;
  case	FMT_U16_BE:
    ptru16=ptr;
    if(nch==1)
      for(i=0;i<max;i++)
	left[i]=GUINT16_FROM_BE(*ptru16++)^32768;
    else
      for(i=0;i<max;i++)
	{
	  left[i]=GUINT16_FROM_BE(*ptru16++)^32768;
	  right[i]=GUINT16_FROM_BE(*ptru16++)^32768;
	}
    break;
  case	FMT_U16_NE:
    ptru16=ptr;
    if(nch==1)
      for(i=0;i<max;i++)
	left[i]=(*ptru16++)^32768;
    else
      for(i=0;i<max;i++)
	{
	  left[i]=(*ptru16++)^32768;
	  right[i]=(*ptru16++)^32768;
	}
    break;
  case	FMT_S16_LE:
    ptru16=ptr;
    if(nch==1)
      for(i=0;i<max;i++)
	left[i]=GINT16_FROM_LE(*ptru16++);
    else
      for(i=0;i<max;i++)
	{
	  left[i]=GINT16_FROM_LE(*ptru16++);
	  right[i]=GINT16_FROM_LE(*ptru16++);
	}
    break;
  case	FMT_S16_BE:
    ptru16=ptr;
    if(nch==1)
      for(i=0;i<max;i++)
	left[i]=GINT16_FROM_BE(*ptru16++);
    else
      for(i=0;i<max;i++)
	{
	  left[i]=GINT16_FROM_BE(*ptru16++);
	  right[i]=GINT16_FROM_BE(*ptru16++);
	}
    break;
  case	FMT_S16_NE:
    ptru16=ptr;
    if(nch==1)
      for(i=0;i<max;i++)
	left[i]=(*ptru16++);
    else
      for(i=0;i<max;i++)
	{
	  left[i]=(*ptru16++);
	  right[i]=(*ptru16++);
	}
    break;
  }
  switch(type=[Input getVisType])
    {
    case	INPUT_VIS_ANALYZER:
      spec_base=(int)(log(max)/log(2));
      if(spec_base>10)
	spec_base=10;
      spec_len=(int)pow(2,spec_base);
      
      re=malloc(spec_len*sizeof(short));
      im=malloc(spec_len*sizeof(short));
      loud=malloc(spec_len*sizeof(short));
      
      for(i=0;i<spec_len;i++)
	{
	  if(nch==2)
	    re[i]=(left[i]+right[i])>>1;
	  else
	    re[i]=left[i];
	  im[i]=0;
	}			
      window(re,spec_len);
      fix_fft(re,im,spec_base,FALSE);
      fix_loud(loud,re,im,spec_len,0);
      step=spec_len/150;
      for(i=0,pos=0;i<75;i++,pos+=step)
	{
	  if(loud[pos]>-60)
	    vis[i]=(loud[pos]+60)/2;
	  else
	    vis[i]=0;
	  if(vis[i] > 15)
	    vis[i] = 15;
	}
      free(re);
      free(im);
      free(loud);
      break;
    case	INPUT_VIS_SCOPE:
      if(nch==2)
	for(i=0;i<max;i++)
	  left[i]=(left[i]+right[i])>>1;
      step=max/75;
      for(i=0,pos=0;i<75;i++,pos+=step)
	{
	  vis[i]=((left[pos])/2500)+6;
	  if(vis[i]>12)
	    vis[i]=12;
	  if(vis[i]<0)
	    vis[i]=0;
	}
      break;
    case	INPUT_VIS_VU:
      vu = 0;
      for(i = 0;i < max;i++)
	{
	  val = abs(left[i]);
	  if(val > vu)
	    vu = val;
	}
      vis[0] = (vu * 37) / 32768;
      if(vis[0] > 37) 
	vis[0] = 37;
      if(nch == 2)
	{
	  vu = 0;
	  for(i = 0;i < max;i++)
	    {
	      val = abs(right[i]);
	      if(val > vu)
		vu = val;
	    }
	  vis[1] = (vu * 37) / 32768;
	  if(vis[1] > 37) 
	    vis[1] = 37;
	  
	}
      else
	vis[1] = vis[0];
      
      
      break;
    case INPUT_VIS_OFF:
      break;
    }
  [Input addVisTime:time data:(const unsigned char *)vis type:type];
  free(left);
  free(right);
}

+ (InputVisType)getVisType
{
  if ( [cfg vis_type] == VIS_ANALYZER ) {
    if ( [cfg player_shaded] && [cfg player_visible] )
      return INPUT_VIS_VU;
    return INPUT_VIS_ANALYZER;
  }
  if ( [cfg vis_type] == VIS_SCOPE)
    return INPUT_VIS_SCOPE;
  return INPUT_VIS_OFF;
}

static fixed Sinewave[1024] = {
      0,    201,    402,    603,    804,   1005,   1206,   1406,
   1607,   1808,   2009,   2209,   2410,   2610,   2811,   3011,
   3211,   3411,   3611,   3811,   4011,   4210,   4409,   4608,
   4807,   5006,   5205,   5403,   5601,   5799,   5997,   6195,
   6392,   6589,   6786,   6982,   7179,   7375,   7571,   7766,
   7961,   8156,   8351,   8545,   8739,   8932,   9126,   9319,
   9511,   9703,   9895,  10087,  10278,  10469,  10659,  10849,
  11038,  11227,  11416,  11604,  11792,  11980,  12166,  12353,
  12539,  12724,  12909,  13094,  13278,  13462,  13645,  13827,
  14009,  14191,  14372,  14552,  14732,  14911,  15090,  15268,
  15446,  15623,  15799,  15975,  16150,  16325,  16499,  16672,
  16845,  17017,  17189,  17360,  17530,  17699,  17868,  18036,
  18204,  18371,  18537,  18702,  18867,  19031,  19194,  19357,
  19519,  19680,  19840,  20000,  20159,  20317,  20474,  20631,
  20787,  20942,  21096,  21249,  21402,  21554,  21705,  21855,
  22004,  22153,  22301,  22448,  22594,  22739,  22883,  23027,
  23169,  23311,  23452,  23592,  23731,  23869,  24006,  24143,
  24278,  24413,  24546,  24679,  24811,  24942,  25072,  25201,
  25329,  25456,  25582,  25707,  25831,  25954,  26077,  26198,
  26318,  26437,  26556,  26673,  26789,  26905,  27019,  27132,
  27244,  27355,  27466,  27575,  27683,  27790,  27896,  28001,
  28105,  28208,  28309,  28410,  28510,  28608,  28706,  28802,
  28897,  28992,  29085,  29177,  29268,  29358,  29446,  29534,
  29621,  29706,  29790,  29873,  29955,  30036,  30116,  30195,
  30272,  30349,  30424,  30498,  30571,  30643,  30713,  30783,
  30851,  30918,  30984,  31049,
  31113,  31175,  31236,  31297,
  31356,  31413,  31470,  31525,  31580,  31633,  31684,  31735,
  31785,  31833,  31880,  31926,  31970,  32014,  32056,  32097,
  32137,  32176,  32213,  32249,  32284,  32318,  32350,  32382,
  32412,  32441,  32468,  32495,  32520,  32544,  32567,  32588,
  32609,  32628,  32646,  32662,  32678,  32692,  32705,  32717,
  32727,  32736,  32744,  32751,  32757,  32761,  32764,  32766,
  32767,  32766,  32764,  32761,  32757,  32751,  32744,  32736,
  32727,  32717,  32705,  32692,  32678,  32662,  32646,  32628,
  32609,  32588,  32567,  32544,  32520,  32495,  32468,  32441,
  32412,  32382,  32350,  32318,  32284,  32249,  32213,  32176,
  32137,  32097,  32056,  32014,  31970,  31926,  31880,  31833,
  31785,  31735,  31684,  31633,  31580,  31525,  31470,  31413,
  31356,  31297,  31236,  31175,  31113,  31049,  30984,  30918,
  30851,  30783,  30713,  30643,  30571,  30498,  30424,  30349,
  30272,  30195,  30116,  30036,  29955,  29873,  29790,  29706,
  29621,  29534,  29446,  29358,  29268,  29177,  29085,  28992,
  28897,  28802,  28706,  28608,  28510,  28410,  28309,  28208,
  28105,  28001,  27896,  27790,  27683,  27575,  27466,  27355,
  27244,  27132,  27019,  26905,  26789,  26673,  26556,  26437,
  26318,  26198,  26077,  25954,  25831,  25707,  25582,  25456,
  25329,  25201,  25072,  24942,  24811,  24679,  24546,  24413,
  24278,  24143,  24006,  23869,  23731,  23592,  23452,  23311,
  23169,  23027,  22883,  22739,  22594,  22448,  22301,  22153,
  22004,  21855,  21705,  21554,  21402,  21249,  21096,  20942,
  20787,  20631,  20474,  20317,  20159,  20000,  19840,  19680,
  19519,  19357,  19194,  19031,  18867,  18702,  18537,  18371,
  18204,  18036,  17868,  17699,  17530,  17360,  17189,  17017,
  16845,  16672,  16499,  16325,  16150,  15975,  15799,  15623,
  15446,  15268,  15090,  14911,  14732,  14552,  14372,  14191,
  14009,  13827,  13645,  13462,  13278,  13094,  12909,  12724,
  12539,  12353,  12166,  11980,  11792,  11604,  11416,  11227,
  11038,  10849,  10659,  10469,  10278,  10087,   9895,   9703,
   9511,   9319,   9126,   8932,   8739,   8545,   8351,   8156,
   7961,   7766,   7571,   7375,   7179,   6982,   6786,   6589,
   6392,   6195,   5997,   5799,   5601,   5403,   5205,   5006,
   4807,   4608,   4409,   4210,   4011,   3811,   3611,   3411,
   3211,   3011,   2811,   2610,   2410,   2209,   2009,   1808,
   1607,   1406,   1206,   1005,    804,    603,    402,    201,
      0,   -201,   -402,   -603,   -804,  -1005,  -1206,  -1406,
  -1607,  -1808,  -2009,  -2209,  -2410,  -2610,  -2811,  -3011,
  -3211,  -3411,  -3611,  -3811,  -4011,  -4210,  -4409,  -4608,
  -4807,  -5006,  -5205,  -5403,  -5601,  -5799,  -5997,  -6195,
  -6392,  -6589,  -6786,  -6982,  -7179,  -7375,  -7571,  -7766,
  -7961,  -8156,  -8351,  -8545,  -8739,  -8932,  -9126,  -9319,
  -9511,  -9703,  -9895, -10087, -10278, -10469, -10659, -10849,
 -11038, -11227, -11416, -11604, -11792, -11980, -12166, -12353,
 -12539, -12724, -12909, -13094, -13278, -13462, -13645, -13827,
 -14009, -14191, -14372, -14552, -14732, -14911, -15090, -15268,
 -15446, -15623, -15799, -15975, -16150, -16325, -16499, -16672,
 -16845, -17017, -17189, -17360, -17530, -17699, -17868, -18036,
 -18204, -18371, -18537, -18702, -18867, -19031, -19194, -19357,
 -19519, -19680, -19840, -20000, -20159, -20317, -20474, -20631,
 -20787, -20942, -21096, -21249, -21402, -21554, -21705, -21855,
 -22004, -22153, -22301, -22448, -22594, -22739, -22883, -23027,
 -23169, -23311, -23452, -23592, -23731, -23869, -24006, -24143,
 -24278, -24413, -24546, -24679, -24811, -24942, -25072, -25201,
 -25329, -25456, -25582, -25707, -25831, -25954, -26077, -26198,
 -26318, -26437, -26556, -26673, -26789, -26905, -27019, -27132,
 -27244, -27355, -27466, -27575, -27683, -27790, -27896, -28001,
 -28105, -28208, -28309, -28410, -28510, -28608, -28706, -28802,
 -28897, -28992, -29085, -29177, -29268, -29358, -29446, -29534,
 -29621, -29706, -29790, -29873, -29955, -30036, -30116, -30195,
 -30272, -30349, -30424, -30498, -30571, -30643, -30713, -30783,
 -30851, -30918, -30984, -31049, -31113, -31175, -31236, -31297,
 -31356, -31413, -31470, -31525, -31580, -31633, -31684, -31735,
 -31785, -31833, -31880, -31926, -31970, -32014, -32056, -32097,
 -32137, -32176, -32213, -32249, -32284, -32318, -32350, -32382,
 -32412, -32441, -32468, -32495, -32520, -32544, -32567, -32588,
 -32609, -32628, -32646, -32662, -32678, -32692, -32705, -32717,
 -32727, -32736, -32744, -32751, -32757, -32761, -32764, -32766,
 -32767, -32766, -32764, -32761, -32757, -32751, -32744, -32736,
 -32727, -32717, -32705, -32692, -32678, -32662, -32646, -32628,
 -32609, -32588, -32567, -32544, -32520, -32495, -32468, -32441,
 -32412, -32382, -32350, -32318, -32284, -32249, -32213, -32176,
 -32137, -32097, -32056, -32014, -31970, -31926, -31880, -31833,
 -31785, -31735, -31684, -31633, -31580, -31525, -31470, -31413,
 -31356, -31297, -31236, -31175, -31113, -31049, -30984, -30918,
 -30851, -30783, -30713, -30643, -30571, -30498, -30424, -30349,
 -30272, -30195, -30116, -30036, -29955, -29873, -29790, -29706,
 -29621, -29534, -29446, -29358, -29268, -29177, -29085, -28992,
 -28897, -28802, -28706, -28608, -28510, -28410, -28309, -28208,
 -28105, -28001, -27896, -27790, -27683, -27575, -27466, -27355,
 -27244, -27132, -27019, -26905, -26789, -26673, -26556, -26437,
 -26318, -26198, -26077, -25954, -25831, -25707, -25582, -25456,
 -25329, -25201, -25072, -24942, -24811, -24679, -24546, -24413,
 -24278, -24143, -24006, -23869, -23731, -23592, -23452, -23311,
 -23169, -23027, -22883, -22739, -22594, -22448, -22301, -22153,
 -22004, -21855, -21705, -21554, -21402, -21249, -21096, -20942,
 -20787, -20631, -20474, -20317, -20159, -20000, -19840, -19680,
 -19519, -19357, -19194, -19031, -18867, -18702, -18537, -18371,
 -18204, -18036, -17868, -17699, -17530, -17360, -17189, -17017,
 -16845, -16672, -16499, -16325, -16150, -15975, -15799, -15623,
 -15446, -15268, -15090, -14911, -14732, -14552, -14372, -14191,
 -14009, -13827, -13645, -13462, -13278, -13094, -12909, -12724,
 -12539, -12353, -12166, -11980, -11792, -11604, -11416, -11227,
 -11038, -10849, -10659, -10469, -10278, -10087,  -9895,  -9703,
  -9511,  -9319,  -9126,  -8932,  -8739,  -8545,  -8351,  -8156,
  -7961,  -7766,  -7571,  -7375,  -7179,  -6982,  -6786,  -6589,
  -6392,  -6195,  -5997,  -5799,  -5601,  -5403,  -5205,  -5006,
  -4807,  -4608,  -4409,  -4210,  -4011,  -3811,  -3611,  -3411,
  -3211,  -3011,  -2811,  -2610,  -2410,  -2209,  -2009,  -1808,
  -1607,  -1406,  -1206,  -1005,   -804,   -603,   -402,   -201,
};

#if N_LOUD != 100
        ERROR: N_LOUD != 100
#endif
static fixed Loudampl[100] = {
  32767,  29203,  26027,  23197,  20674,  18426,  16422,  14636,
  13044,  11626,  10361,   9234,   8230,   7335,   6537,   5826,
   5193,   4628,   4125,   3676,   3276,   2920,   2602,   2319,
   2067,   1842,   1642,   1463,   1304,   1162,   1036,    923,
    823,    733,    653,    582,    519,    462,    412,    367,
    327,    292,    260,    231,    206,    184,    164,    146,
    130,    116,    103,     92,     82,     73,     65,     58,
     51,     46,     41,     36,     32,     29,     26,     23,
     20,     18,     16,     14,     13,     11,     10,      9,
      8,      7,      6,      5,      5,      4,      4,      3,
      3,      2,      2,      2,      2,      1,      1,      1,
      1,      1,      1,      0,      0,      0,      0,      0,
      0,      0,      0,      0,
};



@end


@implementation Output

+ (void)registerPlugin:(Plugin *)ip
{
  if ( output_plugins == nil )
    output_plugins = [[NSMutableArray array] retain];
  
  [output_plugins addObject:ip];
}


+ (NSArray *)plugins
{

  return output_plugins;
}

- (void)getVolumeLeft:(int *)l right:(int *)r
{
}

- (void)setVolumeLeft:(int )l right:(int)r
{
}

- (BOOL)openAudioFormat:(AFormat )fmt rate:(int)rate numChannels:(int)nch
{
  return NO;
}

- (void)writeAudioData:(const void *)ptr length:(int)length
{
}

- (void)closeAudio
{
}

- (void)flush:(int)time
{
}

- (void)pause:(BOOL) paused
{
}

- (int)bufferFree
{
  return 0;
}

- (BOOL)bufferPlaying
{
  return 0;
}

- (int)outputTime
{
  return 0;
}

- (int)writtenTime
{
  return 0;
}

- (NSString *)description
{
  return [description autorelease];
}

+ (Output *)output
{
  return selectedOutputPlugin;
}

- (void)wait
{
}

- (BOOL)enabledByDefault
{
  return NO;
}

@end

@implementation General

+ (void)registerPlugin:(Plugin *)op
{
}

+ (NSArray *)plugins
{
  return nil;
}
@end


@implementation Effect
+ (void)registerPlugin:(Plugin *)op
{
  if ( effect_plugins == nil )
    effect_plugins = [[NSMutableArray array] retain];
  [effect_plugins addObject:op];
}

+ (NSArray *)plugins
{
  return effect_plugins;
}

+ (int)modSampleData:(short int *)data
	      length:(int)length
       bitsPerSample:(int)bps
	 numChannels:(int)numChannels
		freq:(int)srate
{
  int new_len = length;
  int i;
  Effect *ep;
  
  for ( i = 0; i < [effect_plugins count]; i++ ) {
    ep = [effect_plugins objectAtIndex:i];
    if ( [ep enabled] )
      new_len = [ep modSampleData:data
			   length:new_len
		    bitsPerSample:bps
		      numChannels:numChannels
			     freq:srate];
    
  }
  return new_len;
}

- (void)cleanup
{
}

- (int)modSampleData:(short int *)data
	      length:(int)length
       bitsPerSample:(int)bps
	 numChannels:(int)numChannels
		freq:(int)srate
{
  return length;
}

@end

@implementation Plugin

- initWithDescription:(NSString *)_description
{
  description = [_description retain];
  return [super init];
}

- (void)setEnabled:(BOOL)val
{
  [cfg setPluginEnabled:[self class] value:val];
}

- (BOOL)enabled
{
  return [cfg pluginEnabled:[self class]];
}


- (NSString *)description
{
  return [description autorelease];
}

- (void)about
{
}

- (void)configure
{
}

- (BOOL)hasAbout
{
  return NO;
}

- (BOOL)hasConfigure
{
  return NO;
}

+ (void)bundleDidLoad:(NSNotification *)notification
{
  NSArray *classes;
  int i;
  NSString *className, *bundlePath;
  BOOL gotOne = FALSE;
  Class aClass;
  Plugin *plugin;
  
  bundlePath = [[notification object] bundlePath];
  classes = [[notification userInfo] objectForKey:@"NSLoadedClasses"];
  
  for ( i = 0; i < [classes count]; i++ ) {
    className = [classes objectAtIndex:i];
    aClass = NSClassFromString(className);
    if ( [aClass conformsToProtocol:@protocol(PluginProtocol)] ) {
      plugin = [[aClass alloc] init];
      if ( [plugin isKindOfClass:[Output class]] ) {
	[Output registerPlugin:plugin];
	gotOne = TRUE;
      } else if ( [plugin isKindOfClass:[Input class]] ) {
	[Input registerPlugin:plugin];
	gotOne = TRUE;
      } else if ( [plugin isKindOfClass:[General class]] ) {
	[General registerPlugin:plugin];
	gotOne = TRUE;
      } else if ( [plugin isKindOfClass:[Effect class]] ) {
	[Effect registerPlugin:plugin];
	gotOne = TRUE;
      } else
	NSLog(@"MacOSXAmp: unknow plugin type at %@", bundlePath);
    }
  }
  if ( gotOne == FALSE )
    NSLog(@"MacOSXAmp: loaded a plugin bundle(%@) with no plugin classes",
	  bundlePath);
}

+ (void)loadPluginsAtPath:(NSString *)path
{
  NSDirectoryEnumerator *de;
  NSString *entry, *fullPath, *bname, *dname;
  NSBundle *bundle;
  NSFileManager *fm = [NSFileManager defaultManager];

  de = [fm enumeratorAtPath:path];
  
  while ( entry = [de nextObject] ) {
    if ( [[entry pathExtension] isEqualToString:@"bundle"] ) {

      fullPath = [path stringByAppendingPathComponent:entry];

      //seems to be a bug in NSBundle, if only a _debug executable exists
      //it will not be loaded, so we'll trick it
      bname = [entry stringByDeletingPathExtension];
      bname = [fullPath stringByAppendingPathComponent:bname];
      dname = [bname stringByAppendingString:@"_debug"];
      
      if ( [fm fileExistsAtPath:bname] == NO
	   && [fm fileExistsAtPath:dname] ) {
	[fm createSymbolicLinkAtPath:bname pathContent:dname];
      }

//      NSLog(@"loading bundle: %@", entry);
      bundle = [NSBundle bundleWithPath:fullPath];
      [bundle principalClass];
    }
  }
}

+ (void)updateOutputPlugin
{
  int i;
  for ( i = 0; i < [output_plugins count]; i++ ) {
    if ( [[output_plugins objectAtIndex:i] enabled] )
      selectedOutputPlugin = [output_plugins objectAtIndex:i];
  }
  if ( selectedOutputPlugin == nil && [output_plugins count] > 0 ) {
    Output *pl;
    for ( i = 0; i < [output_plugins count]; i++ ) {
      pl = [output_plugins objectAtIndex:0];
      if ([pl enabledByDefault] ) {
	selectedOutputPlugin = pl;
	[selectedOutputPlugin setEnabled:YES];
	break;
      }
    }
  }
}



+ (void)loadAllPluginBundles
{
  NSArray *searchPath;
  int i;

  [[NSNotificationCenter defaultCenter] 
    addObserver:self
       selector:@selector(bundleDidLoad:)
	   name:NSBundleDidLoadNotification
	 object:nil];
  

  searchPath = NSSearchPathForDirectoriesInDomains(NSLibraryDirectory, 
						   NSAllDomainsMask, YES);

  [self loadPluginsAtPath:[[NSBundle mainBundle] resourcePath]];
  
  for ( i = 0; i < [searchPath count]; i++ ) {
    [self loadPluginsAtPath:[[[searchPath objectAtIndex:i] 
			 stringByAppendingPathComponent:PACKAGE] 
			 stringByAppendingPathComponent:@"Plugins"]];
  }

  [self updateOutputPlugin];
    
  {
    Input *plugin;
    
    for ( i = 0; i < [input_plugins count]; i++ ) {
      plugin = [input_plugins objectAtIndex:i];
      if ( [cfg pluginEnabledIsSet:[plugin class]] == NO 
	   && [plugin enabledByDefault] )
	[plugin setEnabled:YES];
    }
  }
}

 
@end

void mysleep(unsigned int microseconds)
{
#if 1
  NSDate *date;
  date = [NSDate dateWithTimeIntervalSinceNow:microseconds/10000];    
  [NSThread sleepUntilDate:date];
#endif
}
