
/**
 * OpenAmp Audio Stream
 *
 * Wilfredo Sanchez | wsanchez@apple.com
 * September 20, 1998
 **/

#import  <Foundation/NSLock.h>
#import  <SoundKit/SoundKit.h>
#include <assert.h>

//#include "common.h"
//#include "SoundDecoder.h"
#import  "SoundStream.h"
#import "Plugins.h"

#undef DEBUG
#ifndef DEBUG
#define DEBUG 0
#endif

@implementation SoundStream

- (BOOL)openWithFormat:(int)afmt rate:(int)rate channels:(int)nch
{
//  int aBitRate    = bitrate[aLayer->version][aLayer->lay-1][aLayer->bitrate_index];
  NXSoundDeviceError anError = NX_SoundDeviceErrorNone;

  frequencyRate = rate;
  numChannels = nch;

#if 0
  if (rate != 22050 && rate != 44100 )
    {
      NSLog(@"Sample frequency %d is not supported.", rate);
      [self release];
      return NO;
    }
#endif

#if 0
  if (aBitRate != 128) {
    NSLog(@"Bit rate %d is not supported.", aBitRate);
    [self release];
    return nil;
  }
#endif

  if (! [NXSoundOut setUseSeparateThread:YES]) {
    NSLog(@"Failed to init sound output.\n");
    errno = ENXIO;
    return NO;
  }

  switch(afmt) {
  case	FMT_U8:
    format = NX_SoundStreamDataEncoding_Linear8;
    break;
  case	FMT_S16_NE:
    format = NX_SoundStreamDataEncoding_Linear16;
    break;
  case	FMT_S8:
  case	FMT_U16_BE:
  case	FMT_U16_NE:
  case	FMT_S16_LE:
  case	FMT_S16_BE:
    NSLog(@"unsupported sound format");
    return NO;
    break;
  }	
  

  mySoundDevice = [[NXSoundOut alloc] init];
  myStreamParameters = nil;

  myStreamParameters = [[NXSoundParameters alloc] init];

  [myStreamParameters setParameter:NX_SoundStreamSamplingRate 
			     toInt:rate];
  [myStreamParameters setParameter:NX_SoundStreamChannelCount 
			     toInt:nch];
  [myStreamParameters setParameter:NX_SoundStreamDataEncoding 
			     toInt:format];
  
  if (!(myPlayStream = [[NXPlayStream alloc] initOnDevice:mySoundDevice
				   withParameters:myStreamParameters]))
    {
      NSLog(@"Failed to init play stream: %@",
	    [NXSoundDevice textForError: anError]);
      return NO;
    }

  [myPlayStream setDelegate: self];

  if ((anError = [myPlayStream activate]) != NX_SoundDeviceErrorNone) {
    NSLog(@"Failed to activate play stream %@: %@",
	  myPlayStream, [NXSoundDevice textForError: anError]);
    return NO;
  }

//  [myPlayStream pause:self];

  myPlayingLock = [[NSLock alloc] init];
  myCounterLock = [[NSLock alloc] init];

  myQueuedBuffersCount    = 0;
  myRemainingBuffersCount = 0;

  ebps=(frequencyRate * numChannels)*2;
  bps = ebps;
#if 0
  if(format==AFMT_U16_BE||format==AFMT_U16_LE||format==AFMT_S16_BE||format==AFMT_S16_LE)
    ebps*=2;
#endif

  buffer_sizes = [[NSMutableDictionary dictionary] retain];


  return YES;
}

- (void)dealloc
{
  [myPlayStream release ];
  [mySoundDevice release ];
  [myStreamParameters release ];
  [myPlayingLock release ];
  [myCounterLock release ];
  [buffer_sizes release];
  
  [super dealloc];
}


- (void)write:(void *)ptr length:(int)length
{
  int  aCount;
  NXSoundDeviceError anError;

  [myCounterLock lock];
  {
    aCount = myQueuedBuffersCount++;
    myRemainingBuffersCount++;
    [buffer_sizes setObject:[[NSNumber alloc] initWithInt:length]
		     forKey:[[NSNumber alloc] initWithInt:aCount]];
  }
  [myCounterLock unlock];

  if (myRemainingBuffersCount == 1) {
//    NSLog(@"lock myPlayingLock");
    [myPlayingLock lock];
  }

#if DEBUG
  NSLog(@"Playing %d bytes via %@. Count = %d.", length, myPlayStream, aCount);
#endif

  anError = [myPlayStream playBuffer:ptr
				size:length
				 tag:aCount];
  
  if (anError) 
    NSLog(@"Error while playing %ld bytes via stream %@: %@",
	  length, myPlayStream, [NXSoundDevice textForError: anError]);

  written += length;
}

- (void)wait
{
//  NSLog(@"waiting");
  [myPlayingLock lock];
  [myPlayingLock unlock];
}

- (void)close
{ 
  [myPlayStream abort:self];
  [myPlayStream deactivate];
//  NSLog(@"close");
  [myPlayingLock unlock]; 
}
- (void)pause:(short)p
{ 
  if ( p )
    [myPlayStream pause:self];
  else
    [myPlayStream resume:self];
}

- (void) soundStreamDidUnderrun: (NXSoundStream*) aSender
{
    NSLog(@"Buffer underrun in sound stream %@.", aSender);
}

- (void) soundStream: (NXSoundStream*) aSender
   didCompleteBuffer: (int           ) aTag
{
  int aCount;
  NSNumber *key = [[NSNumber alloc] initWithInt:aTag];

  [myCounterLock lock];
  {
    aCount = --myRemainingBuffersCount;
    bytes_output += [[buffer_sizes objectForKey:key] intValue];
    [buffer_sizes removeObjectForKey:key];
  }
  [myCounterLock unlock];

  // If this is critical code, it should be in the locked
  // section above. But's it's just the display update code
  // and we don't want to hold up the decoder thread (which
  // calls playBuffer) from processing while a display update
  // is going on.
  // It is therefore possible that myQueuedBuffersCount here
  // could be stale, but that just means the progress bar is a
  // little off. Because of the above lock, however, this code
  // won't execute twice simultaneously, which could cause bad
  // drawing.
#if 0
  if ( [delegate responsToSelector:@selector(soundStream:outputBytes:)] )
    [decoder soundStream:self outputBytes:bytes_output];
#endif
  
#if DEBUG
  NSLog(@"Sound stream %@ finished playing buffer %d. Queue = %d.",
	aSender, aTag, aCount);
#endif

//  printf("aCount = %d\n", aCount);
  if (aCount <= 1) {
//    NSLog(@"unlock myPlayingLock");
    [myPlayingLock unlock];
  }

  assert(aCount >= 0);
}

- (BOOL)isActive
{
  return [myPlayStream isActive];
}

- (int)getOutputTime
{
  int bytes, ot;

  bytes = bytes_output;
	
  if ( bytes < 0 )
    bytes=0;

  ot = (int)((float)((bytes)*1000.0)/(float)ebps);

  return ot;
}

- (int)getWrittenTime
{
  int wt = (int)(((float)written*1000.0)/(float)(bps));
  return wt;
}

- (int)getFree
{
  int gfree =  myRemainingBuffersCount < 50 ? (50-myRemainingBuffersCount)*10000 : 0;
  return gfree;
}

@end
