/*
 * This file contains the current known set of flags and constants for the
 * known architectures.
 */
#import <mach/machine.h>
#import <stuff/bytesex.h>

/*
 * The structure describing an architecture flag with the string of the flag
 * name, and the cputype and cpusubtype.
 */
struct arch_flag {
    char *name;
    cpu_type_t cputype;
    cpu_subtype_t cpusubtype;
};

/*
 * get_arch_from_flag() is passed a name of an architecture flag and returns
 * zero if that flag is not known and non-zero if the flag is known.
 * If the pointer to the arch_flag is not NULL it is filled in with the
 * arch_flag struct that matches the name.
 */
__private_extern__ int get_arch_from_flag(
    char *name,
    struct arch_flag *arch_flag);

/*
 * get_arch_from_host() gets the architecture from the host this is running on
 * and returns zero if the architecture is not known and zero if the
 * architecture is known.  If the parameters family_arch_flag and
 * specific_arch_flag are not NULL they get fill in with the family
 * architecture and specific architecure for the host.  If the architecture
 * is unknown and the parameters are not NULL then all fields are set to zero.
 */
__private_extern__ int get_arch_from_host(
    struct arch_flag *family_arch_flag,
    struct arch_flag *specific_arch_flag);

/*
 * get_arch_flags() returns a pointer to an array of all currently know
 * architecture flags (terminated with an entry with all zeros).
 */
__private_extern__ const struct arch_flag *get_arch_flags(
    void);

/*
 * arch_usage() is called when an unknown architecture flag is encountered.
 * It prints the currently know architecture flags on stderr.
 */
__private_extern__ void arch_usage(
    void);

/*
 * set_arch_flag_name() sets the name field of the specified arch_flag to
 * match it's cputype and cpusubtype.  The string is allocated via malloc by
 * the routines in "allocate.h" and errors are handled by the routines in
 * "error.h".
 */
__private_extern__ void set_arch_flag_name(
    struct arch_flag *p);

/*
 * get_arch_name_from_types() returns the name of the architecture for the
 * specified cputype and cpusubtype if known.  If unknown it returns a pointer
 * to the string "unknown".
 */
__private_extern__ const char *get_arch_name_from_types(
    cpu_type_t cputype,
    cpu_subtype_t cpusubtype);

/*
 * get_arch_family_from_cputype() returns the family architecture for the
 * specified cputype if known.  If unknown it returns NULL.
 */
__private_extern__ const struct arch_flag *get_arch_family_from_cputype(
    cpu_type_t cputype);

/*
 * get_byte_sex_from_flag() returns the byte sex of the architecture for the
 * specified cputype and cpusubtype if known.  If unknown it returns
 * UNKNOWN_BYTE_SEX.  If the bytesex can be determined directly as in the case
 * of reading a magic number from a file that should be done and this routine
 * should not be used as it could be out of date.
 */
__private_extern__ enum byte_sex get_byte_sex_from_flag(
    const struct arch_flag *flag);

/*
 * get_stack_direction_from_flag() returns the direction the stack grows as
 * either positive (+1) or negative (-1) of the architecture for the
 * specified cputype and cpusubtype if known.  If unknown it returns 0.
 */
__private_extern__ long get_stack_direction_from_flag(
    const struct arch_flag *flag);

/*
 * get_stack_addr_from_flag() returns the default starting address of the user
 * stack.
 */
__private_extern__ unsigned long get_stack_addr_from_flag(
    const struct arch_flag *flag);

/*
 * get_stack_size_from_flag() returns the default size of the userstack.
 */
__private_extern__ unsigned long get_stack_size_from_flag(
    const struct arch_flag *flag);
