
/*
 * MartinView:  a BackSpace.app v3.0 module             by Jeffrey Adams
 * version 1.0                                             jeffa@wri.com
 * version 2.0	ported to FreeSpace for Mac OS X Server by Jeff Sickel, jas@sickel.com
 */

#import "MartinView.h"

#import <Foundation/Foundation.h>

#import <math.h>
#import <sys/ucred.h>
#import <unistd.h>

NSString *fname[Nfunc] = { @"martin1", @"martin2", @"ejk1", @"ejk2" };

/* The file used to store remembered fractals */
NSString *fileName = nil;

@implementation MartinView

+ (void)initialize
{
    NSString *neg = [NSString stringWithFormat:@"-1"];
    NSString *one = [NSString stringWithFormat:@"1"];
    NSString *yes = [NSString stringWithFormat:@"YES"];
    NSString *no = [NSString stringWithFormat:@"NO"];
    NSString *dynamic = [NSString stringWithFormat:@"%d", STARTDYNAMPOINTS];
    NSDictionary *MartinViewDefaults = [NSDictionary dictionaryWithObjectsAndKeys:
        neg, @"MartinFunction", neg, @"MartinMaxTot", neg, @"MartinMaxIn",
        neg, @"MartinColorInt", yes, @"MartinRandomCol", one, @"MartinColorMode",
        no, @"MartinUseFile", dynamic, @"MartinDynamicFlush",
        nil];

    [[NSUserDefaults standardUserDefaults] registerDefaults:MartinViewDefaults];

    fileName =
        [[[NSString stringWithCString:"~/.martinView"] stringByExpandingTildeInPath] retain];
}

/* Implemented so you can screen grab the image if you want to (at least I wanted to!) */
- (void)pause:(id)sender
{
	isPaused = [sender state];
}

/* Inherited method for setting the graphics state before sending OneSteps */
- (void)didLockFocus
{
    [currColor set];
}

- (void)oneStep
{
    NSRect *currentPixel = pixels;

    if (isPaused) return;

    /* Stay in here until we hit our flush buffer limit or we hit the maximum
        total points allowed */
    while ((++nd < nD) && (++nP < mxP)) {

        switch (Function) {
            case Ejk1: x1 = y  - ( (x>0) ? (B*x-C) : -(B*x-C) ); break;
            case Martin1:  x1 = y  - ( (x<0) ? sqrt(fabs(B*x-C)) : -sqrt(fabs(B*x-C)) ); break;
            case Ejk2: x1 = y  - ( (x<0) ? log(fabs(B*x-C)) : -log(fabs(B*x-C)) ); break;
            case Martin2:  x1 = y  - sin(x); break;
        }
        y = A - x; x=x1;

        /* seed perturbation */
        if (Pn && ++pn > Pn) {
            x += (x>0) ? -Pv : Pv;
            y += (y>0) ? -Pv : Pv;
            pn = 0;
        }

        /* Do we need to change the color?  */
        if (++nc > nC) {
            color = (color+1)%Ncolors;
            /* Since we are changing color, send buffered points to screen */
            if (numPixels) {
                NSRectFillList(pixels, numPixels);
                numPixels = 0; currentPixel = pixels;
            }

            if (Randomcolor) {
                // too much retain release, but it's needed in order to use NSColor
                [currColor release];
                currColor = (Color) ?
                [NSColor colorWithCalibratedRed:Ranf() green:Ranf() blue:Ranf() alpha:1.0]
                : [NSColor colorWithCalibratedWhite:Ranf() alpha:1.0];
                currColor = [currColor retain];
            } else {
                [currColor release];
                currColor = [colors[color] retain];
            }
            [currColor set];
            nc = 0;
        }

        /*  If we are not within the screen range, do not store them  */
        iy = cy + Zf*y; if (iy < 0 || iy > mxY) continue;
        ix = cx + Zf*x; if (ix < 0 || ix > mxX) continue;

        /* Update the data needed for the NSRectFillList */
        currentPixel->origin.x = ix;  currentPixel->origin.y = iy;
        ++numPixels; ++currentPixel;

        /* Have we hit the in-range limit, if so, send the buffer to the screen */
        if (++np > mxp) {
            if (numPixels) {
                NSRectFillList(pixels, numPixels);
                numPixels = 0; currentPixel = pixels;
            }
            [self newOne:self];
            break;
        }
    }
    nd = 0;
    /* Flush buffer full, display pixels */
    if (numPixels) {
        NSRectFillList(pixels, numPixels);
        numPixels = 0; currentPixel = pixels;
    }
    /* If we left because we hit max total limit, give us a new one */
    if(nP == mxP)
        [self newOne:self];
}

- (void)newOne:(id)sender
{
    NSUserDefaults *defaults = [NSUserDefaults standardUserDefaults];
    double  A1, B1, C1;        /* 2nd part of parameter range             */
    NSString *name = nil;

    [self display];
    if (!sharedInspectorPanel) return;

    /* Record current settings in Defaults database */
    [defaults setInteger:(([funcAuto state]) ? -1 : Function) forKey:@"MartinFunction"];
    [defaults setInteger:(([maxTotFlag state]) ? -1 : [maxTotalPts intValue])
                  forKey:@"MartinMaxTot"];
    [defaults setInteger:(([maxInFlag state]) ? -1 : [maxInRangePts intValue])
                  forKey:@"MartinMaxIn"];
    [defaults setInteger:(([colFlag state]) ? -1 : [colInterval intValue])
                  forKey:@"MartinColorInt"];
    [defaults setInteger:[dynamFlush intValue] forKey:@"MartinDynamicFlush"];


    np = 0; nP = 0; nd = 0; numPixels=0;
    x = y = 0; color = -1;
    Ranfset(time(0));
    W = (int)[self bounds].size.width;
    H = (int)[self bounds].size.height;
    mxX = W-1; mxY = H-1;

    /* If we are using a file, load the settings from '.martinView'  */
    if ([defaults boolForKey:@"MartinUseFile"]) {
        NSString *stream = nil;

        if (!(stream = [NSString stringWithContentsOfFile:file])) {
            [useFileFlag setState:NO];
            [self newOne:self];
        } else {
            NSScanner *scanner = [NSScanner scannerWithString:stream];
            NSString *lf = [NSString stringWithFormat:@"\n"];	// needs other line feeds
            NSCharacterSet *lineEndSet = [NSCharacterSet characterSetWithCharactersInString:lf];

            while ([scanner isAtEnd] == NO) {
                [scanner scanUpToCharactersFromSet:lineEndSet intoString:&name];

                [scanner scanInt:&Pn];
                [scanner scanDouble:&Pv];

                [scanner scanInt:&Function];

                [scanner scanDouble:&A];
                [scanner scanDouble:&B];
                [scanner scanDouble:&C];
                [scanner scanDouble:&Zf];

                [scanner scanInt:&mxp];
                [scanner scanInt:&mxP];
                [scanner scanInt:&nC];

                [scanner scanInt:&moveX];
                [scanner scanInt:&moveY];
            }
        }
    } else {

        if ([seedIntFlag state]) Pn = pow(10., 1+Ranf()*3);
        else Pn = [seedPertInt intValue];
        if ([seedValFlag state]) Pv = pow(10.,Ranf()*3);
        else Pv = [seedPertVal doubleValue];

        /* provide default hopalong parameters if needed */
        if ([funcAuto state]) {
            double r = Ranf();
            if      (r < 0.40) Function = Martin1;
            else if (r < 0.70) Function = Ejk1;
            else if (r < 0.90) Function = Ejk2;
            else               Function = Martin2;
        }
        if (Function == Martin1) {
            if ([aFlag state]) A = 40 + Ranf()*1500;
            if ([bFlag state]) B = 3  + Ranf()*17;
            if ([cFlag state]) C = 100 + Ranf()*3000;
        }
        else if (Function == Martin2) {
            if ([aFlag state]) A = 3.075927 + Ranf()*0.14;
        }
        else if (Function == Ejk1) {
            if ([aFlag state]) A = Ranf()*500;
            if ([bFlag state]) B = Ranf()*.40;
            if ([cFlag state]) C = 10 + Ranf()*100;
        }
        else if (Function == Ejk2) {
            if ([aFlag state]) A = Ranf()*500;
            if ([bFlag state]) B = pow(10.,6+Ranf()*24);
            if ([cFlag state]) C = pow(10.,  Ranf()*9);
        }
        if(![aFlag state]) A = [hopAField doubleValue];
        if(![bFlag state]) B = [hopBField doubleValue];
        if(![cFlag state]) C = [hopCField doubleValue];

        if (A1 = [a1Flag state]) A = Min(A,A1) + Ranf()*(Max(A,A1) - Min(A,A1));
        if ([afFlag state] && Ranf()<0.5) A = -A;
        if (B1 = [b1Flag state]) B = Min(B,B1) + Ranf()*(Max(B,B1) - Min(B,B1));
        if ([bfFlag state] && Ranf()<0.5) B = -B;
        if (C1 = [c1Flag state]) C = Min(C,C1) + Ranf()*(Max(C,C1) - Min(C,C1));
        if ([cfFlag state] && Ranf()<0.5) C = -C;

        if ([magFlag state]) Zf = (Function == Martin2) ? 4.0 : 1.0;
        else Zf = [magField doubleValue];

        // jas 23-Sep-1999  this is actually a spot that makes MartinView appear slower since
        //                  there wasn't a randomization of the max total and max in-range
        if ([maxInFlag state]) {
            // was:   mxp = (int) (0.40* (float)(W*H));
            mxp = Ranf()*(Max(mxp,(int)(0.40* (float)(W*H)))
                          - Min(mxp,(int)(0.40* (float)(W*H))));
            mxp = MIN(abs(mxp),(int)(0.40* (float)(W*H)));
        } else mxp = [maxInRangePts intValue];
        if ([maxTotFlag state]) {
            // was:   mxP = 4 * mxp;
            mxP = Ranf()*(Max(mxP,4 * mxp)) - Min(mxP, 4 * mxp);
            mxP = MIN(abs(mxP),(int)(0.40* (float)(W*H)));
        }
        else mxP = [maxTotalPts intValue];

        /* color processing */
        if ([colFlag state]) nC = (mxP/Ncolors)/2;
        else nC = [colInterval intValue];
    }

    nc = nC;
    cx = W/2+moveX; cy = H/2+moveY;
    [displaceX setIntValue:moveX];
    [displaceY setIntValue:moveY];
    [seedPertInt setIntValue:Pn];
    [seedPertVal setDoubleValue:Pv];
    [dynamFlush setIntValue:nD];
    [funcButton setTitle:((Function != -1) ? fname[Function] : fname[0])];
    [hopAField setDoubleValue:A];
    [hopBField setDoubleValue:B];
    [hopCField setDoubleValue:C];
    [magField setDoubleValue:Zf];
    [maxInRangePts setIntValue:mxp];
    [maxTotalPts setIntValue:mxP];
    [colInterval setIntValue:nC];
}

- (void)changeFunc:(id)sender
{	
    Function = [sender tag];
    [funcAuto setState:NO];
    [[NSUserDefaults standardUserDefaults] setInteger:Function forKey:@"MartinFunction"];
    [self newOne:sender];
}

- (void)changeDynam:(id)sender
{	
    nD = [sender intValue];
    if ((nD <1) || (nD > MAXDYNAMPOINTS))
        nD = MAXDYNAMPOINTS;
    [dynamFlush setIntValue:nD];
    [self makeNewDynam:nD];
    [self newOne:sender];
}

/* We create the array of NXRects ahead of time so we do not waste time
   building the array up as we go along */
- (void)makeNewDynam:(int)num
{
    NSRect *curr;
    NSSize pixSize = { 1.0, 1.0 };

    if (pixels)
        free(pixels);
    pixels = (NSRect *)malloc(sizeof(NSRect)*(num+1));
    for (curr = pixels; curr-pixels < num; ++curr)
        curr->size = pixSize;

    [[NSUserDefaults standardUserDefaults] setInteger:num forKey:@"MartinDynamicFlush"];
}

- (void)setHopA:(id)sender
{
    if ([sender doubleValue] > 0) {
        [aFlag setState:NO];
    }
    [self newOne:sender];
}

- (void)setHopB:(id)sender
{
    if ([sender doubleValue] > 0) {
        [bFlag setState:NO];
    }
    [self newOne:sender];
}

- (void)setHopC:(id)sender
{
    if ([sender doubleValue] > 0) {
        [cFlag setState:NO];
    }
    [self newOne:sender];
}

- (void)setSeedInt:(id)sender
{
    if ([sender intValue] > 0) {
        [seedIntFlag setState:NO];
    }
    [self newOne:sender];
}

- (void)setMagnification:(id)sender
{
    if ([sender doubleValue] >= 0) {
        [magFlag setState:NO];
    }
    [self newOne:sender];
}

- (void)setSeedVal:(id)sender
{
    if ([sender doubleValue] >= 0) {
        [seedValFlag setState:NO];
    }
    [self newOne:sender];
}

- (void)setDisplacement:(id)sender
{
    moveX = [displaceX intValue];
    moveY = [displaceY intValue];
    [self newOne:sender];
}

- (void)setColChange:(id)sender
{
    if ([sender intValue] > 0) {
        [colFlag setState:NO];
        [[NSUserDefaults standardUserDefaults] setInteger:[sender intValue]
                                                   forKey:@"MartinColorInt"];
    }
    [self newOne:sender];
}

- (void)setMaxTot:(id)sender
{
    if ([sender intValue] > 0) {
        [maxTotFlag setState:NO];
        [[NSUserDefaults standardUserDefaults] setInteger:[sender intValue]
                                                   forKey:@"MartinMaxTot"];
    }
    [self newOne:sender];
}

- (void)setMaxIn:(id)sender
{
    if ([sender intValue] > 0) {
        [maxInFlag setState:NO];
        [[NSUserDefaults standardUserDefaults] setInteger:[sender intValue]
                                                   forKey:@"MartinMaxIn"];
    }
    [self newOne:sender];
}

- (void)setRandomColor:(id)sender
{
    Randomcolor = [sender state];
    [[NSUserDefaults standardUserDefaults] setBool:[sender state]
                                            forKey:@"MartinRandomCol"];
    [self newOne:sender];
}

- (void)changeColorMode:(id)sender
{
    int	newColMode;

    newColMode = [sender tag];
    if (Color != newColMode) {
        Color = newColMode;
        [self convertColors];
        [[NSUserDefaults standardUserDefaults] setInteger:Color
                                                   forKey:@"MartinColorMode"];
    }
    [self newOne:self];
}

/* Weak way of changing from nonrandom color array to nonrandom gray array */
- (void)convertColors
{
    if (Color){
        Ncolors = DEFAULTNUMCOLORS;
        [colors[0] release];
        colors[0] = [[NSColor blueColor] retain];
        [colors[1] release];
        colors[1] = [[NSColor brownColor] retain];
        [colors[2] release];
        colors[2] = [[NSColor whiteColor] retain];
    }
    else{
        Ncolors = 3;
        [colors[0] release];
        colors[0] = [[NSColor whiteColor] retain];
        [colors[1] release];
        colors[1] = [[NSColor darkGrayColor] retain];
        [colors[2] release];
        colors[2] = [[NSColor lightGrayColor] retain];
    }
}

/* Append the current fractal info (all of it) in the data file */
- (void)remember:(id)sender
{
    if (file && [[NSFileManager defaultManager] isWritableFileAtPath:file]) {
        NSMutableString *dump = [NSMutableString stringWithContentsOfFile:file];

        [dump appendString:@"MartinView\n"];
        [dump appendFormat:@"%d %.15le\n",Pn,Pv];
        [dump appendFormat:@"%d\n",Function];
        [dump appendFormat:@"%.15le %.15le %.15le %.15le\n",A,B,C,Zf];
        [dump appendFormat:@"%d %d %d\n",mxp,mxP,nC];
        [dump appendFormat:@"%d %d\n",moveX,moveY];

        [dump writeToFile:file atomically:YES];
    }
}

- (void)useFile:(id)sender
{
    if ([sender state]) {
        NSString *stream = nil;

        if (!(stream = [NSString stringWithContentsOfFile:file])) {
            NSFileManager *manager = [NSFileManager defaultManager];
            NSBundle *bundle = [NSBundle bundleForClass:[self class]];
            NSString *source = [bundle pathForResource:@".martinView" ofType:nil];

            if ([manager fileExistsAtPath:source])
                [manager copyPath:source toPath:file handler:nil];
        }

        if (!(stream = [NSString stringWithContentsOfFile:file])) {
            [sender setState:NO];
        } else {
            NSScanner *scanner = [NSScanner scannerWithString:stream];
            NSCharacterSet *theSet = [NSCharacterSet whitespaceAndNewlineCharacterSet];
            NSString *name = nil;

            if ([scanner isAtEnd] == NO) {
                [scanner scanUpToCharactersFromSet:theSet intoString:&name];
                if (!([name isEqualToString:@"MartinView"])) {
                    [sender setState:NO];
                }
            }
        }
    }

    [self newOne:self];

    [[NSUserDefaults standardUserDefaults] setBool:[sender state] forKey:@"MartinUseFile"];
}

- (id)initWithFrame:(NSRect)frameRect
{
    return [self initWithFrame:frameRect loadInspectorPanel:YES];
}

- (id)initWithFrame:(NSRect)frameRect loadInspectorPanel:(BOOL)loadPanel
{	
    if (![super initWithFrame:frameRect]) return nil;

    [self allocateGState];		// For faster lock/unlockFocus

    file = [fileName retain];
    srandom(getpid());
    isPaused = 0;
    Function = 0;
    Ncolors = DEFAULTNUMCOLORS;
    colors[0] = [[NSColor blueColor] retain];
    colors[1] = [[NSColor brownColor] retain];
    colors[2] = [[NSColor whiteColor] retain];
    colors[3] = [[NSColor cyanColor] retain];
    colors[4] = [[NSColor colorWithCalibratedRed:250 green:128 blue:114 alpha:1.0] retain];
    colors[5] = [[NSColor grayColor] retain];
    colors[6] = [[NSColor greenColor] retain];
    colors[7] = [[NSColor colorWithCalibratedRed:255 green:87 blue:33 alpha:1.0] retain];
    colors[8] = [[NSColor magentaColor] retain];
    colors[9] = [[NSColor orangeColor] retain];
    colors[10] = [[NSColor purpleColor] retain];
    colors[11] = [[NSColor redColor] retain];
    colors[12] = [[NSColor colorWithCalibratedRed:227 green:207 blue:87 alpha:1.0] retain];
    colors[13] = [[NSColor yellowColor] retain];
    currColor = [colors[0] retain];
    moveX = 0; moveY = 0;
    Ranfseed=4326;
    nD = [[NSUserDefaults standardUserDefaults] integerForKey:@"MartinDynamicFlush"];

    [self makeNewDynam:nD];	
    return self;
}

- (void)inspectorWillBeRemoved
{
    [[NSUserDefaults standardUserDefaults] synchronize];
    [myPrefPanel  orderOut:self];
}

- (NSView *)inspector:(id)sender
{
    int value;

    if (!sharedInspectorPanel) {
        NSUserDefaults *defaults = [NSUserDefaults standardUserDefaults];

        if (![NSBundle loadNibNamed:@"MartinPrefs.nib" owner:self])  {
            NSLog(@"Failed to load MartinPrefs.nib");
            NSBeep();
            [NSException raise:@"Failed in loadNibNamed"
                        format:@"Failed load of MartinPrefs.nib"];
        }

        if (myPrefPanel)
            [myPrefPanel setFrameAutosaveName:@"MartinViewPreferencesPanel"];

        Function = [defaults integerForKey:@"MartinFunction"];
        if (Function == -1)
            [funcAuto setState:YES];
        else {
            [funcAuto setState:NO];
            [funcButton setTitle:fname[Function]];
        }

        if ((value = [defaults integerForKey:@"MartinMaxTot"]) == -1)
            [maxTotFlag setState:YES];
        else {
            [maxTotFlag setState:NO];
            [maxTotalPts setIntValue:value];
        }
        if ((value = [defaults integerForKey:@"MartinMaxIn"]) == -1)
            [maxInFlag setState:YES];
        else {
            [maxInFlag setState:NO];
            [maxInRangePts setIntValue:value];
        }
        if ((value = [defaults integerForKey:@"MartinColorInt"]) == -1)
            [colFlag setState:YES];
        else {
            [colFlag setState:NO];
            [colInterval setDoubleValue:value];
        }
        Randomcolor = [defaults boolForKey:@"MartinRandomCol"];
        [randomFlag setState:Randomcolor];
        Color=[defaults integerForKey:@"MartinColorMode"];
        if (Color) [colorButton selectItemWithTitle:@"Color"];
        else [colorButton selectItemWithTitle:@"Mono"];
        [self convertColors];

        if (Randomcolor) {
            // too many retain release cycles, but needed for NSColor
            [currColor release];
            currColor = (Color) ?
            [NSColor colorWithCalibratedRed:Ranf() green:Ranf() blue:Ranf() alpha:1.0]
            : [NSColor colorWithCalibratedWhite:Ranf() alpha:1.0];
            currColor = [currColor retain];
        } else {
            currColor = [colors[0] retain];
        }

        [useFileFlag setState:[defaults boolForKey:@"MartinUseFile"]];
        if ([useFileFlag state])
            [self useFile:useFileFlag];
        else {
            [self newOne:self];
        }
    }
    
    return sharedInspectorPanel;
}

- (void)setFrameSize:(NSSize)newSize
{
    [super setFrameSize:newSize];
    [self newOne:self];
}

- (void)drawRect:(NSRect)rect
{
    if (!NSIsEmptyRect(rect)) {
        [[NSColor blackColor] set];
        NSRectFill(rect);
    }
}

- (void)dealloc
{
    if (pixels)
        free(pixels);
    [super dealloc];
}

/* Why not a little fluff */
- (void)windowWillMiniaturize:(NSNotification *)notification
{
    [[notification object] setMiniwindowImage:[NSImage imageNamed:@"MartinMini"]];
}

- (BOOL)isBoringScreenSaver { return NO; }
- (BOOL)useBufferedWindow {  return NO; }
- (NSString *)windowTitle { return @"Martin Fractals";}

@end

@implementation StaticMartinView : MartinView

- (void)drawRect:(NSRect)rects
{
    int i;
    
    [super drawRect:rects];

    np = 0; nP = 0; nd = 0; numPixels=0;
    x = y = 0; color = -1;
    Ranfset(time(0));
    W = (int)[self bounds].size.width;
    H = (int)[self bounds].size.height;
    mxX = W-1; mxY = H-1;

    Pn = pow(10., 1+Ranf()*3);
    Pv = pow(10.,Ranf()*3);

    /* provide default hopalong parameters if needed */
    {
        double r = Ranf();
        if      (r < 0.40) Function = Martin1;
        else if (r < 0.70) Function = Ejk1;
        else if (r < 0.90) Function = Ejk2;
        else               Function = Martin2;
    }
    if (Function == Martin1) {
        A = 40 + Ranf()*1500;
        B = 3  + Ranf()*17;
        C = 100 + Ranf()*3000;
    }
    else if (Function == Martin2) {
        A = 3.075927 + Ranf()*0.14;
    }
    else if (Function == Ejk1) {
        A = Ranf()*500;
        B = Ranf()*.40;
        C = 10 + Ranf()*100;
    }
    else if (Function == Ejk2) {
        A = Ranf()*500;
        B = pow(10.,6+Ranf()*24);
        C = pow(10.,  Ranf()*9);
    }

    if (Ranf()<0.5) A = -A;
    if (Ranf()<0.5) B = -B;
    if (Ranf()<0.5) C = -C;

    Zf = (Function == Martin2) ? 4.0 : 1.0;

    mxp = (int) (0.40* (float)(W*H));
    mxP = 4 * mxp;

    /* color processing */
    nC = (mxP/Ncolors)/2;

    nc = nC;
    cx = W/2+moveX; cy = H/2+moveY;

    [self lockFocus];
    for (i=0; i<128; i++) {
        [self oneStep];
    }
    [self unlockFocus];
}

@end
