#import "UniverseView.h"
#import "PSsetAntialias.h"
#import "UniverseInspector.h"

static NSString *numPlanetsName         = @"UniverseView_numPlanets";
static NSString *scaleName              = @"UniverseView_scale";
static NSString *rScaleName             = @"UniverseView_rScale";
static NSString *backgroundColorName    = @"UniverseView_backgroundColor";
static NSString *defaultPlanetColorName = @"UniverseView_defaultPlanetColor";
static NSString *maxStepsName           = @"UniverseView_maxSteps";
static NSString *bounceName             = @"UniverseView_bounce";

@implementation UniverseView

+ (void)initialize {
   NSDictionary *defaultDictionary = [NSDictionary
      dictionaryWithObjectsAndKeys:
      numPlanetsName, @"30",
      scaleName, @"1.0",
      rScaleName, @"2.0",
      backgroundColorName, @"0.0,0.0,0.0",
      defaultPlanetColorName, @"1.0,1.0,1.0",
      maxStepsName, @"10000",
      bounceName, @"0",
      nil, nil];
   [[NSUserDefaults standardUserDefaults] registerDefaults:defaultDictionary];
}


- (void)showDetailedInspector:(id)sender {
   if (detailedInspector == nil) {
      detailedInspector = [[UniverseInspector alloc] initWithUniverseView:self];
   }
   [detailedInspector show];
}

- (void)setNumPlanets:(id)sender
{
   numPlanets = [sender intValue];
   [numPlanetsSlider setIntValue:numPlanets];
   [numPlanetsField setIntValue:numPlanets];

   [universe setNumPlanets:numPlanets];
   [[NSUserDefaults standardUserDefaults] setObject:[NSString stringWithFormat:@"%d", numPlanets] forKey:numPlanetsName];
   haveUnsavedDefaults = YES;
   needsFullRefresh = YES;
   [self display];
}

- (void)drawRect:(NSRect) rects
{
   [self antialiasHack];
   [self clearUniverse];
   [self drawUniverse];
}

- (id)initWithFrame:(NSRect) frameRect
{
   id tmp, tmp2;

   needsFullRefresh = YES;
   haveUnsavedDefaults = NO;
   nSteps = 0;
   [super initWithFrame:frameRect];

   tmp = [[NSUserDefaults standardUserDefaults] objectForKey:bounceName];
   bounce = ([tmp intValue] != 0);

   tmp = [[NSUserDefaults standardUserDefaults] objectForKey:maxStepsName];
   maxSteps = [tmp intValue];

   tmp = [[NSUserDefaults standardUserDefaults] objectForKey:numPlanetsName];
   numPlanets = [tmp intValue];

   tmp = [[NSUserDefaults standardUserDefaults] objectForKey:scaleName];
   scale = [tmp floatValue];

   tmp = [[NSUserDefaults standardUserDefaults] objectForKey:rScaleName];
   rScale = [tmp floatValue];

   tmp = [[NSUserDefaults standardUserDefaults] objectForKey:backgroundColorName];
   tmp2 = [tmp componentsSeparatedByString:@","];
   backgroundColor = [[NSColor colorWithCalibratedRed:[[tmp2 objectAtIndex:0] floatValue]
                                                green:[[tmp2 objectAtIndex:1] floatValue]
                                                 blue:[[tmp2 objectAtIndex:2] floatValue]
                                                alpha:1.0] retain];

   tmp = [[NSUserDefaults standardUserDefaults] objectForKey:defaultPlanetColorName];
   tmp2 = [tmp componentsSeparatedByString:@","];
   [Planet setStartColor:[NSColor colorWithCalibratedRed:[[tmp2 objectAtIndex:0] floatValue]
                                                   green:[[tmp2 objectAtIndex:1] floatValue]
                                                    blue:[[tmp2 objectAtIndex:2] floatValue]
                                                   alpha:1.0]];

   inspector = [self inspector:self];

   [numPlanetsField setIntValue:numPlanets];
   [numPlanetsSlider setIntValue:numPlanets];

   universe = [self newUniverseWithBounds:NSMakeRect(0, 0,
                                                     NSWidth(frameRect)/scale, NSHeight(frameRect)/scale)
                               numPlanets:numPlanets];
   
   return self;
}

- (void)dealloc {
   [universe release];
   if (detailedInspector != nil)
      [detailedInspector release];
}

- (void)setFrameSize:(NSSize)newSize {
   [super setFrameSize:newSize];
   [universe setSize:NSMakeSize(newSize.width/scale, newSize.height/scale)];
   universeBounds = [universe bounds];
   needsFullRefresh = YES;
}

- (NSTimeInterval)animationDelayTime
{
    return (0.04);
}

- (NSView *)inspector:sender
{
   if (!inspector) {
      [NSBundle loadNibNamed:@"inspector.nib" owner:self];
   }

   return inspector;
}

- (void)clearUniverse {
   [backgroundColor set];
   NSRectFill([self bounds]);
   needsFullRefresh = NO;
}

- (void)drawUniverse {
   int i;
   NSArray *planets = [universe planets];

   for (i = 0; i < [planets count]; i++) {
      [self drawPlanet:(Planet *)[planets objectAtIndex:i]];
   }
}

- (void)oneStep {
   if (haveUnsavedDefaults) {
      [[NSUserDefaults standardUserDefaults] synchronize];
      haveUnsavedDefaults = NO;
   }

   if (maxSteps > 0 && nSteps >= maxSteps) {
      NSRect bounds = [self bounds];
      universe = [self newUniverseWithBounds:NSMakeRect(0.0, 0.0,
                                                        NSWidth(bounds)/scale, NSHeight(bounds)/scale)
                                  numPlanets:[numPlanetsField intValue]];
      needsFullRefresh = YES;
      nSteps = 0;
   } else {
      [universe oneStep];
      nSteps++;
   }
   
   [self setNeedsDisplay:YES];
}

- (void)antialiasHack {
   static BOOL initialized = NO;
   static NSColor *col;
   static NSBezierPath *path;

   if (!initialized) {
      col = [[NSColor colorWithCalibratedRed:0.2 green:0.4 blue:0.6 alpha:1.0] retain];
      path = [[NSBezierPath bezierPathWithRect:NSMakeRect(1.0,1.0,1.1,1.1)] retain];
   }
   
   PSsetAntialias();
   [col set];
   [path fill];
   [backgroundColor set];
   [path fill];
}

- (BOOL)useBufferedWindow {
   return YES;
}

- (void)drawPlanet:(Planet *)p {
   [[p color] set];
   [self drawPlanetNoColor:p];
}

- (void)drawPlanetNoColor:(Planet *)p {
   NSBezierPath *path = [[NSBezierPath alloc] init];
   NSPoint position = [p position];
   double r = sqrt([p mass]) * rScale;
   NSRect bounds = [self bounds];
   NSRect planetShape = NSMakeRect(NSMinX(bounds) + scale * ((position.x - NSMinX(universeBounds) - r / 2.0)),
                                   NSMinY(bounds) + scale * ((position.y - NSMinY(universeBounds) - r / 2.0)),
                                   scale * r,
                                   scale * r);

   [path appendBezierPathWithOvalInRect:planetShape];
   [path fill];
   [path release];
}

- (Universe *)newUniverseWithBounds:(NSRect)newBounds numPlanets:(int)n {
   Universe *u;
   universeBounds = newBounds;
   u = [[Universe alloc] initWithBounds:newBounds
                             numPlanets:n];
   [u setBounce:bounce];
   return u;
}

- (double) scale {
   return scale;
}

- (void)setScale:(double)s {
   NSRect bounds = [self bounds];
   scale = s;
   [universe setSize:NSMakeSize(NSWidth(bounds)/scale, NSHeight(bounds)/scale)];
   universeBounds = [universe bounds];
   [[NSUserDefaults standardUserDefaults] setObject:[NSString stringWithFormat:@"%f", s] forKey:scaleName];
   haveUnsavedDefaults = YES;
   needsFullRefresh = YES;
   [self display];
}


- (double)rScale {
   return rScale;
}

- (void)setRScale:(double)rs {
   rScale = rs;
   [[NSUserDefaults standardUserDefaults] setObject:[NSString stringWithFormat:@"%f", rs] forKey:rScaleName];
   haveUnsavedDefaults = YES;
   needsFullRefresh = YES;
   [self display];
}

- (void)addPlanets:(int)n inColor:(NSColor *)color {
   int nOld = [numPlanetsField intValue];
   int nNew = nOld + n;
   double mass = [Planet startMass];
   for (; n > 0; n--) {
      [universe addPlanet:[universe newPlanetWithMass:mass andColor:color]];
   }
   [numPlanetsField setIntValue:nNew];
   [numPlanetsSlider setIntValue:nNew];
   [[NSUserDefaults standardUserDefaults] setObject:[NSString stringWithFormat:@"%d", nNew] 		forKey:numPlanetsName];
   haveUnsavedDefaults = YES;
   [self display];
}

- (NSColor *)backgroundColor {
   return backgroundColor;
}

- (void)setBackgroundColor:(NSColor *)bgCol {
   NSColor *c2 = [bgCol colorUsingColorSpaceName:NSCalibratedRGBColorSpace];
   [backgroundColor autorelease];
   backgroundColor = [bgCol retain];

   [[NSUserDefaults standardUserDefaults] setObject:
      [NSString stringWithFormat:@"%f,%f,%f", [c2 redComponent], [c2 greenComponent], [c2 blueComponent]]
                                             forKey:backgroundColorName];
   haveUnsavedDefaults = YES;
   needsFullRefresh = YES;
   [self display];
}

- (NSColor *)defaultPlanetColor {
   return [Planet startColor];
}

- (void)setDefaultPlanetColor:(NSColor *)col {
   NSArray *planets = [universe planets];
   NSColor *c2 = [col colorUsingColorSpaceName:NSCalibratedRGBColorSpace];
   int i;
   [Planet setStartColor:col];

   for (i = 0; i < [planets count]; i++) {
      [[planets objectAtIndex:i] setColor:col];
   }

   [[NSUserDefaults standardUserDefaults] setObject:
      [NSString stringWithFormat:@"%f,%f,%f", [c2 redComponent], [c2 greenComponent], [c2 blueComponent]]
                                             forKey:defaultPlanetColorName];
   haveUnsavedDefaults = YES;
   needsFullRefresh = YES;
   [self display];
}

- (int)maxSteps {
   return maxSteps;
}

- (void)setMaxSteps:(int)m {
   maxSteps = m;
   [[NSUserDefaults standardUserDefaults] setObject:[NSString stringWithFormat:@"%d", maxSteps] forKey:maxStepsName];
   haveUnsavedDefaults = YES;
}

- (BOOL)bounce {
   return bounce;
}

- (void)setBounce:(BOOL)b {
   bounce = b;
   [universe setBounce:b];
   [[NSUserDefaults standardUserDefaults] setObject:[NSString stringWithFormat:@"%d", bounce] forKey:bounceName];
   haveUnsavedDefaults = YES;
}

@end
