/* Sample code for WWDC 1998 session 122: Yellow Box Foundation in Depth

You may freely use this code.  Apple provides this code as is without
warranty to its correctness or fitness for use for any purpose.
*/

/* Implementation for ThreadPerformer class which uses DO */

#import "ThreadPerformerDO.h"

@protocol ThreadPerformerRemoteProtocol
- (id)_performSelector:(SEL)selector
	onTarget:(unsigned int)target
	withObject:(unsigned int)arg1;
@end

// Nearly same example as ThreadPerformer, but written to use D.O.

@implementation ThreadPerformerDO

+ (ThreadPerformerDO *)defaultPerformer {
    ThreadPerformerDO *performer;
    NSMutableDictionary *mdict;

    // You should only access a thread's dictionary
    // from the execution context of that thread itself
    mdict = [[NSThread currentThread] threadDictionary];
    performer = [mdict objectForKey:@"ThreadPerformerDO"];
    if (nil == performer) {
	performer = [[self allocWithZone:NULL] init];
        [mdict setObject:performer forKey:@"ThreadPerformerDO"];
	[performer release];
	// (thread dictionary keeps a retain on the performer)
    }
    return performer;
}

- (id)init {
    // Protecting this counter with a lock is
    // left as an exercise for the reader
    static int counter = 0;

    self = [super init];
    if (!self) return nil;
    _conn = [[NSConnection alloc] init];
    [_conn setRootObject:self];
    do {
	// This string needs to be unique across all processes
	// on the current machine, but isn't since there can be
	// multiple processes with the same name; we loop until
	// we're successfully registered
	_name = [NSString stringWithFormat:@"ThreadPerformerDO-%s-%x",
	    [[NSProcessInfo processInfo] processName], counter++];
    } while (![_conn registerName:_name]);
    _name = [_name retain];
    return self;
}

- (void)dealloc {
    [_conn invalidate];
    [_conn release];
    [_name release];
    [super dealloc];
}

- (id)performSelector:(SEL)selector
	onTarget:(id)target
	withObject:(id)arg1 {
    id proxy;
    id result;

    // Look up the myself in the other thread, in effect
    proxy = [NSConnection rootProxyForConnectionWithRegisteredName:_name host:@""];

    // Make sure D.O. knows about the argument/return
    // types for the remote message we're about to send
    [proxy setProtocolForProxy:@protocol(ThreadPerformerRemoteProtocol)];

    // Send the message to myself in the
    // other thread of execution 

    // The target is passed over D.O. as an integer
    // for a subtle reason: if the target is typed 'id',
    // D.O. will either create a copy of the object to
    // send to the other thread, or a proxy, depending
    // on what the class of target does to transport
    // itself.  Copying will often be undesirable for
    // obvious reasons, but the proxy case is more
    // subtle: the other thread will get a D.O. proxy
    // to target and performing the method on that
    // will cause D.O. messaging back to this current
    // thread and the code will be executed by *this*
    // thread, not the other one which is what you want.
    // So we circumvent some of D.O.'s sophistication.
    // Sending a proxy might be undesirable for other
    // reasons in some situations as well.

    result = [proxy _performSelector:selector onTarget:(unsigned int)target withObject:(unsigned int)arg1];
    return result;
}

- (id)_performSelector:(SEL)selector
	onTarget:(unsigned int)target
	withObject:(unsigned int)arg1 {
    // Perform the method
    return [(id)target performSelector:selector withObject:(id)arg1];
}

@end

