/*
 * Name: vumount.c
 * Description: Unmount the given vmounted volume
 * Author: Christian Starkjohann <cs@hal.kph.tuwien.ac.at>
 * Date: 1997-04-30
 * Copyright: GNU-GPL
 * Tabsize: 4
 */

#include <libc.h>
#include <errno.h>
#include <sys/param.h>
#include <sys/ucred.h>
#include <sys/mount.h>
#include "my_defines.h"

/* ------------------------------------------------------------------------- */

extern int	errno;
static char	*my_name;

/* ------------------------------------------------------------------------- */

static int	umount_dir(char *path)
{
	struct statfs 	*buf;
	long 		fsno = 0,j = 0;
	int 		flags = MNT_WAIT;
	int		pid, rval = 0, found_any = 0;

	char	*toname, *frname;

	fsno = getfsstat(NULL, 0, flags);
	buf = (struct statfs *)malloc(fsno * sizeof(struct statfs));
	if(getfsstat(buf, (long)(fsno * sizeof(struct statfs)), flags) == -1){
		fprintf(stderr, "%s: problems encountered while doing getfsstat: %s\n", my_name, strerror(errno));
		free(buf);
		return -1;
	}else{
		for (j=fsno-1; j>=0; j--) {
			toname = buf[j].f_mntonname;
			frname = buf[j].f_mntfromname;
			if (!strcmp(toname, path)){
				if(sscanf(frname, "vmount-%d", &pid) == 1){
					if(kill(pid, 30) == -1){
						fprintf(stderr, "%s: cannot ummount ->%s<-: %s\n", my_name, toname, strerror(errno));
						rval = -1;
					}
				}else{
					fprintf(stderr, "%s: type ->%s<- not mounted by vmount\n", my_name, frname);
					rval = -1;
				}

				found_any = 1;
				break;
			}
		}

		if(!found_any){
			fprintf(stderr, "%s: directory ->%s<- not found in mtab\n", my_name, path);
		}
	}
	free(buf);
	return rval;
}

/* ------------------------------------------------------------------------- */

static void	print_all(void)
{
struct statfs	*buf;
long		fsno;
int		flags = MNT_WAIT;
int				pid;

	fsno = getfsstat(NULL, 0, flags);
	buf = (struct statfs *)malloc(fsno * sizeof(struct statfs));

	if(getfsstat(buf, (long)(fsno * sizeof(struct statfs)), flags) == -1){
		fprintf(stderr, "%s: problems while doing getfsstat: %s\n", my_name, strerror(errno));
		return;
	}else{
		int i;
		for(i = 0; i < fsno; i++){
			if(sscanf(buf[i].f_mntfromname, "vmount-%d", &pid) == 1)
				printf("%s\n", buf[i].f_mntonname);
		}

	}
}

/* ------------------------------------------------------------------------- */

int	main(int argc, char **argv)
{
int		i, error = 0, rval;

	my_name = argv[0];
	if(argc == 1){
		printf("vmount mounted directories:\n");
		print_all();
	}else{
		for(i=1;i<argc;i++){
			error |= umount_dir(argv[i]);
		}
	}
	rval = error ? 1 : 0;
	return rval;
}

/* ------------------------------------------------------------------------- */
