#import "MosaicAction.h"
#import "PixelCache.h"


@implementation MosaicAction

// Read the parameters from the ActionData object.
- initWithEngineData: (ActionData *)data
{
	self = [super initWithEngineData: data];
	if(self)
	{
		size	= [[CSMeasurementUnit alloc] initWithStringValue:
					[data objectForKey: @"Size"]];
	}

	return self;
}

// Cleanup.
- (void)dealloc
{
	[size release];

	[super dealloc];
}


// Called before any calls to the getRGBAPixel: and getCMYKPixel: methods are
// done. This is guaranteed to run in the main thread, so avoid time consuming
// initialization here. (Not used in this implementation)
- (void)preLaunchMainThread
{
	[super preLaunchMainThread];
}

// Called before any calls to the getRGBAPixel: and getCMYKPixel: methods are
// done. May already run in a separate thread, so no AppKit calls there!
- (void)preLaunchExecThread
{
	PixelCache	*cache;
	int			 n;

	[super preLaunchExecThread];

	side		= CSRound([size valueInUnit: CSUNIT_PIXEL  dpi: [layer dpi]]);
	if(side < 1)	side = 1;

	n		= side + [source width];
	buffer	= NSZoneMalloc([self zone], n*side * sizeof(CMYKAPixel));

	cache	= [[PixelCache alloc] initWithSource: [source autorelease]];
	source	= cache;
	[cache hintCacheSize: side];

	oldU		= -1;
	oldV		= -1;
	oldW		= -1;
}

// Cleanup in action thread.
- (void)postEndExecThread
{
	[super postEndExecThread];

	NSZoneFree([self zone], buffer);
}

// Cleanup in main thread. (Not used in this implementation)
- (void)postEndMainThread
{
	[super postEndMainThread];
}


// This method is called for every row when an RGB image is processed.
- (void)getRGBAPixel: (RGBAPixel *)p  at: (int)x: (int)y  count: (int)n
{
	RGBAPixel	*q,*t;
	int		 cr,cg,cb,ca;
	int		 w, u,v, i,j,k;

	q	= buffer;
	u	= x  -  (x % side);
	v	= y  -  (y % side);
	w	= x+n - u;

	if(w % side != 0)
		w	= w + side - (w % side);

	// If we can reuse the results from the previous call, we skip the whole
	// calculation.
	if(oldU != u  ||  oldV != v  ||  oldW != w)
	{
		// Fetch the pixels from the source.
		for(i=0 ; i<side ; i++)
			[source getRGBAPixel: q + i*w  at: u: v+i  count: w];

		// Average square regions in the buffer. This is not an
		// very fast routine, because every pixel's R,G, and B value
		// has to be multiplied with its Alpha value. As almost all
		// Alpha values in an image are either 0 or CHANNEL_MAX,
		// there's room for optimization there.
		for(k=0 ; k<w ; k+=side)
		{
			cr = cg = cb = ca = 0;
	
			for(i=0 ; i<side ; i++)
			{
				t	= q + i*w + k;
				for(j=0 ; j<side ; j++)
				{
					cr	= cr + t->a * t->r;		cg	= cg + t->a * t->g;
					cb	= cb + t->a * t->b;		ca	= ca + t->a;
	
					t++;
				}
			}
	
			if(ca > 0)
			{
				cr	= cr / ca;	cg	= cg / ca;
				cb	= cb / ca;	ca	= ca / (side*side);
			}
	
			t	= q + k;
			for(j=0 ; j<side ; j++)
			{
				t->r	= cr;		t->g	= cg;
				t->b	= cb;		t->a	= ca;
	
				t++;
			}
		}

		oldU	= u;
		oldV	= v;
		oldW	= w;
	}

	// Copy the results from the buffer to the output array.
	memmove(p, q + (x-u), n*sizeof(RGBAPixel));
}

// This method is called for every row when an CMYK image is processed.
- (void)getCMYKAPixel: (CMYKAPixel *)p  at: (int)x: (int)y  count: (int)n
{
	CMYKAPixel	*q,*t;
	int			 cc,cm,cy,ck,ca;
	int			 w, u,v, i,j,k;

	q	= buffer;
	u	= x  -  (x % side);
	v	= y  -  (y % side);
	w	= x+n - u;

	if(w % side != 0)
		w	= w + side - (w % side);

	// If we can reuse the results from the previous call, we skip the whole
	// calculation.
	if(oldU != u  ||  oldV != v  ||  oldW != w)
	{
		// Fetch the pixels from the source.
		for(i=0 ; i<side ; i++)
			[source getCMYKAPixel: q + i*w  at: u: v+i  count: w];

		// Average square regions in the buffer. This is not an
		// very fast routine, because every pixel's C,M,Y, and K value
		// has to be multiplied with its Alpha value. As almost all
		// Alpha values in an image are either 0 or CHANNEL_MAX,
		// there's room for optimization there.
		for(k=0 ; k<w ; k+=side)
		{
			cc = cm = cy = ck = ca = 0;
	
			for(i=0 ; i<side ; i++)
			{
				t	= q + i*w + k;
				for(j=0 ; j<side ; j++)
				{
					cc	= cc + t->a * t->c;		cm	= cm + t->a * t->m;
					cy	= cy + t->a * t->y;		ck	= ck + t->a * t->k;
					ca	= ca + t->a;
	
					t++;
				}
			}
	
			if(ca > 0)
			{
				cc	= cc / ca;	cm	= cm / ca;
				cy	= cy / ca;	ck	= ck / ca;
				ca	= ca / (side*side);
			}
	
			t	= q + k;
			for(j=0 ; j<side ; j++)
			{
				t->c	= cc;		t->m	= cm;
				t->y	= cy;		t->k	= ck;
				t->a	= ca;
	
				t++;
			}
		}

		oldU	= u;
		oldV	= v;
		oldW	= w;
	}

	// Copy the results from the buffer to the output array.
	memmove(p, q + (x-u), n*sizeof(CMYKAPixel));
}

@end
