//************************************************************************
//
//	SSLSocket.m
//
//	by Juergen Moellenhoff <jm@oic.de>		
//
//
//	This code is supplied "as is" the author makes no warranty as to its 
//	suitability for any purpose.  This code is free and may be distributed 
//	in accordance with the terms of the:
//		
//			GNU GENERAL PUBLIC LICENSE
//			Version 2, June 1991
//			copyright (C) 1989, 1991 Free Software Foundation, Inc.
// 			675 Mass Ave, Cambridge, MA 02139, USA
//
//************************************************************************

#import "SSLSocket.h"

#import "CTXHandler.h"
#import "HTTPSPreferences.h"
#import "HTTPSFunctions.h"

#import	<OmniFoundation/OFUserDefaults.h>
#import	<OmniBase/rcsid.h>

#import	<Foundation/Foundation.h>

RCS_ID("$Header: /LocalDeveloper/CVS/HTTPS/SSLSocket.m,v 1.13 1999/06/22 15:14:07 jurgen Exp $")

@implementation SSLSocket

- (void)_prepareConnect
{
	CTXHandler	*ctxHandler;
	
	// Create the CTX-Structure and do some initialization
	// for the OpenSSL-lib. The CTX-Structure is shared by
	// all SSLSocket-Instances.

	if ((ctxHandler = [CTXHandler sharedCTXHandler]) == nil)
		[NSException raise:ONInternetSocketConnectFailedExceptionName format:@"Unable to create CTXHandler: [%@]", sslErrorString()];

	// Free the old SSL-Handle	
	if (_sslHandle != NULL)
	{
    	SSL_free(_sslHandle);
		_sslHandle = NULL;
	}

	// Create the SSL-Handle which is different 
	// for each Instance of SSLSocket
	if ((_sslHandle = SSL_new([ctxHandler ctx])) == NULL)
		[NSException raise:ONInternetSocketConnectFailedExceptionName 
			format:@"Unable to create SSL-Handle: [%@]", sslErrorString()];

	SSL_set_connect_state(_sslHandle);

	if (SSL_set_fd(_sslHandle, socketFD) == 0)
		[NSException raise:ONInternetSocketConnectFailedExceptionName 
			format:@"Unable to set socketFD to SSL-Socket: [%@]", sslErrorString()];

	// SSL_set_read_ahead(_sslHandle, YES);
}

- (BOOL)isSecureConnection
{
	return (_sslHandle != NULL);
}

- (SSL *)sslHandle
{
	return _sslHandle;
}

- (unsigned int)bits
{
	if (![self isSecureConnection])
		return 0;

	return (unsigned int)SSL_CIPHER_get_bits(SSL_get_current_cipher(_sslHandle), NULL);
}

- (NSString *)protocolString
{
	if (![self isSecureConnection])
		return @"none";

	return [NSString stringWithFormat:@"%s", SSL_get_version(_sslHandle)];
}

- (NSString *)cipherString
{
	if (![self isSecureConnection])
		return @"none";

	return [NSString stringWithFormat:@"%s", SSL_CIPHER_get_name(SSL_get_current_cipher(_sslHandle))];
}

- (void)sslConnect
{
	[self _prepareConnect];

	if (SSL_connect(_sslHandle) <= 0) 
		[NSException raise:ONInternetSocketConnectFailedExceptionName 
			format:@"Unable to connect, SSL_connect() failed: [%@]", sslErrorString()];
}

- (void)dealloc
{
	if (_sslHandle != NULL)
	{
    	SSL_free(_sslHandle);
		_sslHandle = NULL;
	}

    [super dealloc];
}

- (unsigned int)readBytes:(unsigned int)byteCount intoBuffer:(void *)aBuffer
{
	if (_sslHandle != NULL)
	{
		int	bytesRead;

		bytesRead = SSL_read(_sslHandle, aBuffer, byteCount);
		if (bytesRead == -1)
			[NSException raise:ONInternetSocketReadFailedExceptionName 
				format:@"Unable to read from socket: [%@]", sslErrorString()];
		
		return (unsigned int)bytesRead;
	}
	
	return [super readBytes:byteCount intoBuffer:aBuffer];
}

- (unsigned int)writeBytes:(unsigned int)byteCount fromBuffer:(const void *)aBuffer
{
	if (_sslHandle != NULL)
	{
		int	bytesWritten;
		
		bytesWritten = SSL_write(_sslHandle, (char *)aBuffer, byteCount);
		if (bytesWritten == -1)
			[NSException raise:ONInternetSocketWriteFailedExceptionName 
				format:@"Unable to write to socket: [%@]", sslErrorString()];

		return (unsigned int)bytesWritten;
	}
	
	return [super writeBytes:byteCount fromBuffer:aBuffer];
}

- (void)abortSocket
{
	if (_sslHandle != NULL)
		SSL_shutdown(_sslHandle);

	[super abortSocket];
}

- (NSMutableDictionary *)debugDictionary
{
	NSMutableDictionary *debugDictionary;

	debugDictionary = [super debugDictionary];

    [debugDictionary setObject:[self isSecureConnection] ? @"YES" : @"NO" forKey:@"secureConnected"];
	if ([self isSecureConnection])
	{
		[debugDictionary setObject:[NSNumber numberWithUnsignedInt:[self bits]]  forKey:@"sslBits"];
		[debugDictionary setObject:[self protocolString]  forKey:@"sslProtocolString"];
		[debugDictionary setObject:[self cipherString]  forKey:@"sslCipherString"];
	}

	return debugDictionary;
}

@end
