/*
	pxoread.c
		pxo2gif		by Takeshi Ogihara (Mar. 2000)
 */

#include  <stdio.h>
#include  <libc.h>
#include  "pxo2gif.h"

#define  PXOF		0xff
#define  PAOF		0xfe
#define  UnknownHD	0
/* PX original Format
    PX			: Header
    Width Height Colors	: Colors := Number of colors in palette - 1
    Count		: if Trans is there Count=1, otherwise Count=0.
    Trans		: Transparent + 256
    [Palette]		: Binary
    [Bitmap]		: Binary
*/

static char pxocomm[MAX_COMMENT];

static void pxo_skip(FILE *fp)
{
	int c;

	while ((c = getc(fp)) != EOF) {
		if (c == '\n') {
			c = getc(fp);
			while (c == '#') {
				while ((c = getc(fp)) != '\n')
					if (c == EOF)
						return;
				c = getc(fp);
			}
		}
		if (c > ' ') {
			ungetc(c, fp);
			return;
		}
	}
}

static int pxo_head(FILE *fp)
{
	int c, i, kind = UnknownHD;

	if ((c = getc(fp)) != 'P')
		return UnknownHD;
	c = getc(fp);
	if (c >= '4' && c <= '6') /* Binary PPM only */
		kind = c - '0';
	else if (c == 'X')
		kind = PXOF;
	else if (c == 'A')
		kind = PAOF;
	pxocomm[0] = 0;
	while ((c = getc(fp) & 0xff) <= ' ') ;
	if (c == '#') {
		for (i = 0; (c = getc(fp)) != '\n'; i++)
			if (i < MAX_COMMENT-1) pxocomm[i] = c;
		pxocomm[i] = 0;
		ungetc('\n', fp);
		pxo_skip(fp);
	}else
		ungetc(c, fp);
	return kind;
}

static int pxo_getint(FILE *fp)
{
	int c;
	int v = 0;

	c = getc(fp);
	while (c < '0' || c > '9') {
		if (c == EOF) return -1;
		c = getc(fp);
	}
	while (c >= '0' && c <= '9') {
		v = (v * 10) + c - '0';
		c = getc(fp);
	}
	if (c != EOF && c < ' ')
		ungetc(c, fp);
	return v;
}

static const char *begin_comm(const char *comm)
	/* Get begining of comment (... : comment) */
{
	if (comm == NULL)
		return NULL;
	while (*comm && *comm != ':')
		comm++;
	if (*comm == ':')
		while (*++comm == ' ') ;
	if (*comm == 0)
		return NULL;
	return comm;
}

static paltype *read_palette(FILE *fp, int cols, int transp)
{
	int	i, j;
	paltype *palette;
	unsigned char *p;

	palette = (paltype *)malloc(sizeof(paltype) * 256);
	for (i = 0; i < cols; i++) {
		p = (unsigned char *)&palette[i];
		for (j = 0; j < 3; j++)
			p[j] = (unsigned char)getc(fp);
	}
	while (cols <= transp && cols < 255) {
		p = (unsigned char *)&palette[cols++];
		for (j = 0; j < 3; j++)
			p[j] = 255;
	}
	return palette;
}

commonInfo *pxoread(FILE *fin)
{
	int cnt;
	int pxokind, pxomax;
	int width, height, transp;
	long	total;
	const char *kp = NULL;
	commonInfo *info;
	unsigned char *p;

	pxokind = pxo_head(fin);
	width = pxo_getint(fin);
	pxo_skip(fin);
	height = pxo_getint(fin);
	if (pxokind != PXOF || width <= 0 || height <= 0) {
		fprintf(stderr, "ERROR: Unknown format\n");
		exit(1);
	}

	info = (commonInfo *)malloc(sizeof(commonInfo));
	info->memo = begin_comm(pxocomm);
	info->width = width;
	info->height = height;

	pxo_skip(fin);
	info->palsteps = pxomax = pxo_getint(fin) + 1;
	kp = "PXOF";
	pxo_skip(fin);
	cnt = pxo_getint(fin);
	transp = -1;
	if (cnt-- > 0) {
		pxo_skip(fin);
		transp = pxo_getint(fin);
		transp = (transp > 256) ? (transp - 256) : -1;
		while (cnt-- > 0) {	/* skip unknown parameters */
			pxo_skip(fin);
			(void) pxo_getint(fin);
		}
	}
	info->transp = transp;
	(void)getc(fin);	/* feed last CR */
	info->palette = read_palette(fin, pxomax, transp);
	total = width * height;
	info->pixels = p = (unsigned char *)malloc(total);
	while (total-- > 0)
		*p++ = (unsigned char)getc(fin);

	if (verbose)
	    fprintf(stderr, "%s, %dx%d, max:%d\n", kp, width, height, pxomax);
	return info;
}

void freePxoInfo(commonInfo *info)
{
	if (info->pixels)
		free((void *)info->pixels);
	if (info->palette)
		free((void *)info->palette);
	free((void *)info);
}
