/*
	gifsave.c

	gifsave.c is based on "ppmtogif" of Jef Poskanzer.
	It is modified for ToyViewer.app by T. Ogihara. (1995)

	pxo2gif		by T.Ogihara (Mar. 2000)
*/

/* ppmtogif.c - read a portable pixmap and produce a GIF file
**
** Based on GIFENCOD by David Rowley <mgardi@watdscu.waterloo.edu>.A
** Lempel-Zim compression based on "compress".
**
** Copyright (C) 1989 by Jef Poskanzer.
**
** Permission to use, copy, modify, and distribute this software and its
** documentation for any purpose and without fee is hereby granted, provided
** that the above copyright notice appear in all copies and that both that
** copyright notice and this permission notice appear in supporting
** documentation.  This software is provided "as is" without express or
** implied warranty.
**
** The Graphics Interchange Format(c) is the Copyright property of
** CompuServe Incorporated.  GIF(sm) is a Service Mark property of
** CompuServe Incorporated.
*/

#include <stdio.h>
#include "pxo2gif.h"
// #include "gif.h"


static void put_short(int k, FILE *fp)
{
	putc(k & 0xff, fp);
	putc((k >> 8) & 0xff, fp);
}


#define MAXCOLORS 256

/*
 * a code_int must be able to hold 2**BITS values of type int, and also -1
 */
typedef int	code_int;

#define INTERLACE		0x40
#define LOCALCOLORMAP		0x80
#define TRANSPARENCY		0x01

#ifdef SIGNED_COMPARE_SLOW
typedef unsigned long int count_int;
#else /*SIGNED_COMPARE_SLOW*/
typedef long int	count_int;
#endif /*SIGNED_COMPARE_SLOW*/

static int GIFNextPixel(void);
static void compress( int init_bits, FILE* outfile );
static void init_output(void);
static void output( code_int code );
static void cl_block( void );
static void cl_hash( count_int hsize );
static void char_out( int c );
static void flush_char( void );

/*****************************************************************************
 *
 * GIFENCODE.C	- GIF Image compression interface
 *
 * GIFEncode( FName, commonInfo, interlace )
 *
 *****************************************************************************/

static int Width, Height;
static int BitsPerPixel, isInterlaced;
static int yPtr, yPass, xPtr;
static unsigned char *imageMap;

/*
 * Return the next pixel from the image
 */

static void resetPixels(commonInfo *info)
{
	yPtr = xPtr = 0, yPass = 0;
	Width = info->width;
	Height = info->height;
	imageMap = info->pixels;
}

static int GIFNextPixel(void)
{
	int eol, pix;

	if (yPtr < 0)
		return EOF;
	pix = imageMap[yPtr * Width + xPtr];
	eol = (++xPtr >= Width);
	if (eol) {
		if (isInterlaced) {
			yPtr += yPass ? (0x10 >> yPass) : 0x08;
			if (yPtr >= Height)
				yPtr = (++yPass > 3) ? -1 :(8 >> yPass);
		}else if (++yPtr >= Height)
			return -1;
		xPtr = 0;
	}
	return pix;
}

void GIFEncode( FILE *fp, commonInfo *cinf, int interlace )
{
	int Resolution, ColorMapSize, InitCodeSize;
	int i, j, power, n, B, alphaindex;
	const char *comment;
	int gif89;

	isInterlaced = interlace;
	resetPixels(cinf);
	alphaindex = cinf->transp;
	comment = cinf->memo;
	n = (cinf->transp > cinf->palsteps) ? cinf->transp : cinf->palsteps;
	for (i = 1, power = 0x02; i < 8; i++, power <<= 1)
		if (power >= n) break;
	BitsPerPixel = i;
	ColorMapSize = 1 << BitsPerPixel;
	Width = cinf->width;
	Height = cinf->height;
	Resolution = BitsPerPixel;

	/* The initial code size */
	InitCodeSize = ( BitsPerPixel <= 1 ) ? 2 : BitsPerPixel;

	/* Write the Magic header */
	gif89 = (alphaindex >= 0 || comment);
	fprintf(fp, "%s", gif89 ? "GIF89a":"GIF87a");

	/* Write out the screen width and height */
	put_short( cinf->width, fp );
	put_short( cinf->height, fp );

	/* Indicate that there is a global colour map */
	B = 0x80;			/* Yes, there is a color map */
	B |= (Resolution - 1) << 5;	/* OR in the resolution */
	B |= (BitsPerPixel - 1);	/* OR in the Bits per Pixel */
	fputc( B, fp );

	/* Write out the Background colour */
	fputc( 0, fp );
	fputc( 0, fp );	/* Byte of 0's (future expansion) */

	/* Write out the Global Colour Map */
	for(i = 0; i < ColorMapSize; ++i ) {
		unsigned char *p = cinf->palette[i];
		for (j = 0; j < 3; j++)
			fputc( p[j], fp );
	}
	for( ; i < ColorMapSize; ++i ) {
		for (j = 0; j < 3; j++)
			fputc( 255, fp );	/* stab */
	}

	if (gif89) {
		if (alphaindex >= 0) {
			fputc('!', fp);		/* Extension */
			fputc(0xf9, fp);	/* Graphic Control Extension */
			fputc(4, fp);		/* Block Size */
			fputc(TRANSPARENCY, fp);
			fputc(0, fp);
			fputc(0, fp);		/* Delay Time */
			fputc(alphaindex, fp);	/* Transparent Color Index */
			fputc(0, fp);		/* Block Terminator */
		}
		if (comment && (i = strlen(comment)) > 0) {
			fputc('!', fp);		/* Extension */
			fputc(0xfe, fp);	/* Comment Extension */
			fputc(i, fp);		/* Block Size */
			fprintf(fp, "%s", comment);
			fputc(0, fp);		/* Block Terminator */
		}
	}

	/* Write an Image separator */
	fputc( ',', fp );

	/* Write the Image header */
	put_short( 0, fp );	/* LeftOfs */
	put_short( 0, fp );	/* TopOfs */
	put_short( Width, fp );
	put_short( Height, fp );
	/* No Local Color Table & Interlace Mode */
	fputc( interlace ? 0x40 : 0x00, fp );

	/* Write out the initial code size */
	fputc( InitCodeSize, fp );

	/*
	 * Go and actually compress the data
	 */
	compress( InitCodeSize+1, fp );

	/* Write out a Zero-length packet (to end the series) */
	fputc( 0, fp );
	/* Write the GIF file terminator */
	fputc( ';', fp );
}


/***************************************************************************
 *
 *  GIFCOMPR.C	- GIF Image compression routines
 *
 *  Lempel-Ziv compression based on 'compress'.  GIF modifications by
 *  David Rowley (mgardi@watdcsu.waterloo.edu)
 *
 ***************************************************************************/

/*
 * General DEFINEs
 */

#define BITS	12
#define HSIZE	5003	/* 80% occupancy */

/*
 *
 * GIF Image compression - modified 'compress'
 *
 * Based on: compress.c - File compression ala IEEE Computer, June 1984.
 *
 * By Authors:  Spencer W. Thomas	(decvax!harpo!utah-cs!utah-gr!thomas)
 *		Jim McKie		(decvax!mcvax!jim)
 *		Steve Davies		(decvax!vax135!petsd!peora!srd)
 *		Ken Turkowski		(decvax!decwrl!turtlevax!ken)
 *		James A. Woods		(decvax!ihnp4!ames!jaw)
 *		Joe Orost		(decvax!vax135!petsd!joe)
 */
#include <ctype.h>

static int n_bits;		/* number of bits/code */
static int maxbits = BITS;	/* user settable max # bits/code */
static code_int maxcode;	/* maximum code, given n_bits */
static code_int maxmaxcode = (code_int)1 << BITS;
				/* should NEVER generate this code */
#ifdef COMPATIBLE		/* But wrong! */
# define MAXCODE(n_bits)	((code_int) 1 << (n_bits) - 1)
#else /*COMPATIBLE*/
# define MAXCODE(n_bits)	(((code_int) 1 << (n_bits)) - 1)
#endif /*COMPATIBLE*/

static count_int htab [HSIZE];
static unsigned short codetab [HSIZE];
#define HashTabOf(i)       htab[i]
#define CodeTabOf(i)    codetab[i]

static code_int hsize = HSIZE;	/* for dynamic table sizing */

/*
 * To save much memory, we overlay the table used by compress() with those
 * used by decompress().  The tab_prefix table is the same size and type
 * as the codetab.  The tab_suffix table needs 2**BITS characters.  We
 * get this from the beginning of htab.  The output stack uses the rest
 * of htab, and contains characters.  There is plenty of room for any
 * possible stack (stack used to be 8000 characters).
 */

#define tab_prefixof(i) CodeTabOf(i)
#define tab_suffixof(i)	((unsigned char *)(htab))[i]
#define de_stack	((unsigned char *)&tab_suffixof((code_int)1<<BITS))

static code_int free_ent = 0;	/* first unused entry */

/*
 * block compression parameters -- after all codes are used up,
 * and compression rate changes, start over.
 */
static int clear_flg = 0;

static int offset;
static long int in_count = 1;	/* length of input */
static long int out_count = 0;	/* # of codes output (for debugging) */

/*
 * compress stdin to stdout
 *
 * Algorithm:  use open addressing double hashing (no chaining) on the
 * prefix code / next character combination.  We do a variant of Knuth's
 * algorithm D (vol. 3, sec. 6.4) along with G. Knott's relatively-prime
 * secondary probe.  Here, the modular division first probe is gives way
 * to a faster exclusive-or manipulation.  Also do block compression with
 * an adaptive reset, whereby the code table is cleared when the compression
 * ratio decreases, but after the table fills.  The variable-length output
 * codes are re-sized at this point, and a special CLEAR code is generated
 * for the decompressor.  Late addition:  construct the table according to
 * file size for noticeable speed improvement on small files.  Please direct
 * questions about this implementation to ames!jaw.
 */

static int g_init_bits;
static FILE* g_outfile;

static int ClearCode;
static int EOFCode;

static void
compress( int init_bits, FILE *outfile )
{
    long fcode;
    code_int i /* = 0 */;
    int c;
    code_int ent;
    code_int disp;
    code_int hsize_reg;
    int hshift;

    /*
     * Set up the globals:	g_init_bits - initial number of bits
     *				g_outfile   - pointer to output file
     */
    g_init_bits = init_bits;
    g_outfile = outfile;

    /*
     * Set up the necessary values
     */
    offset = 0;
    out_count = 0;
    clear_flg = 0;
    in_count = 1;
    maxcode = MAXCODE(n_bits = g_init_bits);

    ClearCode = (1 << (init_bits - 1));
    EOFCode = ClearCode + 1;
    free_ent = ClearCode + 2;

    init_output();	/* Set up the 'byte output' routine */

    ent = GIFNextPixel();

    hshift = 0;
    for ( fcode = (long) hsize;  fcode < 65536L; fcode *= 2L )
	++hshift;
    hshift = 8 - hshift;		/* set hash code range bound */

    hsize_reg = hsize;
    cl_hash( (count_int) hsize_reg);	/* clear hash table */

    output( (code_int)ClearCode );

    while ( (c = GIFNextPixel()) != EOF ) {
	++in_count;

	fcode = (long) (((long) c << maxbits) + ent);
	i = (((code_int)c << hshift) ^ ent);	/* xor hashing */

	if ( HashTabOf (i) == fcode ) {
	    ent = CodeTabOf (i);
	    continue;
	} else if ( (long)HashTabOf (i) < 0 )	/* empty slot */
	    goto nomatch;
	disp = hsize_reg - i;	/* secondary hash (after G. Knott) */
	if ( i == 0 )
	    disp = 1;
probe:
	if ( (i -= disp) < 0 )
	    i += hsize_reg;

	if ( HashTabOf (i) == fcode ) {
	    ent = CodeTabOf (i);
	    continue;
	}
	if ( (long)HashTabOf (i) > 0 )
	    goto probe;
nomatch:
	output ( (code_int) ent );
	++out_count;
	ent = c;
	if ( free_ent < maxmaxcode ) {
	    CodeTabOf (i) = free_ent++; /* code -> hashtable */
	    HashTabOf (i) = fcode;
	} else
		cl_block();
    }
    /*
     * Put out the final code.
     */
    output( (code_int)ent );
    ++out_count;
    output( (code_int) EOFCode );
}

/*****************************************************************
 * Output the given code.
 * Inputs:
 *      code:	A n_bits-bit integer.  If == -1, then EOF.
 *		This assumes that n_bits =< (long)wordsize - 1.
 * Outputs:
 *      Outputs code to the file.
 * Assumptions:
 *      Chars are 8 bits long.
 * Algorithm:
 *      Maintain a BITS character long buffer (so that 8 codes will
 * fit in it exactly).  Use the VAX insv instruction to insert each
 * code in turn.  When the buffer fills up empty it and start over.
 */

static unsigned long cur_accum = 0;
static int cur_bits = 0;

static void output( code_int code )
{
	static unsigned long masks[] = {
		0x0000, 0x0001, 0x0003, 0x0007, 0x000F,
		0x001F, 0x003F, 0x007F, 0x00FF,
		0x01FF, 0x03FF, 0x07FF, 0x0FFF,
		0x1FFF, 0x3FFF, 0x7FFF, 0xFFFF };

	if( cur_bits > 0 ) {
		cur_accum &= masks[ cur_bits ];
		cur_accum |= ((long)code << cur_bits);
	}else
		cur_accum = code;
	cur_bits += n_bits;
	while( cur_bits >= 8 ) {
		char_out( (unsigned int)(cur_accum & 0xff) );
		cur_accum >>= 8;
		cur_bits -= 8;
	}
	/*
	 * If the next entry is going to be too big for the code size,
	 * then increase it, if possible.
	 */
	if ( free_ent > maxcode || clear_flg ) {
		if( clear_flg ) {
			maxcode = MAXCODE (n_bits = g_init_bits);
			clear_flg = 0;
		} else {
			++n_bits;
			maxcode = ( n_bits == maxbits )
				? maxmaxcode : MAXCODE(n_bits);
		}
	}
	if( code == EOFCode ) {
		/* At EOF, write the rest of the buffer. */
		while( cur_bits > 0 ) {
			char_out( (unsigned int)(cur_accum & 0xff) );
			cur_accum >>= 8;
			cur_bits -= 8;
		}
		flush_char();
		fflush( g_outfile );
		/* if( ferror( g_outfile ) ) writeerr(); */
	}
}

/*
 * Clear out the hash table
 */
static void cl_block(void)	/* table clear for block compress */
{
	cl_hash ( (count_int) hsize );
	free_ent = ClearCode + 2;
	clear_flg = 1;
	output( (code_int)ClearCode );
}

static void cl_hash(count_int hsize)	/* reset code table */
{

	count_int *htab_p = htab+hsize;
	long i;
	long m1 = -1;

	i = hsize - 16;
	do {		/* might use Sys V memset(3) here */
		*(htab_p-16) = m1;
		*(htab_p-15) = m1;
		*(htab_p-14) = m1;
		*(htab_p-13) = m1;
		*(htab_p-12) = m1;
		*(htab_p-11) = m1;
		*(htab_p-10) = m1;
		*(htab_p-9) = m1;
		*(htab_p-8) = m1;
		*(htab_p-7) = m1;
		*(htab_p-6) = m1;
		*(htab_p-5) = m1;
		*(htab_p-4) = m1;
		*(htab_p-3) = m1;
		*(htab_p-2) = m1;
		*(htab_p-1) = m1;
		htab_p -= 16;
	} while ((i -= 16) >= 0);

	for ( i += 16; i > 0; --i )
		*--htab_p = m1;
}

/****************************************************************************
 *
 * GIF Specific routines
 *
 ****************************************************************************/

static int a_count;	/* Number of characters so far in this 'packet' */
static char accum[256];	/* Define the storage for the packet accumulator */

static void init_output(void)
{
	cur_accum = 0;
	cur_bits = 0;
	a_count = 0;
}

/*
 * Add a character to the end of the current packet, and if it is 254
 * characters, flush the packet to disk.
 */
static void char_out( int c )
{
	accum[ a_count++ ] = c;
	if( a_count >= 254 )
		flush_char();
}

/*
 * Flush the packet to disk, and reset the accumulator
 */
static void flush_char( void )
{
	if( a_count > 0 ) {
		fputc( a_count, g_outfile );
		fwrite( accum, 1, a_count, g_outfile );
		a_count = 0;
	}
}
