#import "ImageShrinker.h"
#import <Foundation/NSString.h>
#import <AppKit/NSImage.h>
#import <AppKit/NSImageRep.h>
#import <AppKit/NSColor.h>
#import <AppKit/psopsOpenStep.h>
#import "../ColorSpaceCtrl.h"
#import "DCTscaler.h"
#import "../common.h"
#import "../imagefunc.h"
#import "../getpixel.h"
#import "funcs.h"

static int shrink(int asz, PIXmat apix);
static commonInfo *makeDCTResizedMap(commonInfo *cinf, int bsz, int asz,
		unsigned char *map[], unsigned char *newmap[]);
static void get_tabindex(float ratio, int *a, int *b);
static void trim_image(commonInfo *info,
		int width, int height, unsigned char **map);

/* Enhancement */
void enhance(const commonInfo *cinf, float factor,
		const unsigned char **map, unsigned char **planes);


@implementation ImageShrinker

- initWidth:(int)wid height:(int)hei
{
	[super init];
	to_width = wid;
	to_height = hei;
	return self;
}

- (NSImage *)shrinkedImage:(NSImage *)origimage size:(NSSize)sz
{
	NSSize	newsz;
	float	w, h, ratio;
	int	a, b, spp, pn;
	commonInfo info, *newinf = NULL;
	NSBitmapImageRep *imageRep;
	NSImage	*tmpimage, *newimage;
	id rep;
        NSString *cs;
	unsigned char	*map[MAXPLANE], *wmap[MAXPLANE], *newmap[MAXPLANE];

	if (sz.width <= 0.0 || sz.height <= 0.0) {
		rep = [origimage bestRepresentationForDevice:nil];
		// Because NSImage does lazy loading
		sz = getImageSize(origimage);
		if (sz.width <= to_width && sz.height <= to_height)
			return origimage;
	}
	w = to_width / sz.width;
	h = to_height / sz.height;
	ratio = (h < w) ? h : w;

	tmpimage = getImageMap(origimage, &info, map);
	if (tmpimage == nil)
		goto ErrEXIT;
	if (tmpimage != origimage) {
		[origimage release];
		origimage = tmpimage;
	}

	get_tabindex(ratio, &a, &b);
	if (a == b)	/* ratio is near to 1.00 */
		return origimage;
	/* Note: ratio >= (image_size / thumbnail_size) */
	if ((newinf = makeDCTResizedMap(&info, b, a, map, wmap)) == NULL)
		goto ErrEXIT;
	/* Edge Enhancement */
	pn = (newinf->numcolors == 1) ? 1 : 3;
	if (allocImage(newmap, newinf->width, newinf->height, 8, pn) == 0) {
		enhance(newinf, 1.2, wmap, newmap);
		free((void *)wmap[0]);
	}else {
		int i;
		for (i = 0; i < MAXPLANE; i++)
			newmap[i] = wmap[i];
	}

	trim_image(newinf, to_width, to_height, newmap);
	spp = newinf->numcolors;
	if (newinf->alpha) spp++;
        cs = [ColorSpaceCtrl colorSpaceName: newinf->cspace];
        imageRep = [[NSBitmapImageRep alloc]
		initWithBitmapDataPlanes:newmap
		pixelsWide:newinf->width pixelsHigh:newinf->height
		bitsPerSample:newinf->bits samplesPerPixel:spp
		hasAlpha:newinf->alpha isPlanar:YES colorSpaceName:cs
		bytesPerRow:newinf->xbytes bitsPerPixel:newinf->bits];
	if (imageRep == nil) {
		free((void *)newmap[0]);
		goto ErrEXIT;
	}
	newsz.width = newinf->width;
	newsz.height = newinf->height;
	if ((newimage = [[NSImage alloc] initWithSize:newsz]) == nil) {
		[imageRep release];
		goto ErrEXIT;
	}
	[newimage setDataRetained:YES];
	[newimage addRepresentation:imageRep];
	[imageRep release];	/* Because imageRep is retained by newimage */
	[origimage release];
	free((void *)newinf);
	return newimage;
ErrEXIT:
	newsz.width = (int)(sz.width * ratio);
	newsz.height = (int)(sz.height * ratio);
	[origimage setScalesWhenResized:YES];
	[origimage setSize:newsz];
	if (newinf)
		free((void *)newinf);
	return origimage;
}


static void trim_image(commonInfo *info,
		int width, int height, unsigned char **map)
{
	int	skipx, skipy, neww, newh;
	int	n, cnum;

	skipx = info->width - width;
	skipy = info->height - height;
	if (skipx <= 0 && skipy <= 0)
		return;
	if (skipx > 0) {
		neww = width + (skipx = (skipx + 1) / 2);
		/* even if skipx == 1, skipx should > 0 */
	}else	neww = info->width, skipx = 0;
	if (skipy > 0) {
		newh = height + (skipy /= 2);
		info->height = height;
	}else	newh = info->height, skipy = 0;
	cnum = info->numcolors;
	if (info->alpha) cnum++;
	for (n = 0; n < cnum; n++) {
		unsigned char *a, *mp;
		int	x, y;
		a = map[n];
		for (y = skipy; y < newh; y++) {
			mp = map[n] + info->xbytes * y + skipx;
			for (x = skipx; x < neww; x++)
				*a++ = *mp++;
		}
	}
	if (skipx > 0) {
		info->width = width;
		info->xbytes = byte_length(info->bits, width);
	}
}

static int shrink(int asz, PIXmat apix)
{
	int	i, j, val;
	unsigned char *p;

	val = 0;
	for (i = 0; i < asz; i++) {
		p = apix[i];
		for (j = 0; j < asz; j++)
			val += p[j];
	}
	val /= (asz * asz);
	if (val > 255) return 255;
	return val;
}

static commonInfo *makeDCTResizedMap(commonInfo *cinf, int bsz, int asz,
		unsigned char *map[], unsigned char *newmap[])
{
	commonInfo *newinf = NULL;
	unsigned char *planes[MAXPLANE], *bufa[MAXPLANE], *bufb[MAXPLANE];
	PIXmat	apix, bpix;
	int	wida, widb;
	int	pn, i, x, y, nx, ny, ptr;
	DCTscaler *dct = nil;

	planes[0] = NULL;
	bufa[0] = bufb[0] = NULL;
	if (asz % bsz == 0) {
		if (bsz > 1) asz /= bsz, bsz = 1;
		/* dct = nil */
	}else
		dct = [[DCTscaler alloc] init:bsz :asz];

	if ((newinf = (commonInfo *)malloc(sizeof(commonInfo))) == NULL)
		goto ErrEXIT;
	*newinf = *cinf;
	newinf->width = cinf->width * bsz / asz;
	newinf->height = cinf->height * bsz / asz;
	x = (cinf->width + asz - 1) / asz;
	wida = x * asz;
	widb = x * bsz;
	if (newinf->width > widb) newinf->width = widb;
	newinf->bits = 8;
	newinf->xbytes = byte_length(newinf->bits, newinf->width);
	newinf->isplanar = YES;
	newinf->alpha = NO;	/* remove Alpha */
	if (cinf->cspace == CS_Black)
		newinf->cspace = CS_White;
	if (newinf->width > MAXWidth)
		goto ErrEXIT;
	pn = (cinf->numcolors == 1) ? 1 : 3;

	if (allocImage(bufa, wida, asz, 8, pn)
	 || allocImage(bufb, widb, bsz, 8, pn))
		goto ErrEXIT;
	if (allocImage(planes, newinf->width, newinf->height, 8, pn))
		goto ErrEXIT;
	if (initGetPixel(cinf) != 0)
		goto ErrEXIT;

	resetPixel(map, 0);
	// [theWaitMsg messageDisplay:@"Resizing..."];
	// [theWaitMsg setProgress:(cinf->height - 1)];
	for (y = 0, ny = 0, ptr = 0; y < cinf->height; ) {
		int	z, idx;
		int	elm[MAXPLANE];
		// [theWaitMsg progress: y];
		for (z = 0, idx = 0; z < asz; z++) {
			if (++y > cinf->height) {
				int	svx = idx - wida;
				for (x = 0; x < wida; x++, idx++, svx++) {
					for (i = 0; i < pn; i++)
						bufa[i][idx] = bufa[i][svx];
				}
				continue;
			}
			for (x = 0; x < cinf->width; x++, idx++) {
				getPixelA(elm);
				for (i = 0; i < pn; i++)
					bufa[i][idx] = elm[i];
			}
			for ( ; x < wida; x++, idx++) {
				for (i = 0; i < pn; i++)
					bufa[i][idx] = elm[i];
			}
		}
		for (i = 0; i < pn; i++) {
		    for (x = 0, nx = 0; x < cinf->width; x += asz, nx += bsz) {
			for (z = 0; z < asz; z++)
				apix[z] = &bufa[i][wida * z + x];
			for (z = 0; z < bsz; z++)
				bpix[z] = &bufb[i][widb * z + nx];
			if (dct)
				[dct DCTrescale:bpix from:apix];
			else
				bufb[i][nx] = shrink(asz, apix);
		    }
		}
		for (z = 0; z < bsz; z++) {
			if (++ny > newinf->height)
				break;
			for (i = 0; i < pn; i++)
				memcpy(&planes[i][ptr], &bufb[i][widb * z],
					newinf->width);
			ptr += newinf->width;
		}
	}
	// [theWaitMsg resetProgress];
	// [theWaitMsg messageDisplay: nil];

	for (i = 0; i < pn; i++)
		newmap[i] = planes[i];

	if (dct) [dct release];
	free((void *)bufa[0]);
	free((void *)bufb[0]);
	return newinf;

ErrEXIT:
	if (newinf) free((void *)newinf);
	if (dct) [dct release];
	if (planes[0]) free((void *)planes[0]);
	if (bufa[0]) free((void *)bufa[0]);
	if (bufb[0]) free((void *)bufb[0]);
	return NULL;
}

typedef struct {
	float	r;
	int	b, a;
} ratio_tab;

static const ratio_tab ratioTab[] = {
	{  6.250, 1, 16 },
	{  6.667, 1, 15 },
	{  7.143, 1, 14 },
	{  8.333, 1, 12 },
	{  9.091, 1, 11 },
	{ 10.000, 1, 10 },
	{ 11.111, 1, 9 },
	{ 12.500, 1, 8 }, /* 2 & 16 */
	{ 13.333, 2, 15 },
	{ 14.286, 2, 14 },
	{ 15.385, 2, 13 },
	{ 16.667, 2, 12 },
	{ 18.182, 2, 11 },
	{ 18.750, 3, 16 },
	{ 20.000, 2, 10 },
	{ 22.222, 2, 9 },
	{ 25.000, 2, 8 }, /* 3 & 12 */
	{ 27.273, 3, 11 },
	{ 30.000, 3, 10 },
	{ 33.333, 3, 9 }, /* 4 & 12 */
	{ 37.500, 3, 8 }, /* 6 & 16 */
	{ 40.000, 4, 10 },
	{ 43.750, 7, 16 },
	{ 46.154, 6, 13 },
	{ 50.000, 4, 8 }, /* 5&10, 6&12, 7&14, 8&16 */
	{ 53.333, 8, 15 },
	{ 57.143, 8, 14 },
	{ 60.000, 6, 10 },
	{ 62.500, 5, 8 }, /* 10 & 16 */
	{ 66.667, 6, 9 }, /* 8 & 12 */
	{ 70.000, 7, 10 },
	{ 71.429, 10, 14 },
	{ 75.000, 6, 8 }, /* 12 & 16, 9 & 12 */
	{ 77.778, 7, 9 },
	{ 80.000, 8, 10 },
	{ 83.333, 10, 12 },
	{ 86.667, 13, 15 },
	{ 90.000, 9, 10 },
	{ 93.750, 15, 16 },
	{ 100.000, 8, 8 },
	{ 1000.0,  10, 1 } /* Never */
};

static void get_tabindex(float ratio, int *a, int *b)
{
	int	x;

	ratio *= 100.0;
	for (x = 0; ratioTab[x].r < ratio; x++)
		;
	*a = ratioTab[x].a;
	*b = ratioTab[x].b;
}

@end
