#include "Regexp.h"
#include <string.h>
#include <libc.h>
#include <ctype.h>

#define  RangeByte	16
#define  SUCCESS	0
#define  FAILURE	1
#define  SKIP		2

enum {
	EOEXP = 0,
	OneCharSym,	/* ? */
	NumberSym,	/* # */
	AlphaSym,	/* @ */
	RangeSym,	/* [ ] */
	OptSym,		/* ^0 */
	PlusSym,	/* + */
	AsterSym,	/* * */
};


static const char *get_range(char *code, const char *sptr)
{
	int c1, c2, i, j;
	int notflag = 0;
	const char *s;
	char chs[128];

	for (i = 0; i < 128; i++)
		chs[i] = 0;
	for (s = sptr; ; s++) {
		if ((c1 = *s) == 0)
			return NULL;	/* Error */
		if (c1 == ']') {
			if (s != sptr)
				break;	/* BREAK the loop */
		}else if (c1 == '\\') c1 = *++s;
		else if (c1 == '^' && s == sptr) {
			notflag = 1;
			continue;
		}

		if ((c1 &= 0x7f) == 0)
			return NULL;	/* Error */
		if (*(s+1) != '-') {
			chs[c1] = 1;
			continue;
		}
		s += 2;
		if (*s == '\\') c2 = *++s;
		else c2 = *s;
		if ((c2 &= 0x7f) == 0)
			return NULL;	/* Error */
		if (c1 > c2)
			return NULL;	/* Error */
		for (i = c1; i <= c2; i++)
			chs[i] = 1;
	}
	for (i = 0; i < RangeByte; i++) {
		int n = 0;
		for (j = 0; j < 8; j++)
			if (chs[(i << 3) | j])
				n |= (1 << j);
		code[i] = n;
	}
	if (notflag) code[0] |= 1;	/* [^ ] */
	return s;
}

static rgx_type rgx_compile(const char *s)
{
	int i, b;
	char *xp, *p;
	const char *q;

	if (!s || s[0] == 0)
		return NULL;
	for (i = 0, b = 0; s[i]; i++)
		if (s[i] == '[') b++;
	xp = malloc(b * (RangeByte+1) + i * 2 + 2);
	for (i = 0, q = s; *q; q++) {
		if (*q == '?')
			xp[i++] = OneCharSym;
		else if (*q == '#')
			xp[i++] = NumberSym;
		else if (*q == '@')
			xp[i++] = AlphaSym;
		else if (*q == '^') {	/* ^0  ^+  or  ^* */
			int prv, cc;
			if (i == 0)
				return NULL;	/* Error */
			prv = xp[i-1];
			if (prv == OptSym || prv == PlusSym || prv == AsterSym)
				return NULL;	/* Error */
			cc = *++q;
			if (cc == '0') xp[i] = OptSym;
			else if (cc == '+') xp[i] = PlusSym;
			else if (cc == '*') xp[i] = AsterSym;
			else
				return NULL;	/* Error */
			i++;
		}else if (*q == '+') {
			xp[i++] = OneCharSym;
			xp[i++] = PlusSym;
		}else if (*q == '*') {
			xp[i++] = OneCharSym;
			xp[i++] = AsterSym;
		}else if (*q == '\\') {
			if (*++q == 0)
				return NULL;  /* Error */
			xp[i++] = *q;
		}else if (*q == '[') {
			xp[i++] = RangeSym;
			if ((q = get_range(&xp[i], ++q)) == NULL)
				return NULL;	/* Error */
			i += RangeByte;
			xp[i++] = RangeSym;
		}else
			xp[i++] = *q;
	}
	xp[i] = EOEXP;
	p = malloc(i + 1);
	for (b = 0; b <= i; b++)
		p[b] = xp[b];
	free(xp);
	return (rgx_type)p;
}

static int compare_char(rgx_type p, int ch, int fold, int repeat)
{
	int i, b;

	if (*p == EOEXP)
		return (ch == 0) ? SUCCESS : FAILURE;
	if (*p == RangeSym) {
		int nxt = *(p+RangeByte+2);
		int match, notmatch;
		if (!repeat && (nxt == OptSym || nxt == AsterSym))
			return SKIP;
		if (ch == 0) return FAILURE;	/* fail */
		if (p[1] & 1) /* [^ ] */
			match = FAILURE, notmatch = SUCCESS;
		else
			match = SUCCESS, notmatch = FAILURE;
		b = ch >> 3;
		i = ch & 0x07;
		if (p[b+1] & (1 << i))
			return match;
		if (!(fold && isalpha(ch)))
			return notmatch;
		ch = isupper(ch) ? tolower(ch) : toupper(ch);
		b = ch >> 3;
		i = ch & 0x07;
		return (p[b+1] & (1 << i)) ? match : notmatch;
	}
	if (!repeat && (*(p+1) == OptSym || *(p+1) == AsterSym))
		return SKIP;
	if (ch == 0) return FAILURE;	/* fail */
	switch (*p) {
	case OneCharSym: /* ? */
		break;
	case NumberSym: /* # */
		return (ch < '0' || ch > '9') ? FAILURE : SUCCESS;
	case AlphaSym: /* @ */
		return (isalpha(ch)) ? SUCCESS : FAILURE;
		return FAILURE; /* fail */
	default:
		if (fold)
			return (tolower(*p) == tolower(ch))? SUCCESS : FAILURE;
		else
			return (*p == ch) ? SUCCESS : FAILURE;
	}
	return SUCCESS;
}

static int rgx_compare(rgx_type rgx, const char *str, int fold)
{
	int i, t, sym;
	const char *p;
	rgx_type ex;

	p = str;
	for (i = 0; (sym = rgx[i]) != EOEXP; ) {
	    if (sym == OptSym || sym == PlusSym || sym == AsterSym) {
		ex = (rgx[i-1] == RangeSym) ? &rgx[i-RangeSym-2] : &rgx[i-1];
		t = 0;
		if (sym == OptSym) {
			if (*p && compare_char(ex, *p, fold, 1) == SUCCESS)
				t++;
		}else {
			const char *q = p;
			while (*q && compare_char(ex, *q, fold, 1) == SUCCESS)
				q++, t++;
		}
		for ( ; t > 0; --t) {
			if (rgx_compare(&rgx[i+1], p+t, fold) == 0)
				return SUCCESS;
		}
		++i;
	    }else {
		int rc = compare_char(&rgx[i], *p, fold, 0);
		if (rc == FAILURE)
			return FAILURE; /* fail */
		if (sym == RangeSym) i += RangeByte + 2;
		else i++;
		if (*p && rc != SKIP) p++;
	    }
	}
	return *p;
}

@implementation Regexp

- (id)initExpression:(NSString *)exp
{
	[super init];
	rgx = rgx_compile([exp cString]);
	if (rgx == NULL) {
		[self dealloc];
		return nil;
	}
	return self;
}

- (BOOL)match:(NSString *)str ignoreCase:(BOOL)fold
{
	return rgx_compare(rgx, [str cString], fold) == 0;
}

- (void)dealloc
{
	if (rgx) free((void *)rgx);
	[super dealloc];
}

@end


#ifdef DIRTEST
#include <sys/types.h>
#include <sys/dir.h>  /* POSIX applications #include <dirent.h> */

void main(void)
{
	char *pttn;
	char buf[120];
	DIR *dirp;
	struct direct *dp;

	(void)gets(buf);
	pttn = rgx_compile(buf);

	dirp = opendir(".");
	for (dp = readdir(dirp); dp != NULL; dp = readdir(dirp)) {
		if (rgx_compare(pttn, dp->d_name, 1) == 0)
			printf("%s\n", dp->d_name);
	}
	closedir(dirp);
}

#endif

#ifdef ALONE
void main(void)
{
	char *pttn;
	char buf[120];

	(void)gets(buf);
	pttn = rgx_compile(buf);
	while (gets(buf)) {
		if (rgx_compare(pttn, buf, 0) == 0)
			printf("OK\n");
		else
			printf("NO\n");
	}
}
#endif
