//
//  _MiscMergeIfCommand.m
//            Written by Don Yacktman and Carl Lindberg
//        Copyright 1998 by Don Yacktman and Carl Lindberg.
//                     All rights reserved.
//      This notice may not be removed from this source code.
//
//	This header is included in the MiscKit by permission from the author
//	and its use is governed by the MiscKit license, found in the file
//	"License.rtf" in the MiscKit distribution.  Please refer to that file
//	for a list of all applicable permissions and restrictions.
//	

#import "_MiscMergeIfCommand.h"
#import <Foundation/NSUtilities.h>
#import <Foundation/NSValue.h>
#import <Foundation/NSScanner.h>
#import "MiscMergeCommandBlock.h"

@implementation _MiscMergeIfCommand

- init
{
	[super init];
	trueBlock = [[MiscMergeCommandBlock alloc] initWithOwner:self];
	return self;
}

- (void)dealloc
{
	[trueBlock release];
	[elseBlock release];
	[field1 release];
	[field2 release];
	[super dealloc];
}

- (BOOL)parseFromScanner:(NSScanner *)aScanner template:(MiscMergeTemplate *)template
{
	[self eatKeyWord:@"if" fromScanner:aScanner isOptional:NO];

	field1 = [[self getArgumentStringFromScanner:aScanner toEnd:NO] retain];
	operator = [self getConditionalFromScanner:aScanner];
    /*
     * if none, then we just expect value1 to be a single field and the
     * condition will evaluate true if it is non-empty. (Or, it may be a
     * keyword that always evals true or false...)
     */
	if (operator != MiscMergeOperatorNone) {
		field2 = [[self getArgumentStringFromScanner:aScanner toEnd:YES] retain];
	}
	
	[template pushCommandBlock:trueBlock];
	return YES;
}

- (void)handleElseInTemplate:(MiscMergeTemplate *)template
{
	if (elseBlock == nil) elseBlock = [[MiscMergeCommandBlock alloc] initWithOwner:self];
	[template popCommandBlock:trueBlock];
	[template pushCommandBlock:elseBlock];
}

- (void)handleEndifInTemplate:(MiscMergeTemplate *)template
{
	[template popCommandBlock];
}

- (void)executeForMerge:(MiscMergeEngine *)aMerger
{
	if ([self evaluateConditionInMerger:aMerger])
		[aMerger executeCommandBlock:trueBlock];
	else
		if (elseBlock) [aMerger executeCommandBlock:elseBlock];
}

static BOOL isNumber(id anObject)
{
	if ([anObject isKindOfClass:[NSNumber class]]) return YES;
	if ([anObject isKindOfClass:[NSString class]])
	{
		NSScanner *scanner = [NSScanner scannerWithString:anObject];
		float  floatValue;
		return [scanner scanFloat:&floatValue];
	}
	
	return NO;
}

static NSComparisonResult compareFloats(float num1, float num2)
{
	if (num1 < num2) return NSOrderedAscending;
	if (num1 > num2) return NSOrderedDescending;
	return NSOrderedSame;
}

/*"
 * Evaluates the receiver's condition in the context of the merge in
 * progress in %{anEngine}.  Returns YES if the condition evaluated true
 * and NO if not.
"*/
- (BOOL)evaluateConditionInMerger:(MiscMergeEngine *)anEngine
{
	NSString *upperField1 = [field1 uppercaseString];
	id firstOperand, secondOperand;
	NSComparisonResult comparison;

	// special evaluators for first argument
	if ([upperField1 isEqualToString:@"--NONE--"]) return NO;
	if ([upperField1 isEqualToString:@"--NO--"])   return NO;
	if ([upperField1 isEqualToString:@"--ALL--"]) return YES;
	if ([upperField1 isEqualToString:@"--YES--"]) return YES;

	// turn the operands into their field values; literals will
	// come through unchanged...
	firstOperand = [anEngine valueForField:field1];
	
	if (operator == MiscMergeOperatorNone)
	{
		if (isNumber(firstOperand)) return ([firstOperand floatValue])? YES : NO;
		return ([field1 length] > 0 && firstOperand);
	}

	secondOperand = [anEngine valueForField:field2];

	if (isNumber(firstOperand) && isNumber(secondOperand))
	{
		comparison = compareFloats([firstOperand floatValue], [secondOperand floatValue]);
	}
	else
	{
		comparison = [(NSString*)firstOperand compare:secondOperand];
	}

    /*
     * now that we have comparison results, turn them into a YES/NO
     * depending upon the chosen operator.
     */
	switch (operator)
	{
		case MiscMergeOperatorEqual     	 	: return (comparison == NSOrderedSame);
		case MiscMergeOperatorNotEqual 		 	: return (comparison != NSOrderedSame);
		case MiscMergeOperatorLessThanOrEqual	: return (comparison != NSOrderedDescending);
		case MiscMergeOperatorGreaterThanOrEqual: return (comparison != NSOrderedAscending);
		case MiscMergeOperatorLessThan 			: return (comparison == NSOrderedAscending);
		case MiscMergeOperatorGreaterThan 		: return (comparison == NSOrderedDescending);
		case MiscMergeOperatorNone              : return NO; // handled above
	}
	return NO;
}

@end

