//
//  MiscKeyValue.m
//    Written by Carl Lindberg Copyright 1998 by Carl Lindberg.
//                     All rights reserved.
//      This notice may not be removed from this source code.
//
//	This header is included in the MiscKit by permission from the author
//	and its use is governed by the MiscKit license, found in the file
//	"License.rtf" in the MiscKit distribution.  Please refer to that file
//	for a list of all applicable permissions and restrictions.
//	

/*
 * EOF 2.x clone implementations for -valueForKey: and -takeValue:forKey:
 * type methods.
 * 
 * char * are valid for -valueForKey:, but not for -takeValue:forKey:.
 * Not handling Class args either -- same as id?
 */
 
 /* If you wanted to emulate EOF 1.x:
 	 rename -unableToSetNilForKey to -unableToSetNullForKey.
	 remove +accessInstanceVariablesDirectly; it always checks ivars.
	 remove array stuff
	 remove unbound key stuff -- return nil (in valueForKey) or do nothing (in takeValue).
	 -valuesForKeys: and -takeValuesForDictionary: are the only public methods
	 (other than -unableToSetNullForKey:) -- the single key versions and the
	 keypath stuff aren't there.
  */

#import "MiscKeyValue.h"
#import <objc/objc-class.h>
#import <Foundation/NSString.h>
#import <Foundation/NSValue.h>
#import <Foundation/NSObjCRuntime.h>
#import <Foundation/NSException.h>
#import <Foundation/NSDictionary.h>
#import <Foundation/NSUtilities.h>
#import <stdio.h> // for sprintf()
#import <ctype.h> // for islower/toupper
#import "MiscNull.h"

/* Low-level functions that do the raw work */
static id   get_value(id self, NSString *key);
static void set_value(id self, NSString *key, id objectValue);

/*
 * Null object representation, for the -valuesForKeys and
 * -takeValuesFromDictionary methods. I don't want any static references to
 * EONull (since this stuff isn't needed in the first place if you *can*
 * link against EOAccess), so I do it this way.
 */
static id nullObject = nil;

id MiscKeyValueNullObject()
{
	if (nullObject == nil) MiscKeyValueSetNullObject([MiscNull null]);
	return nullObject;
}

void MiscKeyValueSetNullObject(id anObject)
{
	if (anObject != nullObject)
	{
		[nullObject release];
		nullObject = [anObject retain];
	}
}

@implementation NSObject (MiscKeyValue)

+ (BOOL)accessInstanceVariablesDirectly
{
	return YES;
}

/* These next two are not needed because we don't cache anything, but
 * I suppose I should include them just the same.
 */
+ (void)flushClassKeyBindings  {}
+ (void)flushAllKeyBindings    {}

- (id)valueForKey:(NSString *)key
{
	return get_value(self, key);
}

- (void)takeValue:(id)value forKey:(NSString *)key
{
	set_value(self, key, value);
}

- (id)handleQueryWithUnboundKey:(NSString *)key
{
	[NSException raise:NSInvalidArgumentException
		format:@"[%@ -valueForKey:]: attempt get value for unknown key: '%@'",
			NSStringFromClass([self class]), key];

	return nil;
}

- (void)handleTakeValue:(id)value forUnboundKey:(NSString *)key
{
	[NSException raise:NSInvalidArgumentException
		format:@"[%@ -takeValue:forKey:]: attempt to assign value to unknown key: '%@'",
			NSStringFromClass([self class]), key];
}

- (void)unableToSetNilForKey:(NSString *)key
{
	[NSException raise:NSInvalidArgumentException
		format:@"KeyValueCoding: Failed to assign nil to key '%@' in object of class '%@'.  "
			   @"If you want to handle assignments of nil to properties of C scalar types, "
			   @"implement 'unableToSetNilForKey:' on your EO class.",
					key, NSStringFromClass([self class])];
}

- (id)valueForKeyPath:(NSString *)keyPath
{
	NSRange dotRange = [keyPath rangeOfString:@"."];

	if (dotRange.length == 0)
	{
		return [self valueForKey:keyPath];
	}
	else
	{
		NSString *nextKey = [keyPath substringToIndex:dotRange.location];
		NSString *restOfPath = [keyPath substringFromIndex:NSMaxRange(dotRange)];

		return [[self valueForKey:nextKey] valueForKeyPath:restOfPath];
	}
}

- (void)takeValue:(id)value forKeyPath:(NSString *)keyPath
{
	NSRange dotRange = [keyPath rangeOfString:@"."];

	if (dotRange.length == 0)
	{
		return [self takeValue:value forKey:keyPath];
	}
	else
	{
		NSString *nextKey = [keyPath substringToIndex:dotRange.location];
		NSString *restOfPath = [keyPath substringFromIndex:NSMaxRange(dotRange)];

		return [[self valueForKey:nextKey] takeValue:value forKeyPath:restOfPath];
	}
}

- (NSDictionary *)valuesForKeys:(NSArray *)keyArray
{
	int i, count = [keyArray count];
	NSMutableDictionary *valueDict = [NSMutableDictionary dictionaryWithCapacity:count];

	for (i=0; i<count; i++)
	{
		NSString *key = [keyArray objectAtIndex:i];
		id value = [self valueForKey:key];

		if (value == nil) value = MiscKeyValueNullObject();
		if (value) [valueDict setObject:value forKey:key];
	}
	
	return valueDict;
}

- (void)takeValuesFromDictionary:(NSDictionary *)valueDict
{
	NSEnumerator *keyEnum = [valueDict keyEnumerator];
	NSString	 *currKey;
	
	while (currKey = [keyEnum nextObject])
	{
		id value = [valueDict objectForKey:currKey];
		if ([value isKindOfClass:[MiscKeyValueNullObject() class]]) value = nil;
		[self takeValue:value forKey:currKey];
	}
}

@end

@implementation NSDictionary (MiscKeyValue)

- (id)valueForKey:(NSString *)key
{
	return [self objectForKey:key];
}

- (void)takeValue:(id)anObject forKey:(NSString *)key
{
	[self doesNotRecognizeSelector:_cmd];
}

@end

@implementation NSMutableDictionary (MiscKeyValue)

- (void)takeValue:(id)anObject forKey:(NSString *)key
{
	if (anObject)
		[self setObject:anObject forKey:key];
	else
		[self removeObjectForKey:key];
}

@end

@implementation NSArray (MiscKeyValue)

- (id)valueForKey:(NSString *)key
{
	return [self valueForKeyPath:key];
}

- (id)valueForKeyPath:(NSString *)key
{
	if ([key isEqualToString:@"count"]) // special case... are there others?
	{
		return [super valueForKey:key];
	}
	
	if ([key hasPrefix:@"@"])
	{
		NSString *myKey = [key substringFromIndex:1];
		NSString *restOfKey = @"";
		NSRange  dotRange = [myKey rangeOfString:@"."];
		char     selectorBuffer[128];
		SEL		 computeSelector;

		if (dotRange.length > 0)
		{
			restOfKey = [myKey substringFromIndex:NSMaxRange(dotRange)];
			myKey = [myKey substringToIndex:dotRange.location];
		}

		sprintf (selectorBuffer, "compute%sForKey:", [myKey cString]);
		if (islower(selectorBuffer[7])) selectorBuffer[7] = toupper(selectorBuffer[7]);
		computeSelector = sel_getUid (selectorBuffer);
		
		if (![self respondsToSelector:computeSelector])
		{
			[NSException raise:NSInvalidArgumentException
				format:@"-[%@ valueForKeyPath:]: NSArray does not implement selector %s "
				       @"required to compute aggregate %@", NSStringFromClass([self class]),
					   selectorBuffer, key];
			return nil;
		}
		else
		{
			return [self performSelector:computeSelector withObject:restOfKey];
		}
	}
	
	else
	{
		int i, count = [self count];
		id valueArray[count];
		
		for (i=0; i<count; i++)
		{
			valueArray[i] = [[self objectAtIndex:i] valueForKeyPath:key];
		}
		
		return [[[NSArray alloc] initWithObjects:valueArray count:count] autorelease];
	}
}

- (id)computeCountForKey:(NSString *)key
{
	return [NSNumber numberWithUnsignedInt:[self count]];
}

- (id)computeSumForKey:(NSString *)key
{
	NSArray *valueArray = [self valueForKey:key];
	int     i, count = [valueArray count];
	double  sum = 0.0;
	
	if (count == 0) return nil;

	for (i=0; i<count; i++)
	{
		sum += [[valueArray objectAtIndex:i] doubleValue];
	}
	
	return [NSNumber numberWithDouble:sum];
}

- (id)computeAvgForKey:(NSString *)key
{
	NSArray *valueArray = [self valueForKey:key];
	int     i, count = [valueArray count];
	double  sum = 0.0;
	
	if (count == 0) return nil;

	for (i=0; i<count; i++)
	{
		sum += [[valueArray objectAtIndex:i] doubleValue];
	}
	
	return [NSNumber numberWithDouble:(sum / (double)count)];
}

- (id)computeMaxForKey:(NSString *)key
{
	NSArray *valueArray = [self valueForKey:key];
	int     i, count = [valueArray count];
	double  max;
	
	if (count == 0) return nil;

	max = [[valueArray objectAtIndex:0] doubleValue];

	for (i=1; i<count; i++)
	{
		double currValue = [[valueArray objectAtIndex:i] doubleValue];
		if (currValue > max) max = currValue;
	}
	
	return [NSNumber numberWithDouble:max];
}

- (id)computeMinForKey:(NSString *)key
{
	NSArray *valueArray = [self valueForKey:key];
	int     i, count = [valueArray count];
	double  min;
	
	if (count == 0) return nil;

	min = [[valueArray objectAtIndex:0] doubleValue];

	for (i=1; i<count; i++)
	{
		double currValue = [[valueArray objectAtIndex:i] doubleValue];
		if (currValue < min) min = currValue;
	}
	
	return [NSNumber numberWithDouble:min];
}

@end


// The rest is the low-level stuff

typedef union _any {
	char c;
	unsigned char uc;
	short s;
	unsigned short us;
	int i;
	unsigned int ui;
	long l;
	unsigned long ul;
	long long ll;
	unsigned long long ull;
	float f;
	double d;
	char *str;
	id obj;
} any;

/*
 * This macro will set up a correctly-typed local IMP variable, so the
 * compiler will set up the return value correctly for any type
 */
#define GET_VALUE(variable, type) \
	if (method) \
	{	\
		type (*method_imp)(id, SEL) = (type(*)(id,SEL))method->method_imp;	\
		variable = method_imp(self, getter);	\
	}		\
	else	\
	{		\
		variable = (*((type *)((char *)self + ivar->ivar_offset))); \
	}

static id get_value(id self, NSString *key)
{
	any			value;
	SEL			getter = NSSelectorFromString (key);
	Method		method;
	Ivar		ivar = NULL;
	const char	*type;
	Class		myClass = ((Class)self)->isa;  // [self class];

	/* Try to use "getter" method first */
	method = class_getInstanceMethod (myClass, getter);
	if (method != NULL)
	{
		type = method->method_types;
	}
	else
	{
		if ([myClass accessInstanceVariablesDirectly])
			ivar = class_getInstanceVariable (myClass, [key cString]);

		if (ivar == NULL) return [self handleQueryWithUnboundKey:key];

		type = ivar->ivar_type;
	}

	switch (type[0])
	{
		case _C_CHR:
			GET_VALUE(value.c, char);
			return [NSNumber numberWithChar:value.c];
		case _C_UCHR:
			GET_VALUE(value.uc, unsigned char);
			return [NSNumber numberWithUnsignedChar:value.uc];
		case _C_SHT:
			GET_VALUE(value.s, short);
			return [NSNumber numberWithShort:value.s];
		case _C_USHT:
			GET_VALUE(value.us, unsigned short);
			return [NSNumber numberWithUnsignedShort:value.us];
		case _C_INT:
			GET_VALUE(value.i, int);
			return [NSNumber numberWithInt:value.i];
		case _C_UINT:
			GET_VALUE(value.ui, unsigned int);
			return [NSNumber numberWithUnsignedInt:value.ui];
		case _C_LNG:
			GET_VALUE(value.l, long);
			return [NSNumber numberWithLong:value.l];
		case _C_ULNG:
			GET_VALUE(value.ul, unsigned long);
			return [NSNumber numberWithUnsignedLong:value.ul];
		case 'q': // NSObjCLonglongType from NSInvocation.h
			GET_VALUE(value.ll, long long);
			return [NSNumber numberWithLongLong:value.ll];
		case 'Q':
			GET_VALUE(value.ull, unsigned long long);
			return [NSNumber numberWithUnsignedLongLong:value.ull];
		case _C_FLT:
			GET_VALUE(value.f, float);
			return [NSNumber numberWithFloat:value.f];
		case _C_DBL:
			GET_VALUE(value.d, double);
			return [NSNumber numberWithDouble:value.d];
		case _C_CHARPTR:
			GET_VALUE(value.str, char *);
			return [NSString stringWithCString:value.str];
		case _C_ID:
			GET_VALUE(value.obj, id);
			// Should I do the retain/autorelease?  May be too inefficient.
			return [[value.obj retain] autorelease];
		default:
			return [self handleQueryWithUnboundKey:key];
	}

	return nil;
}

/*
 * This macro will set up a correctly-typed local IMP variable, so the
 * compiler will set up the frame correctly for any type argument
 */
#define SET_VALUE(variable, type) \
	if (method) \
	{	\
		void (*method_imp)(id, SEL, type) = (void(*)(id,SEL,type))method->method_imp;	\
		method_imp(self, setter, variable);	\
	}		\
	else	\
	{		\
		(*((type *)((char *)self + ivar->ivar_offset))) = (variable); \
	}

static void set_value(id self, NSString *key, id objectValue)
{
	char		sbuf[128];
	SEL			setter;
	Ivar		ivar = NULL;
	Method		method;
	const char	*type, *keyCString = [key cString];
	any			value;
	int			offset;
	Class		myClass = ((Class)self)->isa;  // [self class];

	/* setter methods are of the form "setVar" to set ivar "var" */
	sprintf (sbuf, "set%s:", keyCString);
	if (islower(sbuf[3])) sbuf[3] = toupper (sbuf[3]);
	setter = sel_getUid (sbuf);

	/* Try to use "setter" method first */
	method = class_getInstanceMethod (myClass, setter);
	if (method != NULL)
	{
		method_getArgumentInfo (method, 2, &type, &offset);
	}
	else
	{
		if ([myClass accessInstanceVariablesDirectly])
			ivar = class_getInstanceVariable (myClass, keyCString);

		if (ivar == NULL)
		{
			[self handleTakeValue:objectValue forUnboundKey:key];
			return;
		}

		type = ivar->ivar_type;
	}

	if (objectValue == nil && type[0] != _C_ID)
	{
		[self unableToSetNilForKey:key];
		return;
	}

	switch (type[0])
	{
		case _C_INT:
			value.i = [objectValue intValue];
			SET_VALUE(value.i, int);
			break;
		case _C_UINT:
			value.ui = [objectValue unsignedIntValue];
			SET_VALUE(value.ui, unsigned int);
			break;
		case _C_LNG:
			value.l = [objectValue longValue];
			SET_VALUE(value.l, long);
			break;
		case _C_ULNG:
			value.ul = [objectValue unsignedLongValue];
			SET_VALUE(value.ul, unsigned long);
			break;
		case 'q': // NSObjCLonglongType from NSInvocation.h
			value.ll = [objectValue longLongValue];
			SET_VALUE(value.ll, long long);
			break;
		case 'Q':
			value.ull = [objectValue unsignedLongLongValue];
			SET_VALUE(value.ull, unsigned long long);
			break;
		case _C_FLT:
			value.f = [objectValue floatValue];
			SET_VALUE(value.f, float);
			break;
		case _C_DBL:
			value.d = [objectValue doubleValue];
			SET_VALUE(value.d, double);
			break;
//		case _C_CHARPTR:
//			value.str = (char*)[objectValue cString];
//			SET_VALUE(value.str, char *);
//			break;
		case _C_ID:
			SET_VALUE(objectValue, id);
			break;
		default:
			[self handleTakeValue:objectValue forUnboundKey:key];
			break;
	}
}

