//
//  MiscKeyValue.h
//    Written by Carl Lindberg Copyright 1998 by Carl Lindberg.
//                     All rights reserved.
//      This notice may not be removed from this source code.
//
//	This header is included in the MiscKit by permission from the author
//	and its use is governed by the MiscKit license, found in the file
//	"License.rtf" in the MiscKit distribution.  Please refer to that file
//	for a list of all applicable permissions and restrictions.
//	

#import <Foundation/NSObject.h>

@class NSDictionary, NSArray;

/*"
 * Functions to query and set the null object, which is used in
 * -valuesForKeys to represent nil values and in
 * -takeValues:fromDictionary: to determine if the value really should be
 * nil before calling -takeValue:forKey:. By default, this is a MiscNull
 * object.
"*/
extern id MiscKeyValueNullObject();
extern void MiscKeyValueSetNullObject(id anObject);

@interface NSObject (MiscKeyValue)

- (id)valueForKey:(NSString *)key;
    // The default implementation searches first for a selector with
    // the same name as the key, and then for an instance variable with
    // the same name as the key.  Classes can override this method to
    // add custom behavior.
    // Default implementation raises an exception if an unknown key is
    // passed in.

- (void)takeValue:(id)value forKey:(NSString *)key;
    // Sets property of the receiver with supplied values.
    // The default implementation searches first for a selector named
    // setKey: (where "Key" is replaced by the key in the dictionary),
    // and then for an instance variable with the same name as the key.
    // Classes can override this method to add custom behavior.
    // Default implementation raises an exception if an unknown key is
    // passed in.

+ (BOOL)accessInstanceVariablesDirectly;
    // This method can be overridden on a class to disable
    // accessing instance variables when a matching method
    // cannot be found.  Default on NSObject is YES.

- (id)valueForKeyPath:(NSString *)key;
    // A "key path" is a key of the form "relname.property" (with one or
    // more relationship names).  For example, [emp valueForKeyPath:@"department.name"]
    // will ask the receiving employee for its department object (via valueForKey:)
    // and will in turn ask the department for its name.  This provides attribute
    // "flattening" by navigating the object graph in memory.

- (void)takeValue:value forKeyPath:(NSString *)key;
    // Sets the property identified by the given key path.  @"address.street"
    // sent to an employee object will first get the address object (via
    // valueForKey:@"address") and then perform [address setValue:value forKey:@"street"]

- (NSDictionary *)valuesForKeys:(NSArray *)keys;
    // aggregate version of valueForKey:
    // Nil values returned from valueForKey: are put in return dictionary
    // as EONull

- (void)takeValuesFromDictionary:(NSDictionary *)dictionary;
    // aggregate version of takeValue:forKey:
    // EONull in the input dictionary is passed to object as takeValue:nil


- (id)handleQueryWithUnboundKey:(NSString *)key;
    // Called by default implementation of valueForKey: when the key
    // does not correspond to a method or instance variable in the
    // receiver.  The default implementation raises an exception.
    // Classes can override this method to be more permissive
    // (perhaps by returing nil).  The return value is passed back
    // from valueForKey:

- (void)handleTakeValue:(id)value forUnboundKey:(NSString *)key;
    // Called by default implementation of takeValue:forKey: when the key
    // does not correspond to a method or instance variable in the
    // receiver.  The default implementation raises an exception.
    // Classes can override this method to be more permissive.
    // (perhaps ignoring the error or logging to the console)

   
- (void)unableToSetNilForKey:(NSString *)key;
    // Classes can implement this method to determine the behavior
    // when nil is assigned to a property in an EO that requires
    // a C scalar type (such as int or float).  One possible implementation
    // is to call takeValue:forKey: recursively with a special constant
    // value (like [NSNumber numberWithInt:0]) or with an object that will
    // return the desired values to assign for the methods floatValue, intValue,
    // unsignedIntValue, etc.

+ (void)flushClassKeyBindings;
+ (void)flushAllKeyBindings;
	// These are methods in NeXT's implementation, since they cache the
	// method/ivar bindings.  We don't, so these methods do nothing.  They
	// are only implemented for completeness' sake.

@end

#import <Foundation/NSArray.h>

@interface NSArray (MiscKeyValue)
- (id)valueForKey:(NSString *)key;
    // When passed a "normal" key, returns an array composed of the
    // results of calling valueForKey: on all elements of the array.
    // When passed a key prefixed with "@" a single value is returned
    // which is the result of invoking an aggregate function on the values
    // of the array.
    // For instance, when passed @sum.budget, it will invoke -computeSumForKey:@"budget"
    // on the array.  The following aggregates are defined:
    //    @sum, @count, @avg, @min, @max
    // New methods added in categories on NSArray will automatically be detected
    // by the connection inspector in InterfaceBuilder.
	// "count" is a special key that will return the count, just like @count.

- (id)computeSumForKey:(NSString *)key;
- (id)computeAvgForKey:(NSString *)key;
- (id)computeCountForKey:(NSString *)key;
- (id)computeMaxForKey:(NSString *)key;
- (id)computeMinForKey:(NSString *)key;

@end
