/*
 * Copyright (c) 1993-1996 by NeXT Software, Inc. as an unpublished work.
 * All rights reserved.
 *
 * QVisionModes.c -- support for QVision driver modes.
 *
 * 09 Oct 1995	Rakesh Dubey
 *	Added more modes.
 * Author:  Derek B Clegg	21 May 1993
 * Based on work by Joe Pasqua, 30 September 1992.
 */
 
#include "QVisionModes.h"

/* Tables of register values for the supported modes. */

/* QVision 1024 x 768 x 8 (QVision mode 0x38) @ 60Hz.
 */
static const struct QVisionMode Mode_38_60Hz = {
    "QVision[1024 x 768 x 8] @ 60Hz", QVisionAdapter, NO,

    /* Control register 1 value (63CA).  */
    0x03,

    /* DAC command register 1 value (13C8). */
    0x40,

    /* Overflow register 1 value (3CF.42). */
    0x00,

    /* Overflow register 2 value (3CF.51). */
    0x00,

    /* Standard VGA data. */
    {
	0xe3,	/* Miscellaneous output register value (3C2). */
	0x00,	/* Feature control register value (3DA). */

	/* Sequencer register values. */
	{ 0x01, 0x01, 0xff, 0x00, 0x0e },

	/* CRTC register values (3D5.00 - 3D5.18). */
	{
	    0x9b, 0x7f, 0x7f, 0x9e, 0x87, 0x17, 0x31, 0xf5, 0x00, 0x60, 0x00,
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x0d, 0x94, 0xff, 0x80, 0x00, 0xff,
	    0x31, 0xe3, 0xff,
	},

	/* Attribute controller register values (3C0.00 - 3C0.13). */
	{
	    0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09,
	    0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f, 0x41, 0x00, 0x0f, 0x00,
	},

	/* Graphics controller register values (3CF.00 - 3CF.08). */
	{ 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x05, 0x0f, 0xff },
    }
};

/* QVision 1024 x 768 x 8 (QVision mode 0x38) @ 66Hz.
 */
static const struct QVisionMode Mode_38_66Hz = {
    "QVision[1024 x 768 x 8] @ 66Hz", QVisionAdapter, NO,
    
    /* Control register 1 value (63CA).  */
    0x03,

    /* DAC command register 1 value (13C8). */
    0x40,

    /* Overflow register 1 value (3CF.42). */
    0x00,

    /* Overflow register 2 value (3CF.51). */
    0x00,

    /* Standard VGA data. */
    {
	0x27,	/* Miscellaneous output register value (3C2). */
	0x00,	/* Feature control register value (3DA). */

	/* Sequencer register values. */
	{ 0x01, 0x01, 0xff, 0x00, 0x0e },

	/* CRTC register values (3D5.00 - 3D5.18). */
	{
	    0xa1, 0x7f, 0x7f, 0x84, 0x85, 0x9b, 0x2e, 0xf5, 0x00, 0x60, 0x00,
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x07, 0x8b, 0xff, 0x80, 0x00, 0xff,
	    0x2e, 0xe3, 0xff,
	},

	/* Attribute controller register values (3C0.00 - 3C0.13). */
	{
	    0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09,
	    0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f, 0x41, 0x00, 0x0f, 0x00,
	},

	/* Graphics controller register values (3CF.00 - 3CF.08). */
	{ 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x05, 0x0f, 0xff },
    }
};

/* QVision 1024 x 768 x 8 (QVision mode 0x38) @ 72Hz.
 */
static const struct QVisionMode Mode_38_72Hz = {
    "QVision[1024 x 768 x 8] @ 72Hz", QVisionAdapter, NO,

    /* Control register 1 value (63CA).  */
    0x03,

    /* DAC command register 1 value (13C8). */
    0x40,

    /* Overflow register 1 value (3CF.42). */
    0x00,

    /* Overflow register 2 value (3CF.51). */
    0x00,

    /* Standard VGA data. */
    {
	0x2b,	/* Miscellaneous output register value (3C2). */
	0x00,	/* Feature control register value (3DA). */

	/* Sequencer register values. */
	{ 0x01, 0x01, 0xff, 0x00, 0x0e },

	/* CRTC register values (3D5.00 - 3D5.18). */
	{
	    0x9e, 0x7f, 0x7f, 0x81, 0x83, 0x93, 0x1e, 0xf1, 0x00, 0x60, 0x00,
	    0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0x83, 0xff, 0x80, 0x00, 0xff,
	    0x1e, 0xe3, 0xff,
	},

	/* Attribute controller register values (3C0.00 - 3C0.13). */
	{
	    0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09,
	    0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f, 0x41, 0x00, 0x0f, 0x00,
	},
	
	/* Graphics controller register values (3CF.00 - 3CF.08). */
	{ 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x05, 0x0f, 0xff },
    }
};

/* Orion12 1024 x 768 x 8 (QVision mode 0x38) @ 76Hz.
 */
static const struct QVisionMode Mode_38_76Hz = {
    "Orion12[1024 x 768 x 8] @ 76Hz", Orion12Adapter, YES,

    /* Control register 1 value (63CA).  */
    0x03,

    /* DAC command register 1 value (13C8). */
    0x40,

    /* Overflow register 1 value (3CF.42). */
    0x00,

    /* Overflow register 2 value (3CF.51). */
    0x00,

    /* Standard VGA data. */
    {
	0x33,	/* Miscellaneous output register value (3C2). */
	0x00,	/* Feature control register value (3DA). */

	/* Sequencer register values. */
	{ 0x01, 0x01, 0xff, 0x00, 0x0e },

	/* CRTC register values (3D5.00 - 3D5.18). */
	{
	    0x9c, 0x7f, 0x7f, 0x1f, 0x85, 0x10, 0x26, 0xf5, 0x00, 0x60, 0x00,
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x03, 0x8b, 0xff, 0x80, 0x00, 0xff,
	    0x26, 0xe3, 0xff,
	},
	
	/* Attribute controller register values (3C0.00 - 3C0.13). */
	{
	    0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09,
	    0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f, 0x41, 0x00, 0x0f, 0x00,
	},
	
	/* Graphics controller register values (3CF.00 - 3CF.08). */
	{ 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x05, 0x0f, 0xff },
    }
};

/* QVision 640 x 400 x 24 (QVision mode 0x4E) @ 60Hz.
 */
static const struct QVisionMode Mode_4E_60Hz = {
    "QVision[640 x 400 x 24] @ 60Hz", QVisionAdapter, NO,

    /* Control register 1 value (63CA). */
    0x07,

    /* DAC command register 1 value (13C8). */
    0x00,

    /* Overflow register 1 value (3CF.42). */
    0x01,

    /* Overflow register 2 value (3CF.51). */
    0x08,

    /* Standard VGA data. */
    {
	0x6f,	/* Miscellaneous output register value (3C2). */
	0x00,	/* Feature control register value (3DA). */

	/* Sequencer register values. */
	{ 0x01, 0x01, 0xff, 0x00, 0x0e },
	
	/* CRTC register values (3D5.00 - 3D5.18). */
	{
	    0x5f, 0x4f, 0x50, 0x81, 0x53, 0x9f, 0x0b, 0x3e, 0x00, 0x40, 0x00,
	    0x00, 0x00, 0x00, 0x00, 0x00, 0xc2, 0x84, 0x8f, 0x40, 0x00, 0x95,
	    0x03, 0xe3, 0xff,
	},
	
	/* Attribute controller register values (3C0.00 - 3C0.13). */
	{
	    0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09,
	    0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f, 0x41, 0x00, 0x0f, 0x00,
	},
	
	/* Graphics controller register values (3CF.00 - 3CF.08). */
	{ 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x05, 0x0f, 0xff },
    }    
};

/* QVision 640 x 400 x 24 (QVision mode 0x4E) @ 70Hz.
 */
static const struct QVisionMode Mode_4E_70Hz = {
    "QVision[640 x 400 x 24] @ 70Hz", QVisionAdapter, NO,

    /* Control register 1 value (63CA). */
    0x07,

    /* DAC command register 1 value (13C8). */
    0x00,

    /* Overflow register 1 value (3CF.42). */
    0x01,

    /* Overflow register 2 value (3CF.51). */
    0x28,

    /* Standard VGA data. */
    {
	0x6f,	/* Miscellaneous output register value (3C2). */
	0x00,	/* Feature control register value (3DA). */

	/* Sequencer register values. */
	{ 0x01, 0x01, 0xff, 0x00, 0x0e },

	/* CRTC register values (3D5.00 - 3D5.18). */
	{
	    0x5f, 0x4f, 0x50, 0x82, 0x54, 0x80, 0xbf, 0x1f, 0x00, 0x40, 0x00,
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x9c, 0x8e, 0x8f, 0x40, 0x00, 0x96,
	    0xb9, 0xe3, 0xff,
	},
	
	/* Attribute controller register values (3C0.00 - 3C0.13). */
	{
	    0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09,
	    0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f, 0x41, 0x00, 0x0f, 0x00,
	},
	
	/* Graphics controller register values (3CF.00 - 3CF.08). */
	{ 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x05, 0x0f, 0xff },
    }
};

/* QVision 800 x 600 x 15 (QVision mode 0x3F) @ 60Hz.
 */
static const struct QVisionMode Mode_3F_60Hz = {
    "QVision[800 x 600 x 15] @ 60Hz", QVisionAdapter, NO,

    /* Control register 1 value (63CA). */
    0x05,

    /* DAC command register 1 value (13C8). */
    0x20,

    /* Overflow register 1 value (3CF.42). */
    0x00,

    /* Overflow register 2 value (3CF.51). */
    0x20,

    /* Standard VGA data. */
    {
	0xf3,	/* Miscellaneous output register value (3C2). */
	0x00,	/* Feature control register value (3DA). */

	/* Sequencer register values. */
	{ 0x01, 0x01, 0xff, 0x00, 0x0e },
	
	/* CRTC register values (3D5.00 - 3D5.18). */
	{
	    0x7f, 0x63, 0x64, 0x80, 0x68, 0x19, 0x73, 0xf0, 0x00, 0x60, 0x00,
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x58, 0x8c, 0x57, 0xc8, 0x00, 0x58,
	    0x72, 0xe3, 0xff,
	},
	
	/* Attribute controller register values (3C0.00 - 3C0.13). */
	{
	    0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09,
	    0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f, 0x41, 0x00, 0x0f, 0x00,
	},
	
	/* Graphics controller register values (3CF.00 - 3CF.08). */
	{ 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x05, 0x0f, 0xff },
    }
};

/* QVision 800 x 600 x 15 (QVision mode 0x3F) @ 72Hz.
 */
static const struct QVisionMode Mode_3F_72Hz = {
    "QVision[800 x 600 x 15] @ 72Hz", QVisionAdapter, NO,

    /* Control register 1 value (63CA). */
    0x05,

    /* DAC command register 1 value (13C8). */
    0x20,

    /* Overflow register 1 value (3CF.42). */
    0x00,

    /* Overflow register 2 value (3CF.51). */
    0x20,

    /* Standard VGA data. */
    {
	0x23,	/* Miscellaneous output register value (3C2). */
	0x00,	/* Feature control register value (3DA). */

	/* Sequencer register values. */
	{ 0x01, 0x01, 0xff, 0x00, 0x0e },
	
	/* CRTC register values (3D5.00 - 3D5.18). */
	{
	    0x84, 0x63, 0x64, 0x83, 0x6b, 0x1c, 0x73, 0xf0, 0x00, 0x60, 0x00,
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x58, 0x5c, 0x57, 0xc8, 0x00, 0x58,
	    0x72, 0xe3, 0xff,
	},

	/* Attribute controller register values (3C0.00 - 3C0.13). */
	{
	    0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09,
	    0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f, 0x41, 0x00, 0x0f, 0x00,
	},
	
	/* Graphics controller register values (3CF.00 - 3CF.08). */
	{ 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x05, 0x0f, 0xff },
    }
};

/* Orion12 1280 x 1024 x 8 (QVision mode 0x3A) @ 68Hz.
 */
static const struct QVisionMode Mode_3A_68Hz = {
    "Orion12[1280 x 1024 x 8] @ 68Hz", Orion12Adapter, YES,

    /* Control register 1 value (63CA). */
    0x03,

    /* DAC command register 1 value (13C8). */
    0x40,

    /* Overflow register 1 value (3CF.42). */
    0x81,

    /* Overflow Register 2 values (3CF.51h). */
    0xe8,

    /* Standard VGA data. */
    {
	0x23,	/* Miscellaneous output register values (3C2). */
	0x00,	/* Feature control register value (3DA). */

	/* Sequencer register values. */
	{ 0x01, 0x01, 0xff, 0x00, 0x0e },

	/* CRTC register values (3D5.00 - 3D5.18h) */
	{
	    0xcf, 0x9f, 0x9f, 0x12, 0xa4, 0x19, 0x2e, 0x5a, 0x00, 0x60, 0x00,
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x03, 0x8b, 0xff, 0x00, 0x00, 0xff,
	    0x2e, 0xe3, 0xff,
	},
	
	/* Attribute controller register values (3C0.00 - 3C0.13). */
	{
	    0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09,
	    0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f, 0x41, 0x00, 0x0f, 0x00,
	},
	
	/* Graphics controller register values (3CF.00 - 3CF.08). */
	{ 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x05, 0x0f, 0xff },
    }
};

/* Orion 1024 x 768 x 15 (QVision mode 0x40) @ 60Hz.
 */
static const struct QVisionMode Mode_40_60Hz = {
    "Orion[1024 x 768 x 15] @ 60Hz", OrionAdapter, NO,

    /* Control register 1 value (63CA). */
    0x05,

    /* DAC command register 1 value (13C8). */
    0x20,

    /* Overflow register 1 value (3CF.42). */
    0x01,

    /* Overflow register 2 value (3CF.51). */
    0x00,

    /* Standard VGA data. */
    {
	0xe3,	/* Miscellaneous output register values (3C2). */
	0x00,	/* Feature control register value (3DA). */

	/* Sequencer register values. */
	{ 0x01, 0x01, 0xff, 0x00, 0x0e },
	
	/* CRTC register values (3D5.00 - 3D5.18). */
	{
	    0x9b, 0x7f, 0x7f, 0x9e, 0x87, 0x17, 0x31, 0xf5, 0x00, 0x60, 0x00,
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x0d, 0x94, 0xff, 0x00, 0x00, 0xff,
	    0x31, 0xe3, 0xff,
	},
	
	/* Attribute controller register values (3C0.00 - 3C0.13). */
	{
	    0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09,
	    0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f, 0x41, 0x00, 0x0f, 0x00,
	},
	
	/* Graphics controller register values (3CF.00 - 3CF.08). */
	{ 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x05, 0x0f, 0xff },
    }
};

/* Orion 1024 x 768 x 15 (QVision mode 0x40) @ 72Hz.
 */
static const struct QVisionMode Mode_40_72Hz = {
    "Orion[1024 x 768 x 15] @ 72Hz", OrionAdapter, NO,

    /* Control register 1 value (63CA). */
    0x05,

    /* DAC command register 1 value (13C8). */
    0x20,

    /* Overflow register 1 value (3CF.42). */
    0x01,

    /* Overflow register 2 value (3CF.51). */
    0x00,

    /* Standard VGA data. */
    {
	0x2b,	/* Miscellaneous output register value (3C2). */
	0x00,	/* Feature control register value (3DA). */

	/* Sequencer register values. */
	{ 0x01, 0x01, 0xff, 0x00, 0x0e },
	
	/* CRTC register values (3D5.00 - 3D5.18). */
	{
	    0x9e, 0x7f, 0x7f, 0x81, 0x83, 0x93, 0x1e, 0xf1, 0x00, 0x60, 0x00,
	    0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0x83, 0xff, 0x00, 0x00, 0xff,
	    0x1e, 0xe3, 0xff,
	},
	
	/* Attribute controller register values (3C0.00 - 3C0.13). */
	{
	    0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09,
	    0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f, 0x41, 0x00, 0x0f, 0x00,
	},
	
	/* Graphics controller register values (3CF.00 - 3CF.08). */
	{ 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x05, 0x0f, 0xff },
    }
};

/* Orion 640 x 480 x 24 (QVision mode 0x4F) @ 60Hz.
 */
static const struct QVisionMode Mode_4F_60Hz = {
    "Orion[640 x 480 x 24] @ 60Hz", OrionAdapter, NO,

    /* Control register 1 values (63CA).  */
    0x07,

    /* DAC command register 1 value (13C8). */
    0x00,

    /* Overflow register 1 value (3CF.42). */
    0x02,

    /* Overflow register 2 value (3CF.51). */
    0x08,

    /* Standard VGA data. */
    {
	0xef,	/* Miscellaneous output register value (3C2). */
	0x00,	/* Feature control register value (3DA). */

	/* Sequencer register values. */
	{ 0x01, 0x01, 0xff, 0x00, 0x0e },
	
	/* CRTC register values (3D5.00 - 3D5.18). */
	{
	    0x5f, 0x4f, 0x50, 0x81, 0x53, 0x9f, 0x0b, 0x3e, 0x00, 0x40, 0x00,
	    0x00, 0x00, 0x00, 0x00, 0x00, 0xea, 0x8c, 0xdf, 0x00, 0x00, 0xe5,
	    0x03, 0xe3, 0xff,
	},

	/* Attribute controller register values (3C0.00 - 3C0.13). */
	{
	    0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09,
	    0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f, 0x41, 0x00, 0x0f, 0x00,
	},
	
	/* Graphics controller register values (3CF.00 - 3CF.08). */
	{ 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x05, 0x0f, 0xff },
    }
};


/*
 * Orion 800 x 600 x 8 (QVision mode 0x34) @ 60Hz.
 */
static const struct QVisionMode Mode_34_60Hz = {
    "Orion[800 x 600 x 8] @ 60Hz", OrionAdapter, NO,

    /* Control register 1 values (63CA).  */
    0x03,

    /* DAC command register 1 value (13C8). */
    0x40,

    /* Overflow register 1 value (3CF.42). */
    0x00,

    /* Overflow register 2 value (3CF.51). */
    0x20,

    /* Standard VGA data. */
    {
	0xf3,	/* Miscellaneous output register value (3C2). */
	0x00,	/* Feature control register value (3DA). */

	/* Sequencer register values. */
	{ 0x03, 0x01, 0xff, 0x00, 0x0e },
	
	/* CRTC register values (3D5.00 - 3D5.18). */
	{
	    0x7f, 0x63, 0x64, 0x80, 0x68, 0x19, 0x73, 0xf0, 0x00, 0x60, 0x00,
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x58, 0xbc, 0x57, 0x80, 0x00, 0x58,
	    0x72, 0xe3, 0xff,
	},

	/* Attribute controller register values (3C0.00 - 3C0.13). */
	{
	    0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09,
	    0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f, 0x41, 0x00, 0x0f, 0x00,
	},
	
	/* Graphics controller register values (3CF.00 - 3CF.08). */
	{ 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x05, 0x0f, 0xff },
    }
};


/*
 * Orion 800 x 600 x 8 (QVision mode 0x34) @ 72Hz.
 */
static const struct QVisionMode Mode_34_72Hz = {
    "Orion[800 x 600 x 8] @ 72Hz", OrionAdapter, NO,

    /* Control register 1 values (63CA).  */
    0x03,

    /* DAC command register 1 value (13C8). */
    0x40,

    /* Overflow register 1 value (3CF.42). */
    0x00,

    /* Overflow register 2 value (3CF.51). */
    0x20,

    /* Standard VGA data. */
    {
	0x3f,	/* Miscellaneous output register value (3C2). */
	0x00,	/* Feature control register value (3DA). */

	/* Sequencer register values. */
	{ 0x03, 0x01, 0xff, 0x00, 0x0e },
	
	/* CRTC register values (3D5.00 - 3D5.18). */
	{
	    0x7d, 0x63, 0x63, 0x00, 0x6d, 0x1c, 0x98, 0xf0, 0x00, 0x60, 0x00,
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x7c, 0xb2, 0x57, 0x80, 0x00, 0x57,
	    0x98, 0xe3, 0xff,
	},

	/* Attribute controller register values (3C0.00 - 3C0.13). */
	{
	    0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09,
	    0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f, 0x41, 0x00, 0x0f, 0x00,
	},
	
	/* Graphics controller register values (3CF.00 - 3CF.08). */
	{ 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x05, 0x0f, 0xff },
    }
};


IODisplayInfo QVisionModeTable[] = {
    /* 0: QVision 1024 x 768 x 8 (Mode 0x38) @ 60Hz. */
    {
	1024, 768, 1024, 1024, 60, 0,
	IO_8BitsPerPixel, IO_OneIsWhiteColorSpace, "WWWWWWWW",
	0, (void *)&Mode_38_60Hz,
    },

    /* 1: QVision 1024 x 768 x 8 (Mode 0x38) @ 66Hz. */
    {
	1024, 768, 1024, 1024, 66, 0,
	IO_8BitsPerPixel, IO_OneIsWhiteColorSpace, "WWWWWWWW",
	0, (void *)&Mode_38_66Hz,
    },

    /* 2: QVision 1024 x 768 x 8 (Mode 0x38) @ 72Hz. */
    {
	1024, 768, 1024, 1024, 72, 0,
	IO_8BitsPerPixel, IO_OneIsWhiteColorSpace, "WWWWWWWW",
	0, (void *)&Mode_38_72Hz,
    },

    /* 3: Orion12 1024 x 768 x 8 (Mode 0x38) @ 76Hz. */
    {
	1024, 768, 1024, 1024, 76, 0,
	IO_8BitsPerPixel, IO_OneIsWhiteColorSpace, "WWWWWWWW",
	0, (void *)&Mode_38_76Hz,
    },

    /* 4: QVision 640 x 400 x 24 (Mode 0x4E) @ 60Hz. */
    {
	640, 400, 640, 2560, 60, 0,
	IO_24BitsPerPixel, IO_RGBColorSpace,"--------RRRRRRRRGGGGGGGGBBBBBBBB",
	0, (void *)&Mode_4E_60Hz,
    },

    /* 5: QVision 640 x 400 x 24 (Mode 0x4E) @ 70Hz. */
    {
	640, 400, 640, 2560, 70, 0,
	IO_24BitsPerPixel, IO_RGBColorSpace,"--------RRRRRRRRGGGGGGGGBBBBBBBB",
	0, (void *)&Mode_4E_70Hz,
    },

    /* 6: QVision 800 x 600 x 15 (QVision mode 0x3F) @ 60Hz. */
    {
	800, 600, 800, 1600, 60, 0,
	IO_15BitsPerPixel, IO_RGBColorSpace, "-RRRRRGGGGGBBBBB",
	0, (void *)&Mode_3F_60Hz,
    },

    /* 7: QVision 800 x 600 x 15 (Mode 0x3F) @ 72Hz. */
    {
	800, 600, 800, 1600, 72, 0,
	IO_15BitsPerPixel, IO_RGBColorSpace, "-RRRRRGGGGGBBBBB",
	0, (void *)&Mode_3F_72Hz,
    },

    /* 8: Orion12 1280 x 1024 x 8 (Mode 0x3A) @ 68Hz. */
    {
	1280, 1024, 2048, 2048, 68, 0,
	IO_8BitsPerPixel, IO_OneIsWhiteColorSpace, "WWWWWWWW",
	0, (void *)&Mode_3A_68Hz,
    },

    /* 9: Orion 1024 x 768 x 15 (Mode 0x40) @ 60Hz. */
    {
	1024, 768, 1024, 2048, 60, 0,
	IO_15BitsPerPixel, IO_RGBColorSpace, "-RRRRRGGGGGBBBBB",
	0, (void *)&Mode_40_60Hz,
    },

    /* 10: Orion 1024 x 768 x 15 (Mode 0x40) @ 72Hz. */
    {
	1024, 768, 1024, 2048, 72, 0,
	IO_15BitsPerPixel, IO_RGBColorSpace, "-RRRRRGGGGGBBBBB",
	0, (void *)&Mode_40_72Hz,
    },

    /* 11: Orion 640 x 480 x 24 (Mode 0x4F) @ 60Hz. */
    {
	640, 480, 1024, 4096, 60, 0,
	IO_24BitsPerPixel, IO_RGBColorSpace,"--------RRRRRRRRGGGGGGGGBBBBBBBB",
	0, (void *)&Mode_4F_60Hz,
    },

	/* 8-bit color modes */
    /* 12: QVision 1024 x 768 x 8 (Mode 0x38) @ 60Hz. */
    {
	1024, 768, 1024, 1024, 60, 0,
	IO_8BitsPerPixel, IO_RGBColorSpace, "PPPPPPPP",
	0, (void *)&Mode_38_60Hz,
    },

    /* 13: QVision 1024 x 768 x 8 (Mode 0x38) @ 66Hz. */
    {
	1024, 768, 1024, 1024, 66, 0,
	IO_8BitsPerPixel, IO_RGBColorSpace, "PPPPPPPP",
	0, (void *)&Mode_38_66Hz,
    },

    /* 14: QVision 1024 x 768 x 8 (Mode 0x38) @ 72Hz. */
    {
	1024, 768, 1024, 1024, 72, 0,
	IO_8BitsPerPixel, IO_RGBColorSpace, "PPPPPPPP",
	0, (void *)&Mode_38_72Hz,
    },

    /* 15: Orion12 1024 x 768 x 8 (Mode 0x38) @ 76Hz. */
    {
	1024, 768, 1024, 1024, 76, 0,
	IO_8BitsPerPixel, IO_RGBColorSpace, "PPPPPPPP",
	0, (void *)&Mode_38_76Hz,
    },

    /* 16: Orion12 1280 x 1024 x 8 (Mode 0x3A) @ 68Hz. */
    {
	1280, 1024, 2048, 2048, 68, 0,
	IO_8BitsPerPixel, IO_RGBColorSpace, "PPPPPPPP",
	0, (void *)&Mode_3A_68Hz,
    },
    /* 17: Orion12 800 x 600 x 8 (Mode 0x34) @ 60Hz. */
    {
	800, 600, 1024, 1024, 60, 0,
	IO_8BitsPerPixel, IO_RGBColorSpace, "PPPPPPPP",
	0, (void *)&Mode_34_60Hz,
    },
    /* 18: Orion12 800 x 600 x 8 (Mode 0x34) @ 60Hz. */
    {
	800, 600, 1024, 1024, 60, 0,
	IO_8BitsPerPixel, IO_OneIsWhiteColorSpace, "WWWWWWWW",
	0, (void *)&Mode_34_60Hz,
    },
    /* 19: Orion12 800 x 600 x 8 (Mode 0x34) @ 72Hz. */
    {
	800, 600, 1024, 1024, 72, 0,
	IO_8BitsPerPixel, IO_RGBColorSpace, "PPPPPPPP",
	0, (void *)&Mode_34_72Hz,
    },
    /* 20: Orion12 800 x 600 x 8 (Mode 0x34) @ 72Hz. */
    {
	800, 600, 1024, 1024, 72, 0,
	IO_8BitsPerPixel, IO_OneIsWhiteColorSpace, "WWWWWWWW",
	0, (void *)&Mode_34_72Hz,
    },
};

const int QVisionModeTableCount =
    sizeof(QVisionModeTable)/sizeof(QVisionModeTable[0]);
