/* Copyright (c) 1993-1996 by NeXT Software, Inc. as an unpublished work.
 * All rights reserved.
 *
 * QVisionDAC.m -- DAC support for the QVision.
 *
 * History
 * Thu Sep 15 16:13:45 PDT 1994, James C. Lee
 *   Added AT&T 20C505 DAC support
 * Author:  Derek B Clegg	30 June 1993
 * Based on work by Joe Pasqua.
 */
#import <driverkit/generalFuncs.h>
#import <driverkit/i386/ioPorts.h>
#import <bsd/dev/ev_types.h>
#import "QVision.h"

/* The `ProgramDAC' category of `QVision'. */

@implementation QVision (ProgramDAC)

// also check for AT&T 20C505 DAC; it's BT485 compatible
static DACtype checkForBrooktreeDAC(void)
{
    DACtype dac;

    /* Unlock the extended graphics registers. */
    outw(VGA_GRFX_INDEX, 0x050f);

    /* Unlock some more extended registers. */
    outb(VGA_GRFX_INDEX, 0x10);
    outb(VGA_GRFX_DATA, 0x08);

    /* Read the status register. */
    switch (inb(DAC_EXT_REG) & 0xF0) {
    case 0x40:
	dac = Bt484;
	break;
    case 0x80:
	dac = Bt485;
	break;
    case 0x20:
	dac = Bt485A;
	break;
    case 0xd0:
	dac = ATT20C505;
	break;
    default:
	dac = UnknownDAC;
	break;
    }
    return dac;
}

- determineDACType
{
    dac = checkForBrooktreeDAC();
    return self;
}

/* Default gamma precompensation table for color displays.
 * Gamma 2.2 LUT for P22 phosphor displays (Hitachi, NEC, generic VGA) */

static const unsigned char gamma16[] = {
      0,  74, 102, 123, 140, 155, 168, 180,
    192, 202, 212, 221, 230, 239, 247, 255
};

static const unsigned char gamma8[] = {
      0,  15,  22,  27,  31,  35,  39,  42,  45,  47,  50,  52,
     55,  57,  59,  61,  63,  65,  67,  69,  71,  73,  74,  76,
     78,  79,  81,  82,  84,  85,  87,  88,  90,  91,  93,  94,
     95,  97,  98,  99, 100, 102, 103, 104, 105, 107, 108, 109,
    110, 111, 112, 114, 115, 116, 117, 118, 119, 120, 121, 122,
    123, 124, 125, 126, 127, 128, 129, 130, 131, 132, 133, 134,
    135, 136, 137, 138, 139, 140, 141, 141, 142, 143, 144, 145,
    146, 147, 148, 148, 149, 150, 151, 152, 153, 153, 154, 155, 
    156, 157, 158, 158, 159, 160, 161, 162, 162, 163, 164, 165,
    165, 166, 167, 168, 168, 169, 170, 171, 171, 172, 173, 174,
    174, 175, 176, 177, 177, 178, 179, 179, 180, 181, 182, 182,
    183, 184, 184, 185, 186, 186, 187, 188, 188, 189, 190, 190, 
    191, 192, 192, 193, 194, 194, 195, 196, 196, 197, 198, 198,
    199, 200, 200, 201, 201, 202, 203, 203, 204, 205, 205, 206, 
    206, 207, 208, 208, 209, 210, 210, 211, 211, 212, 213, 213,
    214, 214, 215, 216, 216, 217, 217, 218, 218, 219, 220, 220, 
    221, 221, 222, 222, 223, 224, 224, 225, 225, 226, 226, 227,
    228, 228, 229, 229, 230, 230, 231, 231, 232, 233, 233, 234, 
    234, 235, 235, 236, 236, 237, 237, 238, 238, 239, 240, 240,
    241, 241, 242, 242, 243, 243, 244, 244, 245, 245, 246, 246, 
    247, 247, 248, 248, 249, 249, 250, 250, 251, 251, 252, 252,
    253, 253, 254, 255, 
};

static void
SetGammaValue(unsigned int r, unsigned int g, unsigned int b, int level)
{
    outb(PALETTE_DATA, EV_SCALE_BRIGHTNESS(level, r));
    outb(PALETTE_DATA, EV_SCALE_BRIGHTNESS(level, g));
    outb(PALETTE_DATA, EV_SCALE_BRIGHTNESS(level, b));
}

- setGammaTable
{
    unsigned int i, j, g;
    const IODisplayInfo *displayInfo;

    displayInfo = [self displayInfo];

    outb(PALETTE_WRITE, 0x00);

    if (redTransferTable != 0) {
	for (i = 0; i < transferTableCount; i++) {
	    for (j = 0; j < 256/transferTableCount; j++) {
		SetGammaValue(redTransferTable[i], greenTransferTable[i],
			      blueTransferTable[i], brightnessLevel);
	    }
	}
    } else {
	switch (displayInfo->bitsPerPixel) {
	case IO_24BitsPerPixel:
	case IO_8BitsPerPixel:
	    for (g = 0; g < 256; g++) {
		SetGammaValue(gamma8[g], gamma8[g], gamma8[g],
			      brightnessLevel);
	    }
	    break;
	case IO_15BitsPerPixel:
	    for (i = 0; i < 32; i++) {
		for (j = 0; j < 8; j++) {
		    SetGammaValue(gamma16[i/2], gamma16[i/2], gamma16[i/2],
				  brightnessLevel);
		}
	    }
	    break;
	default:
	    break;
	}
    }
    return self;
}

- resetDAC
{
    const QVisionMode *mode;

    mode = [self displayInfo]->parameters;

    if (dac == Bt485 || dac == Bt485A || dac == ATT20C505) {
	/* Brooktree 485 or 485A only! */
	outb(DAC_CMD_0, 0x80);		/* Enable the DAC_EXT_REG. */
	outb(PALETTE_WRITE, 0x01);	/* Make 13C6 be the DAC_EXT_REG. */
	outb(DAC_EXT_REG, 0x00);	/* Turn off clock doubling. */
	outb(PALETTE_WRITE, 0x00);	/* Restore 13C6. */
    }

    /* Restore DAC command registers 0, 1, & 2 to their initial VGA values. */
    outb(DAC_CMD_0, 0x40);
    outb(DAC_CMD_1, 0x00);
    outb(DAC_CMD_2, 0x20);

    return self;
}

- programDAC
{
    const QVisionMode *mode;

    mode = [self displayInfo]->parameters;

    /* Load DAC Command regs. */
    outb(DAC_CMD_0, 0x02);		/* 8-bit DAC. */
    outb(DAC_CMD_1, mode->dacCmd1);	/* Set the bits per pixel. */
    outb(DAC_CMD_2, 0x20);		/* Allow advanced modes. */
	
    if (mode->needsPixelDoubling &&
    	(dac == Bt485 || dac == Bt485A || dac == ATT20C505))
    {
	/* Brooktree 485 or 485A, or AT&T 20C505 only! */
	outb(DAC_CMD_0, 0x82);		/* Enable the DAC_EXT_REG. */
	outb(PALETTE_WRITE, 0x01);	/* Make 13C6 be the DAC_EXT_REG. */
	outb(DAC_EXT_REG, 0x08);	/* Turn on clock doubling. */
    }

    /* Set overscan color (black) */
    outb(CO_COLOR_WRITE, 0x00);
    outb(CO_COLOR_DATA, 0x00);		/* Red component. */
    outb(CO_COLOR_DATA, 0x00);		/* Green component. */
    outb(CO_COLOR_DATA, 0x00);		/* Blue component. */

    return self;
}
@end
