/*
    File:       TestController.m

    Contains:   Main application object to test the resource objects.

    Written by: Quinn "The Eskimo!"

    Created:    Mon 19-May-1997

    Copyright:  (c)1997 by Apple Computer, Inc., all rights reserved.

    Change History (most recent first):

    You may incorporate this sample code into your applications without
    restriction, though the sample code has been provided "AS IS" and the
    responsibility for its operation is 100% yours.  However, what you are
    not permitted to do is to redistribute the source as "DSC Sample Code"
    after having made changes. If you're going to re-distribute the source,
    we require that you make it clear in the source that the code was
    descended from Apple Sample Code, but that you've made changes.
*/

#import "TestController.h"

#import "QResourceFile.h"
#import "QResourceEnumerator.h"
#import "QResourceTypesEnumerator.h"
#import "QResourceObject.h"
#import "QStringList.h"

@implementation TestController

- (void)awakeFromNib
{
    [mainWindow makeKeyAndOrderFront:self];
}

////////////////////////////////////////////////////////////////////////////////
// Some primitive utility routines.

- (NSString*)chooseFileToOpen
    // Prompt the user to select a .rsrc file using the standard
    // open panel.
{
    NSArray *fileTypes;
    NSOpenPanel *openPanel;
    NSString *result = nil;
   
    fileTypes = [NSArray arrayWithObject:@"rsrc"];
    openPanel = [NSOpenPanel openPanel];
    if ( [openPanel runModalForTypes:fileTypes] == NSOKButton) {
        result = [ [openPanel filenames] objectAtIndex:0];
    }
    return (result);
}

- (void)outputString:(NSString *)theString
    // Appends theString to the output text field.  Note that textField is
    // an instance variable declared in our header file that was wired
    // to the real text field in the window via Interface Builder.
{
    [textField replaceCharactersInRange:NSMakeRange(
                        [ [textField string] length ], 0)\
                        withString:theString];
}

////////////////////////////////////////////////////////////////////////////////
// Basic resource file tests.

- (void)outputTypesInResourceFile:(QResourceFile *)resFile
    // Iterate through and print the resource types in the file,
    // using the resource object's description method
    // to create the description of the resource.
{
    QResourceTypesEnumerator *typeEnumerator;
    QResourceObject *thisType;

    [self outputString:@"outputTypesInResourceFile\n"];

    typeEnumerator = [resFile enumerateTypes];
    NSAssert(typeEnumerator != nil, @"Could not get typesEnumerator");

    do {
        thisType = [typeEnumerator nextObject];
        if (thisType != nil) {
            [self outputString:[thisType description]];
            [self outputString:@"\n"];
        }
    } while (thisType != nil);

    [self outputString:@"\n"];
}

- (void)outputResourcesInResourceFile:(QResourceFile *)resFile ofType:(QResourceType)resType
    // Iterate and print all the resources of resType in the
    // resource file, using the resource object's description method
    // to create the description of the resource.
{
    QResourceEnumerator *resourceEnumerator;
    QResourceObject *thisResource;

    [self outputString:@"outputResourcesInResourceFile\n"];

    resourceEnumerator = [resFile enumerateResourcesOfType:resType];
    NSAssert(resourceEnumerator != nil, @"Could not get resourceEnumerator");

    do {
        thisResource = [resourceEnumerator nextObject];
        if (thisResource != nil) {
            [self outputString:[thisResource description]];
            [self outputString:@"\n"];
        }
    } while (thisResource != nil);
    
    [self outputString:@"\n"];
}

- (void)outputStringListsInResourceFile:(QResourceFile *)resFile
    // Iterate through the 'STR#' resources in the file.  For each
    // one, create a QStringList object and iterate through and print
    // all the strings in the list.
{
    QResourceEnumerator *resourceEnumerator;
    QResourceObject *thisResource;
    QStringList *stringList;
    long index;

    [self outputString:@"outputStringListsInResourceFile\n"];

    resourceEnumerator = [resFile enumerateResourcesOfType:'STR#'];
    NSAssert(resourceEnumerator != nil, @"Could not get resourceEnumerator");

    do {
        thisResource = [resourceEnumerator nextObject];
        if (thisResource != nil) {

            // Output the description for the 'STR#' resource itself.
            
            [self outputString:[thisResource description]];
            [self outputString:@"\n"];

            // Output each of the strings is the 'STR#' resource.
            
            stringList = [QStringList stringListWithData:[thisResource resourceData]];
            for (index = 1; index <= [stringList count]; index++ ) {
                [self outputString:@"  "];
                [self outputString:[stringList stringAtIndex:index]];
                [self outputString:@"\n"];
            }
            [self outputString:@"\n"];
        }
    } while (thisResource != nil);

    [self outputString:@"\n"];
}

- (void)doResourceTest:(id)sender
    // Run the 'Resource Manager' test suite.  The basic idea is to
    // ask the user for a resource file, create a QResourceFile object
    // that corresponds to that file, and then run a bunch of test
    // methods on that object which print out the contents of various
    // resources.
{
    NSString *fileToOpen;
    QResourceFile *resFile;

    [self outputString:@"Hello Cruel World!\n"];

    // Ask the user to pick a resource file.
    
    fileToOpen = [self chooseFileToOpen];
    if (fileToOpen != nil) {

        // Open the file.
        
        resFile = [ [QResourceFile alloc] initWithContentsOfFile:fileToOpen ];
        NSAssert(resFile != nil, @"Could not open resource file");

        // Run the tests.
        
        [self outputTypesInResourceFile:resFile];

        [self outputResourcesInResourceFile:resFile ofType:'CODE'];

        [self outputStringListsInResourceFile:resFile];

        // Clean up.
        
        [resFile release];
    }

    [self outputString:@"Success!\n"];
}

////////////////////////////////////////////////////////////////////////////////

- (NSView *)createTextItemAtOffset:(long)offset inData:(NSData *)ditlData fromFile:(QResourceFile *)resFile
                   inRect:(NSRect)itemRect allowEdits:(BOOL)editable
    // Create a view that corresponds to the DITL item at the given offset in the
    // given DITL data.
{
    NSTextField *result = nil;

    result = [[[NSTextField alloc] initWithFrame:itemRect] autorelease];
    if (result != nil) {

        // If the text is editable, make it selectable and editable, and
        // put it on a white background with a border.
        
        [result setSelectable: editable];
        [result setEditable: editable];
        [result setDrawsBackground: editable];
        [result setBordered: editable];

        // Set the contents of the text field.
        
        [result setStringValue:[ditlData MacPStringAtOffset:offset + 13]];
    }
    return (result);
}

- (NSView *)createButtonItemAtOffset:(long)offset inData:(NSData *)ditlData fromFile:(QResourceFile *)resFile
                     inRect:(NSRect)itemRect
    // Create a view that corresponds to the DITL item at the given offset in the
    // given DITL data.
{
    NSButton *result = nil;
    
    result = [[[NSButton alloc] initWithFrame:itemRect] autorelease];
    if (result != nil) {

        // Set the button title and style.
        
        [result setTitle:[ditlData MacPStringAtOffset:offset + 13]];
        [result setBezelStyle:NSPushButtonBezelStyle];
    }
    return (result);
}

- (NSView *)createCheckboxItemAtOffset:(long)offset inData:(NSData *)ditlData fromFile:(QResourceFile *)resFile
                       inRect:(NSRect)itemRect
    // Create a view that corresponds to the DITL item at the given offset in the
    // given DITL data.
{
    NSButton *result = nil;
    
    result = [[[NSButton alloc] initWithFrame:itemRect] autorelease];
    if (result != nil) {

        // Set up the various fields of the button to make it look like a checkbox.
        
        [result setTitle:[ditlData MacPStringAtOffset:offset + 13]];
        [result setImage:[NSImage imageNamed:@"NSSwitch"]];
        [result setAlternateImage:[NSImage imageNamed:@"NSHighlightedSwitch"]];
        [result setImagePosition:NSImageLeft];
        [result setButtonType: NSSwitchButton];
        [result setAlignment: NSNaturalTextAlignment];
    }
    return (result);
}

- (NSView *)createRadioItemAtOffset:(long)offset inData:(NSData *)ditlData fromFile:(QResourceFile *)resFile
                    inRect:(NSRect)itemRect
    // Create a view that corresponds to the DITL item at the given offset in the
    // given DITL data.  Radio buttons are a lot more tricky than the rest of this
    // effort.  Under Mac OS, radio buttons come in sets of one, and the application
    // is responsible for ensuring that they act like radio buttons.  Under OpenStep,
    // radio buttons are defined using NSMatrix and generally come in MxN arrays of
    // buttons, for which the matrix ensures radio button behaviour.  So to create
    // a single radio button, I must first create a prototype NSButtonCell, and then
    // create a matrix with one cell that's derived from this prototype radio button.
    //
    // The big caveat here is that there's no way for me to know which radio buttons
    // in a dialog are grouped together to act as a radio button group, so there's
    // no way I can create an NSMatrix to cover a set of buttons.  Even if I could,
    // the radio buttons in the DITL can be laid out in arbitrary positions, but
    // the OpenStep radio button group (ie an NSMatrix) can only lay out buttons
    // in a regular matrix (ie rows and columns).
{
    NSMatrix *result = nil;
    NSButtonCell *radioButton;

    // Create a radio button to act as the prototype for the matrix of radio buttons.
    
    radioButton = [[[NSButtonCell alloc] initImageCell:[NSImage imageNamed:@"NSRadioButton"]] autorelease];

    if (radioButton != nil) {

        // Set up the prototype button look like a radio button.
        
        [radioButton setTitle:[ditlData MacPStringAtOffset:offset + 13]];
        [radioButton setAlternateImage:[NSImage imageNamed:@"NSHighlightedRadioButton"]];
        [radioButton setImagePosition:NSImageLeft];
        [radioButton setButtonType: NSRadioButton];
        [radioButton setAlignment: NSNaturalTextAlignment];

        // Create a 1x1 matrix using the radio button as the prototype.
        
        result = [[[NSMatrix alloc] initWithFrame:itemRect mode:NSRadioModeMatrix prototype:radioButton
                                    numberOfRows:1 numberOfColumns:1] autorelease];
    }

    return (result);
}

- (NSView *)createPICTItemAtOffset:(long)offset inData:(NSData *)ditlData fromFile:(QResourceFile *)resFile
                   inRect:(NSRect)itemRect
    // Create a view that corresponds to the DITL item at the given offset in the
    // given DITL data.
{
    NSImageView *result = nil;

    #define ExperimentalPICTSupport 0
    #if ExperimentalPICTSupport

        long pictID;
        NSData *theImageData;
        NSImage *theImage;
        NSPICTImageRep *thePictImageRep;

        // This code doesn't work and I still haven't quite figured out why.
        // It could be something as simple as the fact that the PICT support
        // in Yellow is *very* new and doesn't deal with my PICT, or it could
        // be I'm making a gross mistake.  I'll return to this in the future.

        result = [[[NSImageView alloc] initWithFrame:itemRect] autorelease];
        if (result != nil) {
            pictID = [ditlData MacSInt16AtOffset:offset+14];
            theImageData = [[resFile resourceOfType:'PICT'
                                        withID:pictID] resourceData];

            // Do not trust the following lines of code.  They do not work, and
            // I'm not even sure if they should work.
            
            theImage = [[[NSImage alloc] initWithSize:itemRect.size] autorelease];
            thePictImageRep = [NSPICTImageRep imageRepWithData:theImageData];
            [theImage addRepresentation:thePictImageRep];
            [result setImage:theImage];

            // Set the image frame so that we can see the image not appear
            // in it (-:

            [result setImageFrameStyle:NSImageFrameButton];
        }
    #endif

    return (result);
}

- (void)createViewsFromDITL:(NSData *)ditlData ofFile:(QResourceFile *)resFile intoWindow:(NSWindow *)dlogWindow
    // Create the appropriate subclass of NSView for each item in
    // the DITL in ditlData.
{
    long ditlCount;
    long offset;
    long ditlIndex;
    NSRect itemRect;
    long itemKind;
    long itemDataLength;
    BOOL itemDisabled;
    NSView *newItem;

    // Loop through each DITL element.
    
    ditlCount = [ditlData MacUInt16AtOffset:0] + 1;
    offset = 2;
    for (ditlIndex = 1; ditlIndex <= ditlCount; ditlIndex++) {

        // Calculate the itemRect.  Note that this is complicated by the fact
        // that Mac have 0,0 at the top left of the window, and OpenStep (by default,
        // and I don't want to change it for this sample) has 0,0 at the bottom
        // left.  So I use the frame of the window's content view to adjust
        // the co-ordinates appropriately.
        
        itemRect = [ditlData MacRectAtOffset:offset + 4];
        itemRect.origin.y = [[dlogWindow contentView] frame].size.height - itemRect.origin.y - itemRect.size.height;

        // Now calculate the item kind and length.
        
        itemKind = [ditlData MacUInt8AtOffset:offset + 12];
        itemDataLength = [ditlData MacUInt8AtOffset:offset + 13];

        // Work out the item's type and whether it's disabled.  Note that we
        // currently ignore the itemDisabled flag.
        
        itemDisabled = (itemKind & 0x80) != 0;
        itemKind &= 0x7F;

        // Now build the item based on its kind.

        newItem = nil;
        switch (itemKind) {
            case 7: // ctrlItem + resCtrl
                // ignored
                break;
            case 4: // button
                newItem = [self createButtonItemAtOffset:offset inData:ditlData fromFile:resFile
                                        inRect:itemRect];
                break;
            case 5: // check box
                newItem = [self createCheckboxItemAtOffset:offset inData:ditlData fromFile:resFile
                                        inRect:itemRect];
                break;
            case 6: // radio
                newItem = [self createRadioItemAtOffset:offset inData:ditlData fromFile:resFile
                                        inRect:itemRect];
                break;
            case  8: // statText
            case 16: // editText
                newItem = [self createTextItemAtOffset:offset inData:ditlData fromFile:resFile
                                        inRect:itemRect allowEdits:(itemKind == 16)];
                break;
            case 32: // iconItem
                // ignored
                break;
            case 64: // picItem
                newItem = [self createPICTItemAtOffset:offset inData:ditlData fromFile:resFile
                                        inRect:itemRect];
                break;
            case 0: // userItem
                // ignored
                break;
            default:
                // ignored
                break;
        }

        if (newItem != nil) {

            // Add the field to the window.  I'm not sure why, but NSWindow's
            // contentView method returns a generic object (ie type "id"), so
            // we have to cast it to an NSView in order to call addSubview
            // without warnings.

            [((NSView *)[dlogWindow contentView]) addSubview:newItem];
        }

        // Move offset along to the next DITL item.
        
        offset += 14 + itemDataLength + (itemDataLength & 1);
    }
}

- (void)makeNewDialogFromFile:(QResourceFile *)resFile withID:(long)dlogID
    // Create an OpenStep window based on a traditional Mac OS toolbox
    // DLOG resource.  resFile is the resource file that contains the DLOG
    // resource, and dlogID is its ID.
{
    NSData *dlogData;
    NSData *ditlData;
    NSWindow *dlogWindow;
    NSRect windowRect;
    long procID;
    long windowStyle;
    BOOL canClose;
    BOOL initiallyVisible;
    long ditlID;
    NSString *windowTitle;

    // Get the dialog resource out of the file.
    
    dlogData = [ [resFile resourceOfType:'DLOG' withID:dlogID] resourceData];
    NSAssert(dlogData != nil, @"Could not get DLOG resource data");

    // Now read some basic information out of the DLOG resource.
    
    procID           = [dlogData MacSInt16AtOffset:8];
    canClose         = [dlogData MacSInt8AtOffset:12] != 0;
    initiallyVisible = [dlogData MacSInt8AtOffset:10] != 0;
    ditlID           = [dlogData MacSInt16AtOffset:18];
    windowTitle      = [dlogData MacPStringAtOffset:20];

    // Calculate the window style and title based on the DLOG parameters.
    // Note that we always OR in NSTitledWindowMask otherwise the results
    // on the screen look very odd (ie a big grey rectangle with nothing inside).
    
    windowStyle = NSTitledWindowMask;
    switch (procID) {
        case 0: // document
            windowStyle |= NSResizableWindowMask;
            break;
        case 1: // dBoxProc
            windowTitle = @"";
            break;
        case 2: // plainDBox
            windowTitle = @"";
            break;
        case 3: // altDBoxProc
            windowTitle = @"";
            break;
        case 4: // noGrowDocProc
            break;
        case 16: // rDocProc
        case 17: // rDocProc
        case 18: // rDocProc
        case 19: // rDocProc
        case 20: // rDocProc
        case 21: // rDocProc
        case 22: // rDocProc
        case 23: // rDocProc
            break;
        case 2048: // movableModal
            break;
        default:
            break;
    }
    if (canClose) {
        windowStyle |= NSClosableWindowMask;
    }

    // Create the dialog window.  We always set the origin of the window to 100,250
    // rather than use the origin specified in the Mac 'DLOG' resource because
    // OpenStep has 0,0 at the bottom left of the screen, and I haven't yet figured
    // out how to find the boundaries of the screen so that I can correctly compensate
    // for this.

    windowRect = [dlogData MacRectAtOffset:0];
    windowRect.origin = NSMakePoint(100, 250);
    dlogWindow = [[NSWindow alloc] initWithContentRect:windowRect
                                    styleMask:windowStyle backing:NSBackingStoreRetained defer:NO];
    NSAssert(dlogWindow != nil, @"Could not create dialog window\n");

    // Now parse the DITL and create the views associated with each item.
    
    ditlData = [ [resFile resourceOfType:'DITL' withID:ditlID] resourceData];
    NSAssert(ditlData != nil, @"Could not get DITL resource data\n");

    [self createViewsFromDITL:ditlData ofFile:resFile intoWindow:dlogWindow];
    
    [dlogWindow setTitle:windowTitle];

    // For testing purposes it's better to ignore the initially visible
    // flag so that you can actually see all the windows that we create.
    // Otherwise the windows come up offscreen which, while semantically
    // accurate is not a very useful test.
    
    #define IgnoreRealityAndUseConvenientSemantics 1
    #if IgnoreRealityAndUseConvenientSemantics
        [dlogWindow orderFront:self];
    #else
        if ( initiallyVisible ) {
            [dlogWindow orderFront:self];
        }
    #endif
}

- (void)doDialogTest:(id)sender
    // Run the 'Dialog Manager' test suite.  We prompt the user for
    // a resource file, create a QResourceFile object for that file,
    // and then iterate through all the 'DLOG' resources in that file
    // and create a corresponding NSWindow.
{
    NSString *fileToOpen;
    QResourceFile *resFile;
    QResourceEnumerator *resourceEnumerator;
    QResourceObject *thisResource;

    // Ask the user for a resource file.

    fileToOpen = [self chooseFileToOpen];
    if (fileToOpen != nil) {

        // Open the resource file.
        
        resFile = [ [QResourceFile alloc] initWithContentsOfFile:fileToOpen ];
        NSAssert(resFile != nil, @"Could not open resource file");

        // Create an enumerator for all the DLOG resources.

        resourceEnumerator = [resFile enumerateResourcesOfType:'DLOG'];
        NSAssert(resourceEnumerator != nil, @"Could not get resourceEnumerator");

        // Call makeNewDialogFromFile:withID: to create the window from the DLOG template.

        do {
            thisResource = [resourceEnumerator nextObject];
            if (thisResource != nil) {
                [self makeNewDialogFromFile:resFile withID:[thisResource resourceID]];
            }
        } while (thisResource != nil);

        // Clean up.

        [resFile release];

        [self outputString:@"Success!\n"];
    }
}

@end
