/*
    File:       QResourceFile.h

    Contains:   A class that encapsulates a Mac OS resources files and gives access
                to individual resources.

    Written by: Quinn "The Eskimo!"

    Created:    Mon 19-May-1997

    Copyright:  (c)1997 by Apple Computer, Inc., all rights reserved.

    Change History (most recent first):

    You may incorporate this sample code into your applications without
    restriction, though the sample code has been provided "AS IS" and the
    responsibility for its operation is 100% yours.  However, what you are
    not permitted to do is to redistribute the source as "DSC Sample Code"
    after having made changes. If you're going to re-distribute the source,
    we require that you make it clear in the source that the code was
    descended from Apple Sample Code, but that you've made changes.
*/

#import <AppKit/AppKit.h>

#import "QMacDataAccess.h"
#import "QResourceObject.h"

// About Resources
//
// Mac OS defines a way of formatting a plain file to provide a "poor man's database"
// mechanism.  A resource is simply a chunk of arbitrary data.  Each resource has
// a type (a four character code, eg 'TEXT' or 'PICT'), an ID (a short, eg -32768 to
// 32767), (optionally) a name (a Pascal string), and some attributes (a short contain
// flags, eg "system heap").
//
// Resources are typically accessed with the following types of query:
//   o Get me the resource with type 'xxxx' and ID=y.
//   o Get me the resource with type 'xxxx' and name "yyyyyy".
//   o Get me all resources of type 'xxxx'.
//
// Certain resource types are used to store well defined data structures, eg 'PICT',
// 'TEXT', and 'STR#'.  This object does not interpret any substructure.  It merely
// deals with individual resources.
//
// This object does not deal with:
//
//   o Write access to resources -- All resource files are considered
//     read-only.
//   o Resource attributes -- Most of these only make sense in a Mac OS
//     environment.
//   o Compressed resources -- Resources can be compressed for decompression
//     either by a standard schemes, or with a custom decompressor whose
//     68000 code is stored in a resource.  Since the compression mechanism
//     was never formally documented, we don't support it here.  This can
//     be a problem if you try to access a compressed resource from a non-Mac OS
//     platform.
//
// The format of a resource file is documented in "Inside Macintosh: I", p128 and
// "Inside Macintosh: More Macintosh Toolbox", p1-121.
//
// Resource files are normally stored in the "resource forks", the second of
// two forks that each Mac OS file supports.  As Yellow current does not support
// resource forks, this implementation only operates on data forks.
//
// Typically you denote a resource fork using the ".rsrc" designator, although
// this code does not enforce that restriction.

@class QResourceEnumerator, QResourceTypesEnumerator;

// Constants for the various bits in the resource attributes.

enum {
    QResourceFileReadOnlyMask = 128,
    QResourceFileCompactMask = 64,
    QResourceFileChangedMask = 32
};

@interface QResourceFile : NSObject
{
    NSData *resourceData;

    long   attributes;			// Resource file attributes.

    // All offsets and lengths are in bytes from the start of resourceData.
    
    long resourceFileLength;		// Length of resourceData
    
    long resourceDataOffset;		// Offset to start of resource data in resourceData
    long resourceDataLength;		// Length of resource data in resourceData
    long resourceMapOffset;		// Offset of start of resource map in resourceData
    long resourceMapLength;		// Length of resource map in resourceData
    
    long resourceTypeListOffset;	// Offset to start of resource type list in resourceData
    long resourceNameListOffset;	// Offset to start of resource name list in resourceData
}

- (id)initWithContentsOfFile:(NSString*)path;
    // Initialises the recipient, which should be newly allocated QResourceFile object.
    // path should be the path to a resource file.  This routine will throw an exception
    // if the file is not properly formed.  This routine maps the file into memory using VM.
    // You should not modify or delete the file while you still have a QResourceFile
    // object pointing at it.

- (QResourceObject *)resourceOfType:(QResourceType)resType withID:(long)resID;
    // Returns a resource given its type and ID.

- (QResourceTypesEnumerator*)enumerateTypes;
    // Returns an enumerator that will return all the resource types
    // in the resource file.  Note that the enumerator does not retain the
    // resource file, so you should not destroy the resource file until
    // you're done enumerating resource types from it.

- (QResourceEnumerator*)enumerateResourcesOfType:(QResourceType)resType;
    // Returns an enumerator that will return all the resources of
    // a specific type in the resource file.  Note that the enumerator does not
    // retain the resource file, so you should not destroy the resource file until
    // you're done enumerating resources from it.

- (long)attributes;
    // Returns the resource file attributes.

- (void)dealloc;
    // Overridden to allow us to release resourceData.

// The stuff below is all private for the use of the two resource enumerator classes.

- (BOOL)findForType:(QResourceType)resType numResources:(long *)numResourcesOfThisType
            refList:(long *) referenceListOffset;
- (QResourceObject *)resourceGivenType:(QResourceType)resType andReferenceOffset:(long)offset;
- (SInt32)longAtOffset:(SInt32)offset;
- (SInt16)shortAtOffset:(SInt32)offset;
- (long)resourceTypeListOffset;

#if 0

    // Obsolete stuff.

    - (NSData *)resourceData;

    - (long)resourceFileLength;

    - (long)resourceNameListOffset;

    - (long)resourceDataOffset;
    - (long)resourceDataLength;

    - (void)checkBoundsOf:(SInt32)value lower:(SInt32)lowerBound upper:(SInt32)upperBound;

#endif

@end
