// RegexTest_main.m
// MOKit - Yellow Box
// RegexTest
//
// Copyright 1996-1999, Mike Ferris.
// All rights reserved.

// ABOUT MOKit
//
// MOKit is a collection of useful and general objects.  Permission is
// granted by the author to use MOKit in your own programs in any way
// you see fit.  All other rights to the kit are reserved by the author
// including the right to sell these objects as part of a LIBRARY or as
// SOURCE CODE.  In plain English, I wish to retain rights to these
// objects as objects, but allow the use of the objects as pieces in a
// fully functional program.  NO WARRANTY is expressed or implied.  The author
// will under no circumstances be held responsible for ANY consequences to
// you from the use of these objects.  Since you don't have to pay for
// them, and full source is provided, I think this is perfectly fair.

#import <Foundation/Foundation.h>
#import <MOKit/MOKit.h>

#define DEFAULT_DATA_FILE @"RegexTestData.plist"

static NSArray *readTestData() {
    NSArray *args = [[NSProcessInfo processInfo] arguments];
    unsigned argCount = [args count];
    NSString *filePath;
    NSString *fileString;
    NSArray *testCases;

    if (argCount == 1) {
        // By default we look for a file with a standard name in the same directory as the test executable.
        filePath = [[[NSBundle mainBundle] bundlePath] stringByAppendingPathComponent:DEFAULT_DATA_FILE];
    } else if (argCount == 2) {
        filePath = [args objectAtIndex:1];
    } else {
        NSLog(@"usage: %@ [<dataFile>]\n\tYou must supply a file of test cases.", [[NSProcessInfo processInfo] processName]);
        exit(1);
    }

    fileString = [NSString stringWithContentsOfFile:filePath];
    if (!fileString) {
        NSLog(@"%@ Could not load data file '%@'", [[NSProcessInfo processInfo] processName], filePath);
        exit(1);
    }
    testCases = [fileString propertyList];
    if (!testCases || ![testCases isKindOfClass:[NSArray class]]) {
        NSLog(@"%@ Could not parse data file '%@'.  The data file must be an NSArray of test cases in property list format.  Each test case is a sub-array with five string objects that define the test case.", [[NSProcessInfo processInfo] processName], [args objectAtIndex:1]);
        exit(1);
    }
    return testCases;
}

static NSString *substituteSubexpressions(NSString *str, MORegularExpression *regex, NSString *candidate) {
    NSMutableString *result = [NSMutableString string];
    unsigned startIndex, curIndex, stopIndex = [str length];
    unichar curChar;
    NSString *substring;
    
    startIndex = curIndex = 0;
    while (curIndex < stopIndex) {
        curChar = [str characterAtIndex:curIndex];
        if (curChar == '\\') {
            if (curIndex + 1 < stopIndex) {
                curIndex++;
                curChar = [str characterAtIndex:curIndex];
                if ((curChar >= '0') && (curChar < '9')) {
                    // substitute a subexpression.
                    [result appendString:[str substringWithRange:NSMakeRange(startIndex, curIndex - 1 - startIndex)]];
                    startIndex = curIndex + 1;
                    substring = [regex substringForSubexpressionAtIndex:(curChar - '0') inString:candidate];
                    if (substring) {
                        [result appendString:substring];
                    }
                    // References to non-existent subexpressions by \n is allowed in regsub.  Nothing is substituted.
                } else {
                    [result appendString:[str substringWithRange:NSMakeRange(startIndex, curIndex - 1 - startIndex)]];
                    [result appendFormat:@"%c", (char)curChar];
                    startIndex = curIndex + 1;
                }
            }
        } else if (curChar == '&') {
            [result appendString:[str substringWithRange:NSMakeRange(startIndex, curIndex - startIndex)]];
            startIndex = curIndex + 1;
            substring = [regex substringForSubexpressionAtIndex:0 inString:candidate];
            if (substring) {
                [result appendString:substring];
            } else {
                // Substitutions of matched expression (subexp 0) when there's no match is not allowed.  This won't happen given the situation in the caller of this function.
                return nil;
            }
        }
        curIndex++;
    }
    if (startIndex != curIndex) {
        [result appendString:[str substringWithRange:NSMakeRange(startIndex, curIndex - startIndex)]];
    }
    return result;
}

typedef enum {
    TestSucceeded = 0,
    TestDidNotCompile = 1,
    TestDidCompile = 2,  // (but shouldn't have)
    TestDidNotMatch = 3,
    TestDidMatch = 4,  // (but shouldn't have)
    TestSubstitutionFailed = 5,
    TestSubstitutionNotCorrect = 6,
    TestCaseInvalid = 7
} TestResult;

static BOOL executeTestCase(NSArray *testCase) {
    // A test case array has five elements defined as follows:
    //     0: The regular expression.
    //     1: The candidate string.
    //     2: "c" if the expression should not compile, "n" if candidate should not match, "y" if candidate should match.
    //     3: A substitution format string where "&" substitutes the full expression match substring and "\1" - "\9" substitutes subexpression match 1-9.
    //     4: The expected result of substituting the subexpression matches into the substitution string.

    unsigned i;
    MORegularExpression *regex;
    BOOL matches;
    NSString *substString;
    
    // Validate test case
    if (![testCase isKindOfClass:[NSArray class]] || ([testCase count] != 5)) {
        return TestCaseInvalid;
    }
    for (i=0; i<5; i++) {
        if (![[testCase objectAtIndex:i] isKindOfClass:[NSString class]]) {
            return TestCaseInvalid;
        }
    }
    
    // Compile it.
    regex = [MORegularExpression regularExpressionWithString:[testCase objectAtIndex:0]];
    if (regex == NULL) {
        if ([[testCase objectAtIndex:2] isEqualToString:@"c"]) {
            return TestSucceeded;
        } else {
            return TestDidNotCompile;
        }
    } else if ([[testCase objectAtIndex:2] isEqualToString:@"c"]) {
        return TestDidCompile;
    }

    // Test it.
    matches = [regex matchesString:[testCase objectAtIndex:1]];
    if (!matches) {
        if ([[testCase objectAtIndex:2] isEqualToString:@"n"]) {
            return TestSucceeded;
        } else {
            return TestDidNotMatch;
        }
    } else if ([[testCase objectAtIndex:2] isEqualToString:@"n"]) {
        return TestDidMatch;
    }

    // Substitute it.
    // MF: We use subexpressionsForString: even though it's basically obsolete since it is implemented in terms of the newer API and it happens to be just what we want.
    substString = substituteSubexpressions([testCase objectAtIndex:3], regex, [testCase objectAtIndex:1]);
    if (!substString) {
        return TestSubstitutionFailed;
    } else if ([substString isEqualToString:[testCase objectAtIndex:4]]) {
        return TestSucceeded;
    } else {
        return TestSubstitutionNotCorrect;
    }
}

static NSString *testFailureString(TestResult result) {
    switch (result) {
        case TestSucceeded:
            return @"Test succeeded.";
            break;
        case TestDidNotCompile:
            return @"Test expression was expected to compile successfully, but it did not.";
            break;
        case TestDidCompile:
            return @"Test expression was not expected to compile, but it was compiled successfully.";
            break;
        case TestDidNotMatch:
            return @"Test match string was expected to match, but it did not.";
            break;
        case TestDidMatch:
            return @"Test match string was not extpected to match, but it did.";
            break;
        case TestSubstitutionFailed:
            return @"Test subexpression substitution referenced non-existent subexpressions.";
            break;
        case TestSubstitutionNotCorrect:
            return @"Test subexpression substitution did not match expected result.";
            break;
        case TestCaseInvalid:
            return @"Test case is not valid.  It either does not contasin the right number of  elements or some of the elements aren't strings.";
            break;
        default:
            return @"Unknown test result.";
            break;
    }
}

int main (int argc, const char *argv[]) {
    NSAutoreleasePool *pool = [[NSAutoreleasePool alloc] init];
    NSArray *testCases = readTestData();
    unsigned i, c = [testCases count];
    TestResult result;
    BOOL oneFailed = NO;
    
    [testCases retain];

    [pool release];
    
    for (i=0; i<c; i++) {
        pool = [[NSAutoreleasePool alloc] init];

        result = executeTestCase([testCases objectAtIndex:i]);
        if (result != TestSucceeded) {
            oneFailed = YES;
            NSLog(@"Test case %u failed: %@", i, testFailureString(result));
        }
        
        [pool release];
    }

    pool = [[NSAutoreleasePool alloc] init];

    if (!oneFailed) {
        NSLog(@"All tests succeeded.");
    }

    [pool release];

    [testCases release];
    
    exit((oneFailed ? 1 : 0)); 
    return 0;
}
