// MORegularExpression.m
// MOKit - Yellow Box
//
// Copyright 1996-1999, Mike Ferris.
// All rights reserved.

// ABOUT MOKit
//
// MOKit is a collection of useful and general objects.  Permission is
// granted by the author to use MOKit in your own programs in any way
// you see fit.  All other rights to the kit are reserved by the author
// including the right to sell these objects as part of a LIBRARY or as
// SOURCE CODE.  In plain English, I wish to retain rights to these
// objects as objects, but allow the use of the objects as pieces in a
// fully functional program.  NO WARRANTY is expressed or implied.  The author
// will under no circumstances be held responsible for ANY consequences to
// you from the use of these objects.  Since you don't have to pay for
// them, and full source is provided, I think this is perfectly fair.

#import <MOKit/MOKit.h>
#import "MORegularExpression_Private.h"

typedef enum {
    MOInitialVersion = 1,
} MOClassVersion;

static const MOClassVersion MOCurrentClassVersion = MOInitialVersion;

@implementation MORegularExpression

+ (void)initialize {
    if (self == [MORegularExpression class])  {
        [self setVersion:MOCurrentClassVersion];
    }
}

+ (BOOL)validExpressionString:(NSString *)expressionString {
    BOOL isValid = NO;
    _MOregexp *re;

    if (!expressionString || ![expressionString canBeConvertedToEncoding:[NSString defaultCStringEncoding]]) {
        isValid = NO;
    } else {
        re = _MOTestAndCompileExpressionString(expressionString, [self zone]);
        if (re) {
            free(re);
            isValid = YES;
        }
    }
    return isValid;
}

+ (id)regularExpressionWithString:(NSString *)expressionString {
    return [[[[self class] alloc] initWithExpressionString:expressionString] autorelease];
}

- (id)initWithExpressionString:(NSString *)expressionString {
    self = [super init];

    if (self) {
        BOOL isValid = NO;
        
        if (expressionString && [expressionString canBeConvertedToEncoding:[NSString defaultCStringEncoding]]) {
            _compiledExpression = _MOTestAndCompileExpressionString(expressionString, [self zone]);
            if (_compiledExpression) {
                isValid = YES;
            }
        }
        if (!isValid) {
            //NSLog(@"%@: argument '%@' is not a valid regular expression.  Check the syntax and make sure the expression can be converted to the defaultCStringEncoding.  %@ only works with expressions that can be converted to cStrings.", MOFullMethodName(self, _cmd), expressionString, [self class]);
            [self release];
            return nil;
        }

        _expressionString = [expressionString copyWithZone:[self zone]];
    }

    return self;
}

- (id)init {
    return [self initWithExpressionString:@""];
}

- (id)copyWithZone:(NSZone *)zone {
    if (zone == [self zone]) {
        return [self retain];
    } else {
        return [[[self class] allocWithZone:zone] initWithExpressionString:_expressionString];
    }
}

- (void)dealloc {
    [_expressionString release];
    [_lastMatch release];
    [_lastSubexpressionRanges release];
    if (_compiledExpression) {
        free(_compiledExpression);
    }
    [super dealloc];
}

- (BOOL)isEqual:(id)otherObj {
    if (!otherObj) {
        return NO;
    } else if (otherObj == self) {
        return YES;
    } else if ([otherObj isKindOfClass:[MORegularExpression class]]) {
        return [_expressionString isEqualToString:[otherObj expressionString]];
    } else {
        return NO;
    }
}

- (unsigned)hash {
    return [_expressionString hash];
}

- (NSString *)expressionString {
    return _expressionString;
}

- (BOOL)matchesString:(NSString *)candidate {
    BOOL isMatch = NO;
    
    if ([_lastMatch isEqualToString:candidate]) {
        isMatch = YES;
    } else {
        const char *cStr;

        if (![candidate canBeConvertedToEncoding:[NSString defaultCStringEncoding]]) {
            [NSException raise:NSInvalidArgumentException format:@"*** %@: argument '%@' cannot be converted to the defaultCStringEncoding.  %@ only works with candidates that can be converted to cStrings.", MOFullMethodName(self, _cmd), candidate, [self class]];
        }

        isMatch = _MOTestAndMatchCandidateWithExpression(candidate, ((_MOregexp *)_compiledExpression), &cStr);

        if (_lastMatch) {
            // Clear the last results.
            [_lastMatch release];
            _lastMatch = nil;
            [_lastSubexpressionRanges release];
            _lastSubexpressionRanges = nil;
        }
        
        if (isMatch) {
            BOOL haveRealSubexpressions = NO;
            NSMutableArray *tempArray = [[NSMutableArray allocWithZone:[self zone]] init];
            int i;
            NSRange range;
            NSValue *rangeVal;
            
            _lastMatch = [candidate copyWithZone:[self zone]];

            // Extract the subexpression ranges.
            for (i = 0; i < MO_REGEX_NSUBEXP; i++) {
                if (((_MOregexp *)_compiledExpression)->startp[i] != NULL && ((_MOregexp *)_compiledExpression)->endp[i] != NULL)  {
                    range.location = ((_MOregexp *)_compiledExpression)->startp[i] - cStr;
                    range.length = ((_MOregexp *)_compiledExpression)->endp[i] - ((_MOregexp *)_compiledExpression)->startp[i];
                    haveRealSubexpressions = YES;
                } else {
                    range.location = NSNotFound;
                    range.length = 0;
                }
                rangeVal = [[NSValue allocWithZone:[self zone]] initWithBytes:&range objCType:@encode(NSRange)];
                [tempArray addObject:rangeVal];
                [rangeVal release];
            }
            if (haveRealSubexpressions) {
                _lastSubexpressionRanges = [tempArray copyWithZone:[self zone]];
            }
            [tempArray release];
        }
    }
    return isMatch;
}

- (NSRange)rangeForSubexpressionAtIndex:(unsigned)index inString:(NSString *)candidate {
    // matchesString does the hard work.  (And avoids the hard work iff it can.)  So let it do it and we'll just grab the info out of _lastSubexpressionRanges.
    if (index > MO_REGEX_NSUBEXP) {
        [NSException raise:NSInvalidArgumentException format:@"*** %@: index '%u' is greater than the supported number of subexpressions (%d).", MOFullMethodName(self, _cmd), index, MO_REGEX_NSUBEXP];
    }
    if ([self matchesString:candidate] && _lastSubexpressionRanges) {
        NSRange range;
        [[_lastSubexpressionRanges objectAtIndex:index] getValue:&range];
        return range;
    } else {
        return NSMakeRange(NSNotFound, 0);
    }
}

- (NSString *)substringForSubexpressionAtIndex:(unsigned)index inString:(NSString *)candidate {
    NSRange subRange = [self rangeForSubexpressionAtIndex:index inString:candidate];
    if (subRange.location != NSNotFound) {
        return [candidate substringWithRange:subRange];
    } else {
        return nil;
    }
}

- (NSArray *)subexpressionsForString:(NSString *)candidate {
    // This method is provided mainly to maintain compatibility with prior versions.

    if ([self matchesString:candidate]) {
        int i;
        NSMutableArray *tempArray = [NSMutableArray array];
        NSString *substring;

        for (i=0; i<MO_REGEX_NSUBEXP; i++) {
            substring = [self substringForSubexpressionAtIndex:i inString:candidate];
            [tempArray addObject:(substring ? substring : @"")];
        }
        return tempArray;
    } else {
        return nil;
    }
}

- (id)initWithCoder:(NSCoder *)coder {
    unsigned classVersion = [coder versionForClassName:@"MORegularExpression"];

    // Do not call super.  NSObject does not conform to NSCoding.

    if (classVersion > MOCurrentClassVersion)  {
        NSLog(@"%@: class version %u cannot read instances archived with version %u",
              MOFullMethodName(self, _cmd), MOCurrentClassVersion, classVersion);
        [self dealloc];
        return nil;
    }
    if (classVersion == MOInitialVersion) {
        _expressionString = [[coder decodeObject] copyWithZone:[self zone]];
        _lastMatch = nil;
        _lastSubexpressionRanges = nil;
    }
    return self;
}

- (void)encodeWithCoder:(NSCoder *)coder {
    // Do not call super.  NSObject does not conform to NSCoding.
    [coder encodeObject:_expressionString];
}

@end
