/*
 * PuzzleView.m
 * 4/27/98 Mike Trent
 *
 * PuzzleView screen saver module.
 */

#import "PuzzleView.h"
#import <AppKit/psopsNeXT.h>

static NSString *speedStrings[] = {
    @"shh!!",		// dummy value.
    @"Ages 4-6",
    @"Ages 7-9",
    @"Ages 10-12",
    @"Ages 13-15",
    @"Ages 16+",
};

static NSString *sizeStrings[] = {
    @"shh!!",		// dummy value.
    @"50 pieces",
    @"100 pieces",
    @"500 pieces",
    @"1000 pieces",
    @"5000 pieces",
};

static float speeds[] = {
    2.0,		// dummy value.
    2.0,
    1.0,
    0.60,
    0.25,
    0.10,
};

static float sizeBases[] = {
    5.0,		// dummy value.
    5.0,
    8.0,
    12.0,
    20.0,
    50.0,
};

//
// Static functions
//
// These are general purpose functions that would normally live in their
// own .m file, but have moved here so they can be 'static'. 
//

// canned function for taking my screen shot. This breaks in X.
// returns a retained NSImage object.
static NSImage *takeScreenShot (NSRect bounds)
{
    id snapShot;
    NSSize screen;
    NSRect imageRect = NSMakeRect(0,0,0,0);
    NSImage *buffer;

    screen = bounds.size;
    imageRect.size = screen;
    snapShot = [[NSWindow alloc] initWithContentRect:imageRect
                                           styleMask:NSBorderlessWindowMask
                                             backing:NSBackingStoreNonretained
                                               defer:NO];
    PSsetwindowlevel(NSPopUpMenuWindowLevel,[snapShot windowNumber]);
    PSsetautofill(NO,[snapShot windowNumber]);
    [snapShot orderFront:nil];
    buffer = [[NSImage alloc] initWithSize:screen];
    [buffer lockFocus];
    PScomposite(0.0,0.0,screen.width,screen.height,[snapShot gState],
                0.0,0.0,NSCompositeCopy);
    [buffer unlockFocus];
    [snapShot release];

    return buffer;
}

// inverts an image. Currently makes some nasty assumptions about what
// kind of format the image is in: non-planar, 8-bpp, etc.
static void invertImage(NSImage *image)
{
    NSBitmapImageRep *bir;
    int i,cnt;
    unsigned char *bits;
    NSRect bounds;

    // initialize our bitmap image rep.
    bounds.size = [image size];
    bounds.origin = NSMakePoint(0,0);
    [image lockFocus];
    bir = [[NSBitmapImageRep alloc] initWithFocusedViewRect:bounds];
    [image unlockFocus];
    [bir autorelease];

    // invert the bits.
    bits = [bir bitmapData];
    cnt = [bir bytesPerPlane];
    for (i = 0; i < cnt; i++) {
            bits[i] = ~bits[i];
    }

    // set our new representation.
    [image removeRepresentation:[image bestRepresentationForDevice:nil]];
    [image addRepresentation:bir];
}

// prints a message in the upper left corner of our screen.
static void printMessage(NSString *message)
{
    NSRect bigRect = [[NSView focusView] bounds];
    NSRect textRect;
    static NSTextFieldCell *cell = nil;

    if (!cell) {
        cell = [[NSTextFieldCell alloc] initTextCell:@"Hello"];
        [cell setBordered:NO];
        [cell setBezeled:NO];
        [cell setDrawsBackground:NO];
    }
    
    if ([message isKindOfClass:[NSAttributedString class]]) {
        [cell setAttributedStringValue:(NSAttributedString*)message];
    } else {
        [cell setWraps:NO];
        [cell setStringValue:message];
        [cell setFont:[NSFont systemFontOfSize:12.0]];
        [cell setTextColor:[NSColor whiteColor]];
    }
    
    textRect.size = [cell cellSizeForBounds:bigRect];
    textRect.size.width = 600; // ?
    textRect.origin = NSMakePoint(0, bigRect.size.height-textRect.size.height);
    [[NSColor blackColor] set];
    NSRectFill(textRect);
    [cell drawWithFrame:textRect inView:[NSView focusView]];
}

//
// PuzzleView
//
// Here is the view object for our screen saver. This doubles as our user 
// interface controller object as well.
//

@implementation PuzzleView

//
// PuzzleView Screensaver methods
//

- (id)initWithFrame:(NSRect)frameRect
{
    self = [super initWithFrame:frameRect];

    if (self) {
        [self allocateGState];
        srandom(time(0));

        _speed = [[NSUserDefaults standardUserDefaults] integerForKey:@"PuzzleSpeed"];
        if ( (_speed < 1) || (_speed > 5) ) _speed = 3;

        _size = [[NSUserDefaults standardUserDefaults] integerForKey:@"PuzzleSize"];
        if ( (_size < 1) || (_size > 5) ) _size = 3;

        _invertImage = [[NSUserDefaults standardUserDefaults] boolForKey:@"PuzzleInvert"];

    }

    return self;
}

- (void)drawRect:(NSRect)rects
{
    [[NSColor blackColor] set];
    NSRectFill(rects);
    printMessage(@"Getting the pieces from the puzzle box ...");
}

- (BOOL)useBufferedWindow
{
    return YES;
}

- (void)oneStep
{
    [_action oneStep];
    if ([_action expired]) {
        _action = [_action newAction];
    }
}

- (void)willEnterScreenSaverMode
{
    _screen = takeScreenShot([self bounds]);
    if (_invertImage) invertImage(_screen);
}

- (void)enteredScreenSaverMode
{
    if (_action) [_action release];
    [PuzzleAction setSpeed:speeds[_speed]];
    [PuzzleAction setPlaySounds:NO];
    _action = [[PuzzleStartAction alloc] initWithImage:_screen tileWidth:floor([self bounds].size.width / sizeBases[_size])];

    [_screen release];
    _screen = nil;
}

- (void)willExitScreenSaverMode
{
    if (_action) [_action release];
    _action = nil;
}

- (NSView *)inspector:sender
{
    if (!sharedInspectorPanel) {
        [NSBundle loadNibNamed:@"Puzzle.nib" owner:self];
        [sizeTxt setStringValue:sizeStrings[_size]];
        [speedTxt setStringValue:speedStrings[_speed]];
        [invertBtn setState:_invertImage];
        [sizeCtl setIntValue:_size];
        [speedCtl setIntValue:_speed];
    }

    return sharedInspectorPanel;
}

//
// PuzzleView Controller methods
//

- (void)invertSet:(id)sender
{
    _invertImage = [invertBtn state];
    [[NSUserDefaults standardUserDefaults] setObject:(_invertImage?@"YES":@"NO") forKey:@"PuzzleInvert"];
}

- (void)sizeSet:(id)sender
{
    _size = [sizeCtl intValue];
    [sizeTxt setStringValue:sizeStrings[_size]];
    [[NSUserDefaults standardUserDefaults] setObject:[NSString stringWithFormat:@"%d",_size] forKey:@"PuzzleSize"];
}

- (void)speedSet:(id)sender
{
    _speed = [speedCtl intValue];
    [speedTxt setStringValue:speedStrings[_speed]];

    [[NSUserDefaults standardUserDefaults] setObject:[NSString stringWithFormat:@"%d",_speed] forKey:@"PuzzleSpeed"];
}


@end

//
// StaticPuzzleView
//
// This subclass of PuzzleView draws our lovely inspector image.
//

@implementation StaticPuzzleView

- (void)drawRect:(NSRect)rects
{
    // build our fake screen shot.
    NSImage *image, *tmp;
    NSRect r;
    NSSize tSize;
    int i;

    if (!_screen) {
        tmp = [[[NSImage alloc] initWithContentsOfFile:[[NSBundle bundleForClass:[self class]] pathForImageResource:@"screen.tiff"]] autorelease];
        tSize = [tmp size];
        r.size = [self bounds].size;
        r.origin = NSMakePoint(0,tSize.height - r.size.height);
        image = [[[NSImage alloc] initWithSize:r.size] autorelease];
        [image lockFocus];
        [tmp compositeToPoint:NSMakePoint(0,0) fromRect:r operation:NSCompositeCopy];
        [image unlockFocus];

        //build scrambled view.
        [PuzzleAction setSpeed:0.0];
        [PuzzleAction setPlaySounds:NO];
        _action = [[PuzzleStartAction alloc] initWithImage:image tileWidth:floor([self bounds].size.width / 10)];
        _screen = [[NSImage alloc] initWithSize:r.size];
        r.origin = NSMakePoint(0,0);
        [_screen lockFocus];
        [[NSColor blackColor] set];
        NSRectFill(r);
        for (i = 0; i < 50; i++) [self oneStep];
        [_action release];
        [_screen unlockFocus];
    }
    
    // draw our scrambled view
    [_screen compositeToPoint:rects.origin fromRect:rects operation:NSCompositeCopy];
}

@end
