// Copyright 1997-1998 Omni Development, Inc.  All rights reserved.
//
// This software may only be used and reproduced according to the
// terms in the file OmniSourceLicense.html, which should be
// distributed with this project and can also be found at
// http://www.omnigroup.com/DeveloperResources/OmniSourceLicense.html.

#import "OFHeap.h"

#import <Foundation/Foundation.h>
#import <OmniBase/OmniBase.h>

RCS_ID("$Header: /Network/Developer/Source/CVS/OmniGroup/OmniFoundation/DataStructures.subproj/OFHeap.m,v 1.5 1998/12/08 04:08:00 kc Exp $")

// In this case, the userInfo is the selector
static NSComparisonResult OFHeapCompareBySelector(OFHeap *heap, void *userInfo, id object1, id object2)
{
    return (NSComparisonResult)objc_msgSend(object1, (SEL)userInfo, object2);
}
                                                  
@implementation OFHeap

- initWithCapacity: (unsigned int)newCapacity
   compareFunction: (OFHeapComparisonFunction) comparisonFunction
          userInfo: (void *) userInfo;
{
    if (!(self = [super init]))
        return nil;

    _capacity = newCapacity ? newCapacity : 4;
    _size     = 0;
    _objects  = (id *)NSZoneMalloc([self zone], sizeof(_objects) * _capacity);
    
    _comparisonFunction = comparisonFunction;
    _userInfo           = userInfo;
    
    return self;
}

- initWithCapacity: (unsigned int)newCapacity
   compareSelector: (SEL) comparisonSelector;
{
    return [self initWithCapacity: newCapacity
                  compareFunction: OFHeapCompareBySelector
                         userInfo: comparisonSelector];
}

- (unsigned int) count;
{
    return _size;
}

#define SWAP(a, b)		\
do {				\
    id swap;			\
        			\
    swap        = _objects[a];	\
    _objects[a] = _objects[b];	\
    _objects[b] = swap;		\
} while(0)

#define LESSTHAN(a, b)  (_comparisonFunction(self, _userInfo, _objects[a], _objects[b]) == NSOrderedAscending)

#define PARENT(a)     ((a - 1) >> 1)
#define LEFTCHILD(a)  ((a << 1) + 1)
#define RIGHTCHILD(a) ((a << 1) + 2)

- (void)addObject:(id) anObject;
{
    unsigned int                upFrom, upTo;

    if (_size == _capacity) {
        _capacity <<= 1;
        _objects = (id *)NSZoneRealloc([self zone], _objects, sizeof(*_objects) * _capacity);
    }

    _objects[_size] = [anObject retain];

    upFrom = _size;

    while (upFrom) {
	// move the new value up the tree as far as it should go
	upTo = PARENT(upFrom);
	if (LESSTHAN(upFrom, upTo)) {
	    SWAP(upFrom, upTo);
	} else
	    break;
	upFrom = upTo;
    }

    _size++;
}

- (id) removeObject;
{
    unsigned int root, left, right, swapWith;
    id           result;

    if (!_size)
	return nil;

    result = _objects[0];
    _objects[0] = _objects[--_size];
    root = 0;
    while (YES) {
	swapWith = root;
	if ((right = RIGHTCHILD(root)) < _size && LESSTHAN(right, root))
	    swapWith = right;
	if ((left = LEFTCHILD(root)) < _size && LESSTHAN(left, swapWith))
	    swapWith = left;
	if (swapWith == root)
	    break;
	SWAP(root, swapWith);
	root = swapWith;
    }

    return [result autorelease];
}

- (id) removeObjectLessThanObject: (id) object;
{
    if (_comparisonFunction(self, _userInfo, _objects[0], object) == NSOrderedAscending)
	return [self removeObject];
    else
	return nil;
}

- (id) peekObject;
{
    if (_size)
        return _objects[0];
    else
	return nil;
}

- (NSMutableDictionary *) debugDictionary;
{
    NSMutableDictionary *dict;
    unsigned int         i;
    NSMutableArray      *objectDescriptions;

    dict = [super debugDictionary];
    objectDescriptions = [[NSMutableArray alloc] init];

    for (i = 0; i < _size; i++)
        [objectDescriptions addObject: [_objects[i] debugDictionary]];
    [dict setObject: objectDescriptions forKey: @"objects"];
    [objectDescriptions release];
    
    return dict;
}

@end
