// Copyright 1997-1998 Omni Development, Inc.  All rights reserved.
//
// This software may only be used and reproduced according to the
// terms in the file OmniSourceLicense.html, which should be
// distributed with this project and can also be found at
// http://www.omnigroup.com/DeveloperResources/OmniSourceLicense.html.

#import "OFDataCursor.h"

#import <Foundation/Foundation.h>
#import <OmniBase/OmniBase.h>

#import "NSString-OFExtensions.h"
#import "OFByteSet.h"

RCS_ID("$Header: /Network/Developer/Source/CVS/OmniGroup/OmniFoundation/DataStructures.subproj/OFDataCursor.m,v 1.8 1998/12/08 04:07:59 kc Exp $")

@implementation OFDataCursor

static OFByteSet *endOfLineByteSet;

+ (void)initialize;
{
    static BOOL initialized = NO;

    [super initialize];
    if (initialized)
	return;
    initialized = YES;

    endOfLineByteSet = [[OFByteSet alloc] init];
    [endOfLineByteSet addBytesFromString:@"\r\n" encoding:NSASCIIStringEncoding];
}

- initWithData:(NSData *)someData;
{
    if (![super init])
	return nil;

    if (!someData) {
	[self release];
	return nil;
    }

    data = [someData retain];
    byteOrder = NS_UnknownByteOrder;
    stringEncoding = NSISOLatin1StringEncoding;

    dataLength = [data length];
    startPosition = (const byte *)[data bytes];
    endPosition = startPosition + dataLength;
    currentPosition = startPosition;

    return self;
}

- (void)dealloc;
{
    [data release];
    [super dealloc];
}

- (BOOL)hasMoreData;
{
    return currentPosition < endPosition;
}

- (unsigned int)seekToOffset:(int)offset fromPosition:(OFDataCursorSeekPosition)position;
{
    const byte *newPosition;

    switch (position) {
        default:
        case OFDataCursorSeekFromCurrent:
            newPosition = currentPosition + offset;
            break;
        case OFDataCursorSeekFromEnd:
            newPosition = endPosition + offset;
            break;
        case OFDataCursorSeekFromStart:
            newPosition = startPosition + offset;
            break;
    }
    if (newPosition < startPosition)
	[NSException raise:NSRangeException format:@"Attempted seek past start of data"];
    else if (newPosition > endPosition)
	[NSException raise:NSRangeException format:@"Attempted seek past end of data"];
    currentPosition = newPosition;

    return currentPosition - startPosition;
}

- (unsigned int)currentOffset;
{
    return currentPosition - startPosition;
}

- (void)rewind;
{
    currentPosition = startPosition;
}

- (void)setByteOrder:(OFByteOrder)newByteOrder;
{
    byteOrder = newByteOrder;
}

#define ENSURE_ENOUGH_DATA(count)					\
    if (currentPosition + count > endPosition)			        \
	[NSException raise:NSRangeException format:@"Attempted read past end of data"];

#define START_READ_DATA(readValue)					\
    ENSURE_ENOUGH_DATA(sizeof(readValue));				\
    memcpy(&readValue, currentPosition, sizeof(readValue));
    
#define SWAP_BYTES(inputValue, returnValue, readType, swapType)		\
{									\
    switch (byteOrder) {						\
        case NS_UnknownByteOrder:	     				\
            memcpy(&returnValue, &inputValue, sizeof(returnValue));	\
            break;	   						\
        case NS_LittleEndian:						\
            returnValue = NSSwapLittle ## swapType ## ToHost(inputValue); \
            break;     							\
        case NS_BigEndian:     						\
            returnValue = NSSwapBig ## swapType ## ToHost(inputValue);	\
            break;	   						\
    }									\
}

#define INCREMENT_OFFSETS(readType)					\
    currentPosition += sizeof(readType);

#define READ_DATA_OF_TYPE(readType, methodType, swapType)		\
- (readType)read ## methodType;						\
{									\
    OFSwapped ## swapType inputValue;					\
    readType returnValue;						\
									\
    START_READ_DATA(inputValue);	   	    			\
    SWAP_BYTES(inputValue, returnValue, readType, swapType);		\
    INCREMENT_OFFSETS(readType);					\
    return returnValue;							\
}

#define PEEK_DATA_OF_TYPE(readType, methodType, swapType)		\
- (readType)peek ## methodType;						\
{									\
    OFSwapped ## swapType inputValue;					\
    readType returnValue;						\
									\
    START_READ_DATA(inputValue);	   	    			\
    SWAP_BYTES(inputValue, returnValue, readType, swapType);		\
    return returnValue;							\
}

#define SKIP_DATA_OF_TYPE(readType, methodType)				\
- (void)skip ## methodType;						\
{									\
    ENSURE_ENOUGH_DATA(sizeof(readType));				\
    INCREMENT_OFFSETS(readType);					\
}

- (void)readBytes:(unsigned int)byteCount intoBuffer:(void *)buffer;
{
    ENSURE_ENOUGH_DATA(byteCount);
    memcpy(buffer, currentPosition, byteCount);
    currentPosition += byteCount;
}

- (void)peekBytes:(unsigned int)byteCount intoBuffer:(void *)buffer;
{
    ENSURE_ENOUGH_DATA(byteCount);
    memcpy(buffer, currentPosition, byteCount);
}

- (void)skipBytes:(unsigned int)byteCount;
{
    ENSURE_ENOUGH_DATA(byteCount);
    currentPosition += byteCount;
}

- (unsigned int)readMaximumBytes:(unsigned int)byteCount
    intoBuffer:(void *)buffer;
{
    if (currentPosition + byteCount > endPosition)
	byteCount = endPosition - currentPosition;
    memcpy(buffer, currentPosition, byteCount);
    currentPosition += byteCount;
    return byteCount;
}

- (unsigned int)peekMaximumBytes:(unsigned int)byteCount
    intoBuffer:(void *)buffer;
{
    if (currentPosition + byteCount > endPosition)
	byteCount = endPosition - currentPosition;
    memcpy(buffer, currentPosition, byteCount);
    return byteCount;
}

- (unsigned int)skipMaximumBytes:(unsigned int)byteCount;
{
    if (currentPosition + byteCount > endPosition)
	byteCount = endPosition - currentPosition;
    currentPosition += byteCount;
    return byteCount;
}

static inline unsigned int offsetToByte(OFDataCursor *self, byte aByte)
{
    const byte *offset;

    for (offset = self->currentPosition; offset < self->endPosition; offset++)
	if (*(byte *)offset == aByte)
	    break;
    return offset - self->currentPosition;
}

static inline unsigned int
offsetToByteInSet(OFDataCursor *self, OFByteSet *byteSet)
{
    const byte *offset;
    byte aByte;

    for (offset = self->currentPosition;
	 offset < self->endPosition;
	 offset++) {
	aByte = *(byte *)offset;
	if (isByteInByteSet(aByte, byteSet))
	    break;
    }
    return offset - self->currentPosition;
}

- (unsigned int)offsetToByte:(byte)aByte;
{
    return offsetToByte(self, aByte);
}

- (unsigned int)offsetToByteInSet:(OFByteSet *)aByteSet;
{
    return offsetToByteInSet(self, aByteSet);
}

typedef long OFSwappedLong;
typedef short OFSwappedShort;
typedef long long OFSwappedLongLong;
typedef NSSwappedFloat OFSwappedFloat;
typedef NSSwappedDouble OFSwappedDouble;

READ_DATA_OF_TYPE(long int, LongInt, Long);
PEEK_DATA_OF_TYPE(long int, LongInt, Long);
SKIP_DATA_OF_TYPE(long int, LongInt);
READ_DATA_OF_TYPE(short int, ShortInt, Short);
PEEK_DATA_OF_TYPE(short int, ShortInt, Short);
SKIP_DATA_OF_TYPE(short int, ShortInt);
READ_DATA_OF_TYPE(long long int, LongLongInt, LongLong);
PEEK_DATA_OF_TYPE(long long int, LongLongInt, LongLong);
SKIP_DATA_OF_TYPE(long long int, LongLongInt);
READ_DATA_OF_TYPE(float, Float, Float);
PEEK_DATA_OF_TYPE(float, Float, Float);
SKIP_DATA_OF_TYPE(float, Float);
READ_DATA_OF_TYPE(double, Double, Double);
PEEK_DATA_OF_TYPE(double, Double, Double);
SKIP_DATA_OF_TYPE(double, Double);

- (byte)readByte;
{
    ENSURE_ENOUGH_DATA(sizeof(byte));
    return *(byte *)currentPosition++;
}

- (byte)peekByte;
{
    ENSURE_ENOUGH_DATA(sizeof(byte));
    return *(byte *)currentPosition;
}

SKIP_DATA_OF_TYPE(byte, Byte);

- (NSData *)readDataOfLength:(unsigned int)aLength;
{
    NSData *returnData;

    ENSURE_ENOUGH_DATA(aLength);
    returnData = [NSData dataWithBytes:currentPosition length:aLength];
    currentPosition += aLength;
    return returnData;
}

- (NSData *)peekDataOfLength:(unsigned int)aLength;
{
    NSData *returnData;

    ENSURE_ENOUGH_DATA(aLength);
    returnData = [NSData dataWithBytes:currentPosition length:aLength];
    return returnData;
}

- (NSData *)readDataUpToByte:(byte)aByte;
{
    int aLength;
    
    aLength = offsetToByte(self, aByte);
    if (aLength == 0)
	return nil;
    return [self readDataOfLength:aLength];
}

- (NSData *)peekDataUpToByte:(byte)aByte;
{
    int aLength;
    
    aLength = offsetToByte(self, aByte);
    if (aLength == 0)
	return nil;
    return [self peekDataOfLength:aLength];
}

- (NSData *)readDataUpToByteInSet:(OFByteSet *)aByteSet;
{
    unsigned int aLength;
    
    aLength = offsetToByteInSet(self, aByteSet);
    if (aLength == 0)
	return nil;
    return [self readDataOfLength:aLength];
}

- (NSData *)peekDataUpToByteInSet:(OFByteSet *)aByteSet;
{
    unsigned int aLength;
    
    aLength = offsetToByteInSet(self, aByteSet);
    if (aLength == 0)
	return nil;
    return [self peekDataOfLength:aLength];
}

- (NSString *)readStringOfLength:(unsigned int)aLength;
{
    NSData *someData;
    NSString *aString;

    ENSURE_ENOUGH_DATA(aLength);
    someData = [[NSData alloc] initWithBytes:currentPosition length:aLength];
    aString = [NSString stringWithData:someData encoding:stringEncoding];
    [someData release];
    currentPosition += aLength;
    return aString;
}

- (NSString *)peekStringOfLength:(unsigned int)aLength;
{
    NSData *someData;
    NSString *aString;

    ENSURE_ENOUGH_DATA(aLength);
    someData = [[NSData alloc] initWithBytes:currentPosition length:aLength];
    aString = [NSString stringWithData:someData encoding:stringEncoding];
    [someData release];
    return aString;
}

- (NSString *)readStringUpToByte:(byte)aByte;
{
    unsigned int aLength;
    
    aLength = offsetToByte(self, aByte);
    if (aLength == 0)
	return nil;
    return [self readStringOfLength:aLength];
}

- (NSString *)peekStringUpToByte:(byte)aByte;
{
    unsigned int aLength;
    
    aLength = offsetToByte(self, aByte);
    if (aLength == 0)
	return nil;
    return [self peekStringOfLength:aLength];
}

- (NSString *)readStringUpToByteInSet:(OFByteSet *)aByteSet;
{
    unsigned int aLength;
    
    aLength = offsetToByteInSet(self, aByteSet);
    if (aLength == 0)
	return nil;
    return [self readStringOfLength:aLength];
}

- (NSString *)peekStringUpToByteInSet:(OFByteSet *)aByteSet;
{
    unsigned int aLength;
    
    aLength = offsetToByteInSet(self, aByteSet);
    if (aLength == 0)
	return nil;
    return [self peekStringOfLength:aLength];
}

- (NSData *)readAllData;
{
    return [self readDataOfLength:endPosition - currentPosition];
}

- (NSString *)readLine;
{
    unsigned int lineLength;
    NSString *line;

    lineLength = offsetToByteInSet(self, endOfLineByteSet);
    if (lineLength == 0)
	line = @"";
    else
	line = [self readStringOfLength:lineLength];

    if (currentPosition + 1 < endPosition) {
	switch (*(byte *)currentPosition) {
	case '\r':
	    currentPosition++;
	    if (currentPosition + 1 < endPosition &&
		*(byte *)currentPosition == '\n')
		currentPosition++;
	    break;
	case '\n':
	    currentPosition++;
	    break;
	default:
	    break;
	}
    }
    return line;
}

- (void)skipLine;
{
    unsigned int lineLength;

    lineLength = offsetToByteInSet(self, endOfLineByteSet);
    currentPosition += lineLength;

    if (currentPosition + 1 < endPosition) {
        switch (*(byte *)currentPosition) {
            case '\r':
                currentPosition++;
                if (currentPosition + 1 < endPosition &&
                    *(byte *)currentPosition == '\n')
                    currentPosition++;
                    break;
            case '\n':
                currentPosition++;
                break;
            default:
                break;
        }
    }
}

- (NSMutableDictionary *)debugDictionary;
{
    NSMutableDictionary *debugDictionary;

    debugDictionary = [super debugDictionary];

    if (data)
	[debugDictionary setObject:data forKey:@"data"];

    return debugDictionary;
}

@end
