/* ncftpget.c
 *
 * A non-interactive utility to grab files from a remote FTP server.
 * Very useful in shell scripts!
 */

#define VERSION "1.4.0 (Novemeber 6, 1997)"

#ifdef HAVE_CONFIG_H
#	include <config.h>
#endif

#include <unistd.h>
#include <sys/types.h>
#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <stdlib.h>
#include <signal.h>
#include <setjmp.h>

#include <ncftp.h>				/* Library header. */
#include <Strn.h>				/* Library header. */
#include "gpshare.h"

jmp_buf gJmp;
int gGotSig = 0;
int gCanJmp = 0;

extern int gFirewallType;
extern char gFirewallHost[64];
extern char gFirewallUser[32];
extern char gFirewallPass[32];
extern unsigned int gFirewallPort;

extern char *optarg;
extern int optind;

static void
Usage(void)
{
	FILE *fp;
	const char *cp;

	cp = (const char *) getenv("PAGER");
	if (cp == NULL)
		cp = "more";
	fp = popen(cp, "w");
	if (fp == NULL)
		fp = stderr;

	(void) fprintf(fp, "NcFTPGet %s.\n\n", VERSION);
	(void) fprintf(fp, "Usages:\n");
	(void) fprintf(fp, "  ncftpget [flags] remote-host local-dir remote-path-names...   (mode 1)\n");
	(void) fprintf(fp, "  ncftpget -f login.cfg [flags] local-dir remote-path-names...  (mode 2)\n");
	(void) fprintf(fp, "  ncftpget [flags] ftp://url.style.host/path/name               (mode 3)\n");
	(void) fprintf(fp, "\nFlags:\n\
  -u XX  Use username XX instead of anonymous.\n\
  -p XX  Use password XX with the username.\n\
  -P XX  Use port number XX instead of the default FTP service port (21).\n\
  -d XX  Use the file XX for debug logging.\n\
  -a     Use ASCII transfer type instead of binary.\n");
	(void) fprintf(fp, "\
  -t XX  Timeout after XX seconds.\n\
  -v/-V  Do (do not) use progress meters.\n\
  -f XX  Read the file XX for host, user, and password information.\n\
  -A     Append to local files, instead of overwriting them.\n");
	(void) fprintf(fp, "\
  -z/-Z  Do (do not) not try to resume downloads (default: -z).\n\
  -F     Use passive (PASV) data connections.\n\
  -DD    Delete remote file after successfully downloading it.\n\
  -r XX  Redial XX times until connected.\n\
  -R     Recursive mode; copy whole directory trees.\n");
	(void) fprintf(fp, "\nExamples:\n\
  ncftpget ftp.wustl.edu . /pub/README /pub/README.too\n\
  ncftpget ftp.wustl.edu . '/pub/README*'\n\
  ncftpget -R ftp.probe.net /tmp /pub/ncftpd  (ncftpd is a directory)\n\
  ncftpget ftp://ftp.wustl.edu/pub/README\n\
  ncftpget -u gleason -p my.password Bozo.probe.net . '/home/mjg/.*rc'\n\
  ncftpget -u gleason Bozo.probe.net . /home/mjg/foo.txt  (prompt for password)\n\
  ncftpget -f Bozo.cfg '/home/mjg/.*rc'\n\
  ncftpget -a -d /tmp/debug.log -t 60 ftp.wustl.edu . '/pub/README*'\n");

	(void) fprintf(fp, "\nLibrary version: %s.\n", gLibNcFTPVersion + 5);
	(void) fprintf(fp, "\nThis is a freeware program by Mike Gleason (mgleason@probe.net).\n");
	(void) fprintf(fp, "This was built using LibNcFTP (http://www.probe.net/~mgleason/libncftp).\n");

	if (fp != stderr)
		(void) pclose(fp);
	exit(kExitUsage);
}	/* Usage */



static void
Abort(int sigNum)
{
	if (gCanJmp != 0) {
		gCanJmp = 0;
		gGotSig = sigNum;
		longjmp(gJmp, 1);
	}
}	/* Abort */




static int 
Copy(FTPCIPtr cip, char *dstdir, const char ** volatile files, int rflag, int xtype, int resumeflag, int appendflag, int deleteflag)
{
	int i;
	int result;
	const char *file;
	int rc = 0;

	for (i=0; ; i++) {
		file = files[i];
		if (file == NULL)
			break;
		result = FTPGetFiles3(cip, file, dstdir, rflag, kGlobYes, xtype, resumeflag, appendflag, deleteflag, 0);
		if (result != 0) {
			(void) fprintf(stderr, "ncftpget: file retrieval error: %s.\n", FTPStrError(result));
			rc = result;
		}
	}
	return (rc);
}	/* Copy */




int
main(int argc, char **argv)
{
	int result, c;
	volatile int rflag = 0;
	volatile int xtype = kTypeBinary;
	volatile int appendflag = kAppendNo;
	volatile int resumeflag = kResumeYes;
	volatile int deleteflag = kDeleteNo;
	int progmeters;
	FTPLibraryInfo li;
	FTPConnectionInfo fi;
	char * volatile dstdir;
	const char ** volatile flist;
	const char * volatile errstr;
	volatile ExitStatus es;
	char url[256];
	char urlfile[128];
	int urlxtype;
	LineList cdlist;
	LinePtr lp;
	int rc;
	int nD = 0;
#ifdef HAVE_GETPASS
	char *password;
#endif

	result = FTPInitLibrary(&li);
	if (result < 0) {
		(void) fprintf(stderr, "ncftpget: init library error %d (%s).\n", result, FTPStrError(result));
		exit(kExitInitLibraryFailed);
	}
	result = FTPInitConnectionInfo(&li, &fi, kDefaultFTPBufSize);
	if (result < 0) {
		(void) fprintf(stderr, "ncftpget: init connection info error %d (%s).\n", result, FTPStrError(result));
		exit(kExitInitConnInfoFailed);
	}

	fi.debugLog = NULL;
	fi.errLog = stderr;
	fi.xferTimeout = 60 * 60;
	fi.connTimeout = 30;
	fi.ctrlTimeout = 135;
	(void) STRNCPY(fi.user, "anonymous");
	fi.host[0] = '\0';
	progmeters = ((isatty(2) != 0) && (getppid() > 1)) ? 1 : 0;
	urlfile[0] = '\0';
	InitLineList(&cdlist);

	while ((c = getopt(argc, argv, "P:u:p:e:d:t:aRr:vVf:ADzZF")) > 0) switch(c) {
		case 'P':
			fi.port = atoi(optarg);	
			break;
		case 'u':
			(void) STRNCPY(fi.user, optarg);
			break;
		case 'p':
			(void) STRNCPY(fi.pass, optarg);	/* Don't recommend doing this! */
			break;
		case 'e':
			if (strcmp(optarg, "stdout") == 0)
				fi.errLog = stdout;
			else if (optarg[0] == '-')
				fi.errLog = stdout;
			else if (strcmp(optarg, "stderr") == 0)
				fi.errLog = stderr;
			else
				fi.errLog = fopen(optarg, "a");
			break;
		case 'D':
			/* Require two -D's in case they typo. */
			nD++;
			break;
		case 'd':
			if (strcmp(optarg, "stdout") == 0)
				fi.debugLog = stdout;
			else if (optarg[0] == '-')
				fi.debugLog = stdout;
			else if (strcmp(optarg, "stderr") == 0)
				fi.debugLog = stderr;
			else
				fi.debugLog = fopen(optarg, "a");
			break;
		case 't':
			SetTimeouts(&fi, optarg);
			break;
		case 'a':
			xtype = kTypeAscii;
			break;
		case 'r':
			SetRedial(&fi, optarg);
			break;
		case 'R':
			rflag = 1;
			break;
		case 'v':
			progmeters = 1;
			break;
		case 'V':
			progmeters = 0;
			break;
		case 'f':
			ReadConfigFile(optarg, &fi);
			break;
		case 'A':
			appendflag = kAppendYes;
			break;
		case 'z':
			resumeflag = kResumeYes;
			break;
		case 'Z':
			resumeflag = kResumeNo;
			break;
		case 'F':
			if (fi.dataPortMode == kPassiveMode)
				fi.dataPortMode = kSendPortMode;
			else
				fi.dataPortMode = kPassiveMode;
			break;
		default:
			Usage();
	}
	if (optind > argc - 1)
		Usage();

	InitOurDirectory();
	LoadFirewallPrefs();

	if (progmeters != 0)
		fi.progress = PrStatBar;

	if (fi.host[0] == '\0') {
		(void) STRNCPY(url, argv[optind]);
		rc = FTPDecodeURL(&fi, url, &cdlist, urlfile, sizeof(urlfile), (int *) &urlxtype, NULL);
		if (rc == kMalformedURL) {
			(void) fprintf(stderr, "Malformed URL: %s\n", url);
			exit(kExitMalformedURL);
		} else if (rc == kNotURL) {
			/* This is what should happen most of the time. */
			if (optind > argc - 3)
				Usage();
			(void) STRNCPY(fi.host, argv[optind]);
			dstdir = argv[optind + 1];
			flist = (const char ** volatile) argv + optind + 2;
		} else {
			/* URL okay */
			flist = NULL;
			if ((urlfile[0] == '\0') && (rflag == 0)) {
				/* It was obviously a directory, and they didn't say -R. */
				(void) fprintf(stderr, "ncftpget: Use -R if you want the whole directory tree.\n");
				es = kExitUsage;
				exit(es);
			}
			xtype = urlxtype;
		}
	} else {
		if (optind > argc - 2)
			Usage();
		dstdir = argv[optind + 0];
		flist = (const char ** volatile) argv + optind + 1;
	}

#ifdef HAVE_GETPASS
	if (strcmp(fi.user, "anonymous") && strcmp(fi.user, "ftp")) {
		if ((fi.pass[0] == '\0') && (isatty(2) != 0)) {
			password = getpass("Password: ");		
			if (password != NULL) {
				(void) STRNCPY(fi.pass, password);
				/* Don't leave cleartext password in memory. */
				(void) memset(password, 0, strlen(fi.pass));
			}
		}
	}
#endif

	if (MayUseFirewall(fi.host) != 0) {
		fi.firewallType = gFirewallType; 
		(void) STRNCPY(fi.firewallHost, gFirewallHost);
		(void) STRNCPY(fi.firewallUser, gFirewallUser);
		(void) STRNCPY(fi.firewallPass, gFirewallPass);
		fi.firewallPort = gFirewallPort;
	}

	if (nD >= 2)
		deleteflag = kDeleteYes;

	if (setjmp(gJmp) == 0) {
		(void) signal(SIGINT, Abort);
		(void) signal(SIGTERM, Abort);
		(void) signal(SIGALRM, Abort);
		gCanJmp = 1;
		es = kExitOpenTimedOut;
		errstr = "could not open remote host";
		if ((result = FTPOpenHost(&fi)) < 0) {
			(void) fprintf(stderr, "ncftpget: cannot open %s: %s.\n", fi.host, FTPStrError(result));
			es = kExitOpenFailed;
			exit(es);
		}
		if (flist == NULL) {
			/* URL mode */
			errstr = "could not change directory on remote host";
			es = kExitChdirTimedOut;
			for (lp = cdlist.first; lp != NULL; lp = lp->next) {
				if (FTPChdir(&fi, lp->line) != 0) {
					(void) fprintf(stderr, "ncftpget: cannot chdir to %s: %s.\n", lp->line, FTPStrError(fi.errNo));
					es = kExitChdirFailed;
					exit(es);
				}
			}

			errstr = "could not read file from remote host";
			es = kExitXferTimedOut;
			if (FTPGetFiles3(&fi, urlfile, ".", rflag, kGlobYes, xtype, resumeflag, appendflag, deleteflag, 0) < 0) {
				(void) fprintf(stderr, "ncftpget: file retrieval error: %s.\n", FTPStrError(fi.errNo));
				es = kExitXferFailed;
			} else {
				es = kExitSuccess;
			}
		} else {
			errstr = "could not read file from remote host";
			es = kExitXferTimedOut;
			if (Copy(&fi, dstdir, flist, rflag, xtype, resumeflag, appendflag, deleteflag) < 0)
				es = kExitXferFailed;
			else
				es = kExitSuccess;
		}
	} else {
		(void) signal(SIGALRM, SIG_IGN);
		(void) signal(SIGINT, SIG_IGN);
		(void) signal(SIGTERM, SIG_IGN);
		if (gGotSig == SIGALRM) {
			(void) fprintf(stderr, "\nncftpget: %s: timed-out.\n", errstr);
			FTPShutdownHost(&fi);
			exit(es);
		} else {
			(void) fprintf(stderr, "\nncftpget: caught signal, cleaning up...\n");
		}
	}

	if (setjmp(gJmp) == 0) {
		(void) signal(SIGINT, Abort);
		(void) signal(SIGTERM, Abort);
		(void) signal(SIGALRM, Abort);
		gCanJmp = 1;
		errstr = "could not close remote host";
		(void) FTPCloseHost(&fi);
	} else {
		/* couldn't close, but don't change exit status. */
		(void) signal(SIGALRM, SIG_IGN);
		(void) signal(SIGINT, SIG_IGN);
		(void) signal(SIGTERM, SIG_IGN);
		if (gGotSig == SIGALRM) {
			(void) fprintf(stderr, "\nncftpget: %s: timed-out.\n", errstr);
		} else {
			(void) fprintf(stderr, "\nncftpget: caught signal, cleaning up...\n");
		}
		(void) FTPShutdownHost(&fi);
	}
	exit(es);
}	/* main */
