/* main.c
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 * 
 * Copyright (c) 1992, 1993, 1994, 1995, 1996, 1997 by Mike Gleason, NCEMRSoft.
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, without modification,
 * are permitted.
 * 
 */

#include "syshdrs.h"
#include "ls.h"
#include "bookmark.h"
#include "shell.h"
#include "cmds.h"
#include "main.h"
#include "getopt.h"
#include "progress.h"
#include "pref.h"
#include "readln.h"
#include "trace.h"
#include "spool.h"
#include "util.h"

int gStartupUrlParameterGiven = 0;
int gIsTTY;
int gScreenColumns;

FTPLibraryInfo gLib;
FTPConnectionInfo gConn;
LineList gStartupURLCdList;
int gTransferTypeInitialized = 0;
int gTransferType;
int gURLMode = 0;
extern int gUnprocessedJobs;
char gLocalCWD[512], gPrevLocalCWD[512];

extern char *gOptArg;
extern int gOptInd;
extern char gRemoteCWD[512], gPrevRemoteCWD[512];
extern Bookmark gBm;
extern int gLoadedBm;
extern int gFirstTimeUser;
extern int gFirewallType;
extern char gFirewallHost[64];
extern char gFirewallUser[32];
extern char gFirewallPass[32];
extern char gFirewallExceptionList[];
extern char gCopyright[];
extern unsigned int gFirewallPort;
extern int gConnTimeout, gXferTimeout, gCtrlTimeout;
extern int gDataPortMode;
extern int gDebug;
extern int gNumProgramRuns;
extern FTPProgressMeterProc gProgressMeter;

static void
Usage(void)
{
	(void) fprintf(stderr, "Usage:  ncftp [-u user] [<host> | <URL>]\n");
	(void) fprintf(stderr, "\nLibrary version: %s.\n", gLibNcFTPVersion + 5);
#ifdef UNAME
	(void) fprintf(stderr, "System: %s.\n", UNAME);
#endif
	(void) fprintf(stderr, "\nThis is a freeware program by Mike Gleason (mgleason@probe.net).\n");
	exit(2);
}	/* Usage */




/* This resets our state information whenever we are ready to open a new
 * host.
 */
void
InitConnectionInfo(void)
{
	int result;

	result = FTPInitConnectionInfo(&gLib, &gConn, kDefaultFTPBufSize);
	if (result < 0) {
		(void) fprintf(stderr, "ncftp: init connection info error %d (%s).\n", result, FTPStrError(result));
		exit(1);
	}

	gConn.debugLog = NULL;
	gConn.errLog = stderr;
	SetDebug(gDebug);
	UseTrace();
	(void) STRNCPY(gConn.user, "anonymous");
	gConn.host[0] = '\0';
	gConn.progress = gProgressMeter;
	gTransferTypeInitialized = 0;
	gTransferType = kTypeBinary;
	gConn.leavePass = 1;		/* Don't let the lib zap it. */
	gConn.connTimeout = gConnTimeout;
	gConn.xferTimeout = gXferTimeout;
	gConn.ctrlTimeout = gCtrlTimeout;
	gConn.dataPortMode = gDataPortMode;
	gConn.maxDials = (-1);	/* Dial forever, until they hit ^C. */
	gUnprocessedJobs = 0;
	gPrevRemoteCWD[0] = '\0';
}	/* InitConnectionInfo */




/* This lets us do things with our state information just before the
 * host is closed.
 */
void
CloseHost(void)
{
	if (gConn.connected != 0) {
		if (gConn.loggedIn != 0) {
			SaveUnsavedBookmark();
		}
		RunBatchIfNeeded();
	}
	gConn.ctrlTimeout = 3;
	(void) FTPCloseHost(&gConn);
}	/* CloseHost */




/* If the user specified a URL on the command-line, this initializes
 * our state information based upon it.
 */
static void
SetStartupURL(const char *const urlgiven)
{
	int rc;
	char url[256];
	char urlfile[128];
	int xtype;

	gLoadedBm = 0;
	(void) STRNCPY(url, urlgiven);

	rc = FTPDecodeURL(&gConn, url, &gStartupURLCdList, urlfile, sizeof(urlfile), (int *) &xtype, NULL);
	if (rc == kMalformedURL) {
		(void) fprintf(stderr, "Malformed URL: %s\n", url);
		exit(1);
	} else if (rc == kNotURL) {
		/* This is what should happen most of the time. */
		(void) STRNCPY(gConn.host, urlgiven);
		gURLMode = 2;
		if (GetBookmark(gConn.host, &gBm) >= 0) {
			gLoadedBm = 1;
			(void) STRNCPY(gConn.host, gBm.name);
			(void) STRNCPY(gConn.user, gBm.user);
			(void) STRNCPY(gConn.pass, gBm.pass);
			(void) STRNCPY(gConn.acct, gBm.acct);
			gConn.hasSIZE = gBm.hasSIZE;
			gConn.hasMDTM = gBm.hasMDTM;
			gConn.hasPASV = gBm.hasPASV;
			gConn.hasUTIME = gBm.hasUTIME;
			gConn.port = gBm.port;
		} else {
			SetBookmarkDefaults(&gBm);
		}

		if (MayUseFirewall(gConn.host) != 0) {
			gConn.firewallType = gFirewallType; 
			(void) STRNCPY(gConn.firewallHost, gFirewallHost);
			(void) STRNCPY(gConn.firewallUser, gFirewallUser);
			(void) STRNCPY(gConn.firewallPass, gFirewallPass);
			gConn.firewallPort = gFirewallPort;
		}
	} else {
		/* URL okay */
		if (urlfile[0] != '\0') {
			/* It was obviously not a directory */
			(void) fprintf(stderr, "Use ncftpget or ncftpput to handle file URLs.\n");
			exit(1);
		}
		gURLMode = 1;
	}
}	/* SetStartupURL */




static void
OpenURL(void)
{
	LinePtr lp;
	int result;

	if (gURLMode == 1) {
		SetBookmarkDefaults(&gBm);
		if (Open() >= 0) {
			for (lp = gStartupURLCdList.first; lp != NULL; lp = lp->next) {
				result = FTPChdir(&gConn, lp->line);
				if (result != kNoErr) {
					FTPPerror(&gConn, result, kErrCWDFailed, "Could not chdir to", lp->line);
					break;
				}
			}
			result = FTPGetCWD(&gConn, gRemoteCWD, sizeof(gRemoteCWD));
			if (result != kNoErr) {
				FTPPerror(&gConn, result, kErrPWDFailed, NULL, NULL);
			} else {
				(void) printf("Current remote directory is %s.\n", gRemoteCWD);
			}
		}
	} else if (gURLMode == 2) {
		(void) Open();
	}
}	/* OpenURL */




/* These things are done first, before we even parse the command-line
 * options.
 */
static void
PreInit(void)
{
	int result;
	char *cp;

	gIsTTY = ((isatty(2) != 0) && (getppid() > 1)) ? 1 : 0;
	InitUserInfo();
	result = FTPInitLibrary(&gLib);
	if (result < 0) {
		(void) fprintf(stderr, "ncftp: init library error %d (%s).\n", result, FTPStrError(result));
		exit(1);
	}
	srand((unsigned int) getpid());
	InitLineList(&gStartupURLCdList);
	CheckForNewV3User();
	InitPrefs();
	LoadFirewallPrefs();
	LoadPrefs();
	InitConnectionInfo();
	InitCommandList();
	InitLs();
	GetoptReset();

	if ((cp = (char *) getenv("COLUMNS")) == NULL)
		gScreenColumns = 80;
	else
		gScreenColumns = atoi(cp);
}	/* PreInit */





/* These things are done at startup, but after we parse the command-line
 * options.
 */
static void
PostInit(void)
{
	PostInitPrefs();
	OpenTrace();
	InitTermcap();
	InitReadline();
	(void) GetCWD(gLocalCWD, sizeof(gLocalCWD));
	gPrevLocalCWD[0] = '\0';
	PrintStartupBanner();
	if (gNumProgramRuns <= 1)
		(void) printf("\n%s\n", gCopyright + 5);

	Trace(0, "Fw: %s  Type: %d  User: %s  Pass: %s  Port: %u\n", 
		gFirewallHost,
		gFirewallType,
		gFirewallUser,
		(gFirewallPass[0] == '\0') ? "(none)" : "********",
		gFirewallPort
	);
	Trace(0, "FwExceptions: %s\n", gFirewallExceptionList);
}	/* PostInit */




/* Try to get the user to evaluate my commercial offerings. */
static void
Plug(void)
{
	if ((gNumProgramRuns % 7) == 2) {
		(void) printf("\n\n\n\tThank you for using NcFTP.\n\tAsk your system administrator to try NcFTPd Server!\n\thttp://www.ncftp.com\n\n\n\n");
	}
}	/* Plug */




/* These things are just before the program exits. */
static void
PostShell(void)
{
	CloseHost();
	DisposeReadline();
	CloseTrace();
	SavePrefs();
	Plug();
}	/* PostShell */




int
main(int argc, const char **const argv)
{
	int c;

	PreInit();
	while ((c = Getopt(argc, argv, "P:u:p:")) > 0) switch(c) {
		case 'P':
		case 'u':
		case 'p':
			gStartupUrlParameterGiven = 1;
			break;
		default:
			Usage();
	}

	if (gOptInd < argc) {
		SetStartupURL(argv[gOptInd]);
		GetoptReset();

		/* Allow command-line parameters to override
		 * bookmark settings.
		 */
		while ((c = Getopt(argc, argv, "P:u:p:")) > 0) switch(c) {
			case 'P':
				gConn.port = atoi(gOptArg);	
				break;
			case 'u':
				(void) STRNCPY(gConn.user, gOptArg);
				break;
			case 'p':
				(void) STRNCPY(gConn.pass, gOptArg);	/* Don't recommend doing this! */
				break;
		}
	} else if (gStartupUrlParameterGiven != 0) {
		Usage();
	}

	PostInit();
	OpenURL();
	Shell();
	PostShell();
	exit(0);
}	/* main */
