#import "IcoImage.h"
#import <Foundation/NSArray.h>
#import <Foundation/NSString.h>
#import <stdio.h>
#import "ColorSpaceCtrl.h"
#import "common.h"

@implementation IcoImage

/* Read a ICO image file and return images packed in an array
 */
+ (NSArray *)imagesWithContentsOfFile:(NSString *)path
{
	icoHeader *ih;
	const char *fname;
	int	i, num, w, errcode;
	FILE	*fp;
	unsigned char *pp;
	paltype	*pal;
	IcoImage *imgs[MAXIcoImages];
	NSSize	sz;

	if (!path || (fname = [path cString]) == NULL || fname[0] == 0)
		return nil;
	if ((fp = fopen(fname, "r")) == NULL)
		return nil;
	if ((ih = loadIcoHeaders(fp, &num, &errcode)) == NULL) {
		(void)fclose(fp);
		return nil;
	}

	for (i = 0; i < num; ) {
		if (ih[i].x != ih[i].y || (ih[i].x & 0x0f) != 0) {
			--num;
			continue;
		}
		sz.width = sz.height = ih[i].x;
		imgs[i] = [[[self alloc] initWithSize:sz] autorelease];
		w = ih[i].x * ih[i].y;
		pp = imageMap(fp, &ih[i]);
		pal = imgs[i]->palette = ih[i].palette;
		ih[i].palette = NULL;
		imgs[i]->size.width = ih[i].x;
		imgs[i]->size.height = ih[i].y;
		imgs[i]->colors = ih[i].colors;
		imgs[i]->map = pp;
		imgs[i]->alp = pp + w;
		[imgs[i] makeUpIcoRep];
		i++;
	}

	freeIcoHeaders(ih, num);
	(void)fclose(fp);
	return [NSArray arrayWithObjects:imgs count:num];
}

+ (BOOL)saveImages:(NSArray *)images as:(NSString *)path
{
	icoHeader *ih;
	icoHeader ihead[MAXIcoImages];
	const char *fname;
	int	i, num;
	FILE	*fp;
	IcoImage *img;

	if ((num = [images count]) <= 0)
		return NO;
	if (!path || (fname = [path cString]) == NULL || fname[0] == 0)
		return NO;
	if ((fp = fopen(fname, "w")) == NULL)
		return NO;

	for (i = 0; i < num; i++) {
		ih = &ihead[i];
		img = [images objectAtIndex:i];
		ih->x = ih->y = [img size].width;
		ih->colors = img->colors;
		ih->planes = 1;
		ih->bits = (img->colors > 16) ? 8: ((img->colors > 2) ? 4: 1);
		/* ih->size ----- Don't Care */
		/* ih->offset --- Don't Care; image begining position */
		ih->palette = img->palette;
	}
	writeIcoHeaders(fp, ihead, num);

	for (i = 0; i < num; i++) {
		img = [images objectAtIndex:i];
		writeImageMap(fp, &ihead[i], img->map);
	}

	(void)fclose(fp);
	return YES;
}

/* Local Class Method */
+ (Class)loadColorMap
{
	NSBundle *bundle;
	NSString *path;
	static Class theClass = Nil;

	if (theClass)
		return theClass;
	bundle = [NSBundle mainBundle];
	path = [bundle pathForResource:@"Bitmap" ofType:@"bundle"];
	bundle = [NSBundle bundleWithPath: path];
	if (bundle == nil) /* ERROR */
		return Nil;
	theClass = [bundle classNamed:@"ColorMap"];
	return theClass;
}


- (id)init
{
	[super init];
	map = alp = NULL;
	palette = NULL;
	cmap = nil;
	size.width = size.height = 0;
	colors = 0;
	planes[0] = NULL;
	return self;
}

- (void)dealloc
{
	if (palette)
		(void)free((void *)palette);
	if (map)
		(void)free((void *)map);
	[cmap release];
	[super dealloc];
}

- (int)colors { return colors; }

/* Over Write: This method returns pixel-wide size */
- (NSSize)size
{
	int x;
	id rep;

	if (size.width > 0 && size.height > 0)
		return size;

	rep = [self bestRepresentationForDevice:nil];
	size = [super size];
	x = [rep pixelsWide];
	if (x != NSImageRepMatchesDevice && x != size.width) {
		size.width = x;
		size.height = [rep pixelsHigh];
	}
	return size;
}

/* This method is used to get information of images read from
   non-ICO file */
- (commonInfo *)getImageInfo
{
	id	rep;

	rep = [self bestRepresentationForDevice:nil];
	size = [self size];
	if (![rep isKindOfClass:[NSBitmapImageRep class]]) /* EPS or cache */
		return NO;
	info.width = size.width;
	info.height = size.height;
	info.xbytes = [rep bytesPerRow];
	info.bits = [rep bitsPerSample];
	info.pixbits = [rep bitsPerPixel];
	info.numcolors = NSNumberOfColorComponents([rep colorSpaceName]);
	info.alpha = [rep hasAlpha];
	info.cspace = [ColorSpaceCtrl colorSpaceID: [rep colorSpaceName]];
	info.isplanar = [rep isPlanar];

	return &info;
}

/* for non-ICO file */
- (int)numberOfColors
{
	id	rep;
	BOOL	alpha;

	cmap = [[[self class] loadColorMap] alloc];
	if (info.width == 0)
		(void)[self getImageInfo];
	rep = [self bestRepresentationForDevice:nil];
	[rep getBitmapDataPlanes:planes];
	[cmap initWithInfo:&info];
	[cmap allocFullColor];
	return [cmap getAllColor:planes limit:0 alpha:&alpha];
		/* limit=0 ... count all colors even if the number > 256 */
}

/* for non-ICO file */
- (void)makeUpIcoImage:(int)cnum
{
	int	w;

	if (cmap == nil) {
		int	t = [self numberOfColors];
		if (cnum > 0 && cnum <= t)
			cnum = 0;
	}
	w = info.width * info.height;
	if ((map = (unsigned char *)malloc(w + (w >> 3))) == NULL)
		return;	/* ERROR */
	alp = map + w;

	if (cnum == 0) { /* without color reduction */
		(void)[cmap getNormalmap:&cnum];
		colors = (cnum <= 16) ? 16: 256;
	}else { /* Color Reduction */
		colors = cnum;
		(void)[cmap getReducedMap:&cnum alpha:NO];
	}
	[cmap makeIcoIndex:map andAlpha:alp size:info.width from:planes];
	palette = [cmap getPalette];
	[cmap release];
	cmap = nil;

	[self makeUpIcoRep];
}

- (void)makeUpIcoRep
{
	unsigned char	*pp, *ap;
	int	x, y, k, i, cc, idx, w;
	id	rep, arr;
	unsigned char *bitmap[MAPPLANES];

	w = size.width * size.height;
	pp = (unsigned char *)malloc(w * MAPPLANES);
	for (k = 0; k < MAPPLANES; k++)
		bitmap[k] = pp + k * w;
	pp = map;
	for (y = size.height - 1; y >= 0; y--) {
		idx = size.width * y;
		for (x = 0; x < size.width; x++, idx++) {
			unsigned char *palp;
			palp = (unsigned char *)palette[*pp++];
			for (k = 0; k < 3; k++)
				bitmap[k][idx] = palp[k];
		}
	}
	ap = alp;
	for (y = size.height - 1; y >= 0; y--) {
		idx = size.width * y;
		for (x = 0; x < size.width; x += 8) {
			cc = *ap++;
			for (k = 0x80; k; k >>= 1)
				bitmap[3][idx++] = (cc & k)
					? AlphaTransp: AlphaOpaque;
		}
	}
	rep = [[NSBitmapImageRep alloc] initWithBitmapDataPlanes:bitmap
		pixelsWide:size.width pixelsHigh:size.height
		bitsPerSample:8 samplesPerPixel:4 hasAlpha:YES
		isPlanar:YES colorSpaceName:NSCalibratedRGBColorSpace
		bytesPerRow:size.width bitsPerPixel:8];
	if (rep == nil)
		return;	/* Error */

	if ((arr = [self representations]) != nil) {
		k = [arr count];
		for (i = 0; i < k; i++)
			[self removeRepresentation:[arr objectAtIndex:i]];
	}
	[self addRepresentation: rep];
	[rep release];	// rep is retain'ed
}

@end
