/*
 SoundEffect.m, class to play sounds
 Originally by Terry Donahue
 Modified by Ali Ozer to manage multiple sounds

 SoundEffect is a class which conveniently groups the sound stream
 functionality with sound data using the Sound class.

 Copyright (c) 1991-1997 by Apple Computer, Inc., all rights reserved.

 You may incorporate this sample code into your applications without
 restriction, though the sample code has been provided "AS IS" and the
 responsibility for its operation is 100% yours.  However, what you are
 not permitted to do is to redistribute the source as "DSC Sample Code"
 after having made changes. If you're going to re-distribute the source,
 we require that you make it clear in the source that the code was
 descended from Apple Sample Code, but that you've made changes.
*/

#ifdef WIN32
#define SOUNDS_NOT_WORKING 1
#else
#define SOUNDS_NOT_WORKING 0
#endif

#if SOUNDS_NOT_WORKING

/* This is the disabled SoundEffect class... */

#import "SoundEffect.h"

@implementation SoundEffect

- (id)initWithSoundResource:(NSString *)sound {
    [self release];
    return nil;
}

- (void)play {}
- (void)play:(float)volume pan:(float)rads {}

+ (void)setSoundEnabled:(BOOL)flag {}
+ (BOOL)soundEnabled { return NO; }

+ (void)setMaxSoundStreams:(unsigned int)max {}
+ (unsigned int)maxSoundStreams {return 0;}

@end

#else

/* And this is the SoundEffect class which is supposed to work someday... */

#import "SoundEffect.h"
#import <SoundKit/SoundKit.h>
#import <SoundKit/NXPlayStream.h>
#import <AppKit/AppKit.h>

@interface SoundEffect(SoundEffectInternalMethods)
// Internal methods.
+ (NXPlayStream *)soundStream;
+ (void)releaseSoundStream:(NXPlayStream *)soundStream;
@end

@implementation SoundEffect

static BOOL soundEnabled = NO;

#define DEFAULTMAXSOUNDSTREAMS 20

static NSMutableArray *soundStreams = nil;		// List of currently idle sound streams
static unsigned int soundStreamsAllocated = 0;	// Total number of sound streams allocated
static unsigned int maxSoundStreams = DEFAULTMAXSOUNDSTREAMS;	// Max allowed

// After calling this, you may call soundEnabled to check to see if it was successful.

+ (void)setSoundEnabled:(BOOL)flag {
    if (flag && !soundEnabled) {
	NXPlayStream *testStream = [self soundStream];
	if (testStream) {
	    soundEnabled = YES;	    
	    [self releaseSoundStream:testStream];
	} else {
	    NSLog(@"Can't enable sounds.");
	}
    } else if (!flag && soundEnabled) {
	soundEnabled = flag;
	soundStreamsAllocated -= [soundStreams count];
	[soundStreams removeAllObjects];
    }
}

+ (BOOL)soundEnabled {
    return soundEnabled;
}

// These two methods let the client set/get the maximum number of
// sound streams to allocate. If this number is exceeded, sound requests
// are simply not honored until sound streams are freed up.

+ (void)setMaxSoundStreams:(unsigned int)max {
    maxSoundStreams = max;
}

+ (unsigned int)maxSoundStreams {
    return maxSoundStreams;
}

// This method returns a sound stream to be used in playing a sound.
// Sound streams allocated through this method should be given back
// via releaseSoundStream:. Note that this is for internal use only;
// it however might be overridden if necessary.

+ (NXPlayStream *)soundStream {
    static BOOL cantPlaySounds = NO;
    static NXSoundOut *dev = nil;			// We only have one instance of this...
    NXPlayStream *newStream = nil;

    if (cantPlaySounds) return nil;	// If we've tried before and failed, just give up.
    
    if (!dev && !(dev = [[NXSoundOut alloc] init])) {	// We allocate this from the default zone so that
	NSLog(@"Couldn't create NXSoundOut");	//  freeing this zone won't accidentally blast it
	cantPlaySounds = YES;
        return nil;
    }

    if (!soundStreams) {
	soundStreams = [[NSMutableArray alloc] init];
    }

    if (![soundStreams count]) {
	if (soundStreamsAllocated < maxSoundStreams) {
	    newStream = [[NXPlayStream alloc] initOnDevice:dev];
	    soundStreamsAllocated++;
	}
    } else {
        newStream = [soundStreams lastObject];
	[soundStreams removeLastObject];
    }
    
    if (newStream) {
	if (![newStream isActive] && ([newStream activate] != NX_SoundDeviceErrorNone)) {
	    [newStream release];
	    newStream = nil;
	    soundStreamsAllocated--;
	}
    }

    return newStream;
}

// When a sound stream is released, put it on the idle list unless sounds were disabled;
// then just free it.

+ (void)releaseSoundStream:(NXPlayStream *)soundStream {
    if ([self soundEnabled]) {
	[soundStreams addObject:soundStream];
    } else {
	[soundStream release];	// This also deactivates.
	soundStreamsAllocated--;
    }
}

// This method lets you create new instances of SoundEffect. If the specified
// sound file does not exist, the allocated instance is freed and nil is returned.

- (id)initWithSoundResource:(NSString *)path {
    [super init];

    if (!(sound = [[Sound allocWithZone:[self zone]] initFromSection:path])) {
	NSLog(@"Couldn't load sound from %s", [path cString]);
	[self release];
	return nil;
    }

    return self;
}

// Free frees the SoundEffect. If this sound effect is being played at the time,
// the free is delayed and happens as soon as all pending sounds are finished.

- (void)dealloc {
    if (flags.refCount) {
	flags.freeWhenDone = YES;
    } else {
	if (sound) [sound release];
	[super dealloc];
    }
}

// These two methods play the sound effect.

- (void)play {
    [self play:1.0 pan:0.0];
}

- (void)play:(float)volume pan:(float)pan {
    float left, right;
    NXPlayStream *soundStream;
    
    if (![[self class] soundEnabled]) return;

    if (!(soundStream = [[self class] soundStream])) return;
    
    [soundStream setDelegate:self];

    left = right = volume;
    if (pan > 0.0) left  *= 1.0 - pan;
    else if (pan < 0.0) right *= 1.0 + pan;
    [soundStream setGainLeft:left right:right];
    if ([soundStream playBuffer:(void *)[sound data]
			    size:(unsigned int)[sound dataSize]
			    tag:0
		    channelCount:(unsigned int)[sound channelCount]
		    samplingRate:[sound samplingRate]] == NX_SoundDeviceErrorNone) {
	flags.refCount++;
    } else {
	[[self class] releaseSoundStream:soundStream];
    }
}

// Delegate methods for internal use only.

- (void)soundStream:(id)sender didCompleteBuffer:(int)tag {
    flags.refCount--;
    [[self class] releaseSoundStream:sender];
    if (flags.freeWhenDone && flags.refCount == 0) {
	[self release];
    }
}

- (void)soundStreamDidAbort:(id)sender deviceReserved:(BOOL)flag {
    [self soundStream:sender didCompleteBuffer:0];
}

@end


#endif 
