/* AppController.m
 *
 * You may freely copy, distribute and reuse the code in this example.
 * Apple Computer, Inc. disclaims any warranty of any kind, expressed or implied, as to
 * its fitness for any particular use.
 */

#import "AppController.h"
#import "AboutPanel.h"
#import "DocController.h"
#import "Inspector.h"
#import "Logger.h"

@implementation AppController

- (id)init
{
    self = [super init];
    documents = [[NSMutableArray alloc] init];
    return self;
}

- (void)dealloc
{
    [aboutPanel release];
    [documents release];
    [super dealloc];
}

- (void)registerDocument:(id)doc
{
    [documents addObject:doc];
}

- (void)unregisterDocument:(id)doc
{
    [documents removeObject:doc];
}

- (id)prefController
{
       return prefController;
}

- (id)logger
{
       return logger;
}

- (void)newDocument:(id)sender
{
    // create and register a new doc
    id	doc = [[DocController alloc] init];
    [self registerDocument:doc];
    [doc release];
}

// This method opens a document stored in a file. It obtains the path of the
// document from DocController using a class method. We chose to keep all
// the file handling code within DocController, thus keeping the AppController
// more general. An alternative would be to put all the document open code
// into AppController, or put all the uniquing logic into the DocController.

- (BOOL)openFile:(NSString *)path
{
    DocController *doc;
    NSEnumerator *e = [documents objectEnumerator];

    // Enforce uniquing here
    while (doc = [e nextObject])
    {
        if ([path isEqualToString:[doc filename]]) break;
    }

    // For a new document, we have to register it
    if (!doc)
    {
        doc = [[DocController alloc] initFromFile:path];
        if (doc) [self registerDocument:doc];
    }

    if (doc)
    {
        [doc show:self];
        return YES;
    }
    return NO;
}

- (void)openDocument:(id)sender
{
    NSString *filename;

    if ((filename = [DocController filenameFromOpen]))
    {
        [self openFile:filename];
    }
}

- (void)saveAll:(id)sender
{
    DocController *doc;
    NSEnumerator *e = [documents objectEnumerator];

    while (doc = [e nextObject])
    {
        [doc save:self];
    }
}

- (void)showAboutPanel:(id)sender
{
    if (!aboutPanel)
    {
        // find the bundle in the mainBundle
        Class class;
        NSBundle *bundle;
        NSBundle *mainBundle = [NSBundle mainBundle];

        bundle = [NSBundle bundleWithPath:
            [mainBundle pathForResource:@"AboutPanel" ofType:@"bundle"]];

        // alloc/init and show:
        if (class = [bundle principalClass])
        {
            aboutPanel = [[class alloc] init];
        }
        else
        {
            NSLog(@"AppController: problem loading AboutPanel");
            return;
        }
    }

    [aboutPanel show: nil];
}

- (void)showInspector:(id)sender
{
    if (!inspector)
    {
        inspector = [[Inspector alloc] init];
    }

    [inspector show: nil];
}

- (void)showLogger:(id)sender
{
    if (!logger)
    {
        logger = [[Logger alloc] init];
    }

    [logger show:self];
}

- (void)applicationDidFinishLaunching:(NSNotification *)aNotification
{
    // Do we show the inspector on startup or not ?
    NSUserDefaults *defaults = [NSUserDefaults standardUserDefaults];
    if ([defaults boolForKey:@"showInspector"])
    {
    [self showInspector:self];
    }

    // Ensure there is at least one doc window open to start with
    if ([documents count] == 0)
    {
        [self newDocument:self];
    }
}

- (BOOL)applicationShouldTerminate:(id)sender
{
    NSEnumerator *enumerator = [documents objectEnumerator];
    NSMutableArray *modified = [NSMutableArray array];
    id doc;

    while (doc = [enumerator nextObject])
    {
        if ([doc isDocumentEdited]) [modified addObject:doc];
    }

    if ([modified count] > 0)
    {
        int ret;
        ret = NSRunAlertPanel(@"Quit",
                              @"There are modified documents",
                              @"Review Unsaved",
                              @"Quit Anyway",
                              @"Cancel"
                              );
        switch (ret)
        {
            case NSAlertDefaultReturn:
                enumerator = [modified objectEnumerator];
                while (doc = [enumerator nextObject])
                {
                    [doc close:self];
                    // if the doc was not unregistered, we should cancel
                    if ([documents indexOfObject:doc] != NSNotFound)
                    {
                        return NO;
                    }
                }
                break;
            case NSAlertAlternateReturn:
                break;
            case NSAlertOtherReturn:
                return NO;
        }
    }

    return YES;
}

- (BOOL)validateMenuItem:(NSMenuItem *)aMenuItem
{
    // Here we are being slightly pedantic ...
    if ([aMenuItem action] == @selector(saveAll:))
    {
        NSEnumerator *enumerator = [documents objectEnumerator];
        NSMutableArray *modified = [NSMutableArray array];
        id doc;

        while (doc = [enumerator nextObject])
        {
            if ([doc isDocumentEdited]) [modified addObject:doc];
        }

        if ([modified count] == 0) return NO;
    }
    return YES;
}

@end
