/* DocController.m
 *
 * You may freely copy, distribute and reuse the code in this example.
 * Apple Computer, Inc. disclaims any warranty of any kind, expressed or implied, as to
 * its fitness for any particular use.
 */

#import "DocController.h"
#import "AppController.h"

static NSString *expenseFileType = @"expense";

@implementation DocController

// This is a private method to offset the new window from the previous one
- (void)_staggerWindow
{
    static NSPoint nextTopLeft = { 0, 0 };

    if (nextTopLeft.x == 0 && nextTopLeft.y == 0)
    {
        // First time around use the default frame from the nib
        NSRect frame = [window frame];
        frame.origin.y += frame.size.height;
        nextTopLeft = frame.origin;
    }

    [window setFrameTopLeftPoint:nextTopLeft];
    nextTopLeft = [window cascadeTopLeftFromPoint:nextTopLeft];
}

+ (NSString *)filenameFromOpen
{
    NSOpenPanel *p = [NSOpenPanel openPanel];
    NSArray *types = [NSArray arrayWithObject:expenseFileType];

    if ([p runModalForTypes:types] == NSOKButton)
    {
        return [p filename];
    }
    return nil;
}

// This is our class's designated initialiser
- (id)initFromFile:(NSString *)file
{
   NSString *nib = @"Document";

   self = [super init];
   if ([NSBundle loadNibNamed:nib owner:self])
   {
       [self _staggerWindow];
   }
   else
   {
       NSLog(@"DocController: unable to load %@.nib", nib);
       [self release];
       return nil;
   }

   if (file)
   {
       // what about file I/O errors?
       [dataSource setArray: [NSUnarchiver unarchiveObjectWithFile:file]];
       [self setFilename:file];
   }
   else
   {
       static int number = 0;
       [self setTitle:[NSString stringWithFormat:@"%@%d", @"UNTITLED", number++]];
   }
   [self show:self];
   return self;
}

// We override init to call the designated initialiser
- (id)init
{
   return [self initFromFile: nil];
}

- (void)dealloc
{
    // else the window might try asking us something
    [window setDelegate:nil];
    [dataSource release];
    [super dealloc];
}

- (NSString *)title
{
    return [window title];
}

- (void)setTitle:(NSString *)newTitle
{
    [window setTitle:newTitle];
}

- (NSString *)filename
{
    return [window representedFilename];
}

- (void)setFilename:(NSString *)name
{
    [window setTitleWithRepresentedFilename: name];
}

- (BOOL)isNew
{
    return [[self filename] isEqual: @""];
}

- (BOOL)isDocumentEdited
{
    return [window isDocumentEdited];
}

- (void)setDocumentEdited:(BOOL)isEdited
{
    [window setDocumentEdited:isEdited];
}

- (BOOL)windowShouldClose:(id)sender
{
    // We should check for document modifications here, prior to
    // closing the document
    if ([self isDocumentEdited])
    {
        int ret;
        [window makeKeyAndOrderFront: nil];
        ret = NSRunAlertPanel(@"Close",
                              @"Save changes to %@?",
                              @"Save",
                              @"Don't Save",
                              @"Cancel",
                              [self filename] 
                              );
        switch(ret)
        {
            case NSAlertDefaultReturn:
                [self save:self];
                break;
            case NSAlertAlternateReturn:
                break;
            case NSAlertOtherReturn:
                return NO;
        }
    }

    // proceed with closure
    [[NSApp delegate] unregisterDocument:self];
    return YES;
}

- (void)show:(id)sender
{
    [window makeKeyAndOrderFront:self];
}

- (void)saveToFile:(NSString *)filename
{
    // what about file I/O errors?
    [NSArchiver archiveRootObject:[dataSource array] toFile:filename];
    [self setDocumentEdited:NO];
}

- (void)saveAs:(id)sender;
{
    NSSavePanel *p = [NSSavePanel savePanel];
    NSString *filename;
    int ret;

    [p setRequiredFileType:expenseFileType];

    if ([self isNew])
    {
        ret = [p runModal];
    }
    else
    {
        ret = [p runModalForDirectory:[self title] file:nil];
    }

    switch (ret)
    {
        case NSOKButton:
            filename = [p filename];
            [self setFilename:filename];
            [self saveToFile:filename];
            break;
        case NSCancelButton:
            break;
    }
}

- (void)save:(id)sender
{
    // Does the document have a true name yet ?
    if ([self isNew]) // always save
        [self saveAs:self];
    else
        [self saveToFile:[self filename]];
}

- (void)revert:(id)sender
{
    // Gotcha: can't revert if not yet saved (untitled document)
    if ([self isDocumentEdited])
    {
        if ([self isNew])
        {
            NSRunAlertPanel(@"Revert",
                            @"Cannot revert - Document has no saved version",
                            @"OK",
                            nil,
                            nil
                            );
        }
        else
        {
            int ret;
            ret = NSRunAlertPanel(@"Revert",
                                    @"Revert to last saved version of %@ (changes will be lost)?",
                                    @"Revert",
                                    @"Cancel",
                                    nil,
                                    [self filename]
                                    );
            switch(ret)
            {
                case NSAlertDefaultReturn:
                    // code duplication here
                    [dataSource setArray:[NSUnarchiver unarchiveObjectWithFile:[self filename]]];
                    [self setDocumentEdited:NO];
                    break;
                case NSAlertAlternateReturn:
                    break;
                case NSAlertOtherReturn:
                    break;
            }
        }
    }
}

- (void)close:(id)sender
{
    [window performClose:sender];
}

- (BOOL)validateMenuItem:(NSMenuItem *)aMenuItem
{
    // This could be done with strings, and not selectors, but then there
    // is more coupling between the main menu and us
    if ([aMenuItem action] == @selector(revert:))
    {
        if (![self isDocumentEdited] || [self isNew]) return NO;
    }
    return YES;
}

@end
