/* testpdf.c -- test program for the ClibPDF library
		A relatively comprehensive test program.
 * Copyright (C) 1998 FastIO Systems, All Rights Reserved.
 * For conditions of use, license, and distribution, see LICENSE.txt or LICENSE.pdf.

NEXTSTEP/OPENSTEP
cc -Wall -o testpdf testpdf.c -lcpdf

BSDI/gcc
gcc -Wall -o testpdf testpdf.c -L. -lcpdf -lm

SunOS5.4/CC-4.0
cc -o testpdf testpdf.c -Xt -L. -lcpdf -lm

*/

#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <string.h>

#include "cpdflib.h"

/* ======= Month Name Arrays for use with cpdf_setMonthNames() =======
   Use MacRomanEncoding table for character codes to define month names.
   See PDF Reference Manual version 1.2, Adobe Systems, Inc.
   http://www.adobe.com/supportservice/devrelations/PDFS/TN/PDFSPEC.PDF
   Appendix C: Predefined Font Encodings.
   Here are some examples.
*/
char *monthNamesFrench[] = {
	"Jan", "F\216v", "Mar", "Avr", "Mai", "Jun",
	"Jul", "Ao\236", "Sep", "Oct", "Nov", "D\216c"
};

char *monthNamesGerman[] = {
	"Jan", "Feb", "M\212r", "Apr", "Mai", "Jun",
	"Jul", "Aug", "Sep", "Okt", "Nov", "Dez"
};

char *monthNamesItalian[] = {
	"Gen", "Feb", "Mar", "Apr", "Mag", "Giu",
	"Lug", "Ago", "Set", "Ott", "Nov", "Dic"
};

char *monthNamesSpanish[] = {
	"Ene", "Feb", "Mar", "Abr", "May", "Jun",
	"Jul", "Ago", "Sep", "Oct", "Nov", "Dic"
};

char *monthNamesSwedish[] = {
	"Jan", "Feb", "Mar", "Apr", "Maj", "Jun",
	"Jul", "Aug", "Sep", "Okt", "Nov", "Dic"
};


char *myAnnotation =  "testpdf.c -- FastIO Systems\n\
Generated by ClibPDF library in C\n\
Sun Oct 25 14:04:42 PST 1998\n";

/* Some plot data. In a real program, these come from external sources. */
/* --------------------------------------------------------------------------- */
int Nxsf = 12;
float xsf[] = {0.05, 0.072547, 0.105262, 0.152729, 0.221600, 0.321529, 0.466520,
		0.676894, 0.982134, 1.425020, 2.067622, 3.0};
float Resp[] = { 1.500, 4.375, 7.312, 8.938, 25.938, 36.125, 59.125,
		67.062, 31.750, 2.500, 1.562, 3.375 };
float nullresp = 1.750;
float respmax = 80.0;
float  xsfmax = 4.0;
float  xsfmin = 0.03;
/* Fit parameter definition for Gaussian:  y = A*exp( - (u/Sigma)^2) + D 
   As a test program for the plot library, a curve fitting routine is not
   included.  It is left as an excercise for the reader. */
float A = 70.49879;
float D = 0.1953;
float fopt = 0.63581;
float Sigma = 0.37157;
/* --------------------------------------------------------------------------- */
extern void do_SFtuning(void);
extern void do_Lissa(void);
extern void generateContent(void);
/* --------------------------------------------------------------------------- */

static unsigned char aimage[256];

void main(int argc, char *argv[])
{
char *bufPDF;
int i, length= 0;
float width = 0.0, height = 0.0, xscale = 2.0, yscale = 2.0;
	cpdf_open(0);
	cpdf_enableCompression(YES);		/* use Flate/Zlib compression */
	/* cpdf_setMonthNames(monthNamesGerman); */	/* see above */
	cpdf_setMonthNames(monthNamesFrench);
	cpdf_init();
	cpdf_pageInit(1, PORTRAIT, LETTER, LETTER);		/* page orientation */

	cpdf_setCreator("ClibPDF Test Program: testpdf.c");
	cpdf_rawSetAnnotation(0, 0, 300, 100, "Title: ClibPDF Test", myAnnotation);

	/* Your drawing code in PDF below */
	generateContent();			/* All drawing/graphics here ! */

	/* a new page 3, can skip page 2 */
	cpdf_pageInit(3, PORTRAIT, LETTER, LETTER);		/* page orientation */
	cpdf_importImage("testimg.jpg", JPEG_IMG, 2.0, 2.0, 20.0, &width, &height, &xscale, &yscale, 1);
	cpdf_setPageTransition(TRANS_BOX, 1.0, 30.0, 0, 0);

	/* try an in-line image of gray ramp here */
	for(i=0; i<256; i++)
	    aimage[i] = (unsigned char)i;

	cpdf_rawPlaceInLineImage(aimage, 256,
		    72.0, 72.0, 0.0, 72.0*6.5, 36.0,
		    256, 1, 8, CS_GRAY, 1);
	cpdf_rawRect(72.0, 72.0, 72.0*6.5, 36.0);	/* put a box around it */
	cpdf_setgray(0.0);
	cpdf_setlinewidth(0.3);
	cpdf_stroke();
	

	/* switch back to page 1, but we will get the default domain there. 
	   Put 10 red stars in a row.
	*/
	cpdf_setCurrentPage(1);
	cpdf_setrgbcolor(1.0, 0.3, 0.3);
	for(i=0; i<10; i++)
	    cpdf_marker(0.7*(float)i + 1.0, 10.6, 4, 16.0);

	/* We never made page 2, so page 3 becomes page 2 in the PDF file. */

	cpdf_finalizeAll();			/* PDF file is actually written here */

	/* if you wish to use memory-resident PDF stream do as follows:
	   bufPDF points to the beginning of buffer containing PDF, and length is a byte count.
	*/
	bufPDF = cpdf_getBufferForPDF(&length);
	/* printf("PDF buffer length= %d\n", length); */

	/* Or you can save generated PDF to file */
	cpdf_savePDFmemoryStreamToFile("atest.pdf");

	cpdf_close();		/* shut down the library resources */
	cpdf_launchPreview();	/* launch Acrobat/PDF viewer on the output file */
}


void generateContent(void)
{
CPDFplotDomain *myDomain, *oldDom;
float width=0.0, height=0.0, xscale=0.0, yscale=0.0;
	cpdf_fillDomainWithRGBcolor(defaultDomain, 0.90, 0.93, 0.92);
	myDomain = cpdf_createPlotDomain( 1.0*inch, 1.0*inch, 6.5*inch, 9.0*inch,
				0.0, 6.5, 0.0, 9.0, LINEAR, LINEAR, 0);
	oldDom = cpdf_setPlotDomain(myDomain);	/* save oldDomain for later restore */

	/* OK, do some real plotting with actual data in a semi-log domain */
	do_SFtuning();

	/* plot in the default domain of x = cos(3t); y = sin(5t); */
	do_Lissa();

	cpdf_comments("\n%% Text examples here.\n");

	cpdf_setgrayFill(0.0);				/* Black */
	cpdf_beginText(0);
	cpdf_setFont("Helvetica-Bold", "MacRomanEncoding", 18.0);
	/* cpdf_rawText() addresses the location by raw point-based coordinate system. */
	cpdf_rawText(110.0, 720.0, 0.0, "Test of ClibPDF library for C (testpdf.c)");
	cpdf_endText();

	cpdf_setrgbcolorFill(1.0, 0.0, 0.0);		/* Red */
	cpdf_beginText(0);
	cpdf_setFont("Times-Bold", "MacRomanEncoding", 14.0);
	/* cpdf_text() uses a flexible domain based coordinate system. See cpdf_setPlotDomain(). */
	cpdf_text(3.5, 1.2, 90.0, "Vertical text line");
	cpdf_text(3.7, 1.2, 30.0, "Oblique text line at 30 degrees");
	cpdf_endText();

	/* Now try importing JPEG images */
	/* use the pixel size of image as points */
	width=0.0; height=0.0; xscale=0.0; yscale=0.0;
	cpdf_importImage("testimg.jpg", JPEG_IMG, 4.0, 6.0, 20.0, &width, &height, &xscale, &yscale, 1);


	cpdf_setrgbcolorFill(0.0, 0.0, 1.0);		/* Blue */
	cpdf_beginText(0);
	cpdf_setFont("Helvetica", "MacRomanEncoding", 14.0);
	cpdf_text(0.2, 8.5, 0.0, "A line of Helvetica at 14 point using the default domain");
	cpdf_text(0.2, 8.2, 0.0, "defined in the inch based coordinate system.");
	cpdf_endText();

	cpdf_setgrayFill(0.0);				/* Black */
	cpdf_beginText(0);
	cpdf_setFont("Courier", "MacRomanEncoding", 14.0);
	cpdf_text(0.2, 7.9, 0.0, "Test of Courier font here.");
	cpdf_endText();

	/* Here we use the same image again, the image data from the previous call will
	   be reused, and will not be duplicated in the PDF file. */
	width = 0.0; height = 0.0;	/* must reset these, as they were used above */
	xscale = 0.5;
	yscale = 0.0;
	cpdf_importImage("testimg.jpg", JPEG_IMG, 4.0, 0.5, 0.0, &width, &height, &xscale, &yscale, 1);

	cpdf_setPlotDomain(oldDom);		/* restore previous plot domain */
	cpdf_freePlotDomain(myDomain);		/* deallocate the plot domain */
}

void do_SFtuning(void)
{
int i;
int marktype = 0;		/* 0=circle, 1=up triangle, 2=diamond, 3=square, 4=star */
float markersize = 9.0;
float yp, xp, xbump;
CPDFplotDomain *sfDomain, *oldDomain, *timeDomain;
CPDFaxis *xAxis, *yAxis, *xxA2;
time_t t;
struct tm tm; 
struct tm tm2;

	/* Create a plot domain for SF tuning curve.  (X-axis logarithmic, Y-axis linear) */
	/* (CPDFplotDomain *)cpdf_createPlotDomain(float x, float y, float w, float h,
				float xL, float xH, float yL, float yH, int xtype, int ytype, int rsvd) */
	sfDomain = cpdf_createPlotDomain( 1.6*inch, 5.5*inch, 4.5*inch, 3.0*inch,
				xsfmin, xsfmax, 0.0, respmax, LOGARITHMIC, LINEAR, 0);
	oldDomain = cpdf_setPlotDomain(sfDomain);	/* save oldDomain for later restore */
	cpdf_fillDomainWithGray(sfDomain, 1.0);				/* white */
	/* cpdf_fillDomainWithRGBcolor(sfDomain, 1.0, 0.9, 0.9); */	/* light pink */

	/* Mesh lines for the domain -------------------------------------------------------------- */
	/* void cpdf_setLinearMeshParams(CPDFplotDomain *aDomain, int xy, float mesh1ValMajor, float intervalMajor,
					      float mesh1ValMinor, float intervalMinor) */
	/* void cpdf_drawMeshForDomain(CPDFplotDomain *aDomain) */
	/* cpdf_setLinearMeshParams(sfDomain, Y_MESH, 0.0, 20.0, 0.0, 10.0); */  /* not needed */
	cpdf_drawMeshForDomain(sfDomain);

	cpdf_setgray(0.0);
	/* X-Axis --------------------------------------------------------------------------------- */
	/* CPDFaxis *cpdf_createAxis(float angle, float axislength, int typeflag, float valL, float valH); */

	xAxis = cpdf_createAxis( 0.0, 4.5*inch, LOGARITHMIC, xsfmin, xsfmax);
	cpdf_setLogAxisNumberSelector(xAxis, LOGAXSEL_13);	/* Nums 1 and 3s */
	cpdf_attachAxisToDomain(xAxis, sfDomain, 0.0, -0.2*inch);
	cpdf_setAxisNumberFormat(xAxis, "%g", "Helvetica", 16.0);
	cpdf_setAxisLabel(xAxis, "Frequency [c/deg]", "Times-Roman", "MacRomanEncoding", 20.0);
	cpdf_drawAxis(xAxis);

	/* A given existing axis can be attached to other similar domains, and reused. */
	/* Here, attach it to the default domain for fun. */
	/*
	cpdf_attachAxisToDomain(xAxis, defaultDomain, 0.0, 0.0);
	cpdf_drawAxis(xAxis);
	*/
	cpdf_freeAxis(xAxis);


	/* Y-Axis --------------------------------------------------------------------------------- */

	yAxis = cpdf_createAxis(90.0, 3.0*inch, LINEAR, 0.0, respmax);
	cpdf_attachAxisToDomain(yAxis, sfDomain, -0.2*inch, 0.0);
	cpdf_setAxisNumberFormat(yAxis, "%2.f", "Helvetica", 16.0);
	/* void cpdf_setAxisTicNumLabelPosition(CPDFaxis *anAx, int ticPos, int numPos, int horizNum, int horizLab); */
	/* cpdf_setAxisTicNumLabelPosition(yAxis, 0, 2, 1, 1); */
	cpdf_setAxisLabel(yAxis, "Response [imp/sec]", "Times-Roman", "MacRomanEncoding", 20.0);
	cpdf_drawAxis(yAxis);
	cpdf_freeAxis(yAxis);


	/* Null Response line */
	cpdf_comments("\n%% spontaneous response level.\n");
	cpdf_setlinewidth(0.6);
	cpdf_setgrayStroke(0.0);	/* black */
	cpdf_moveto(xsfmin, nullresp);
	cpdf_lineto(xsfmax, nullresp);
	cpdf_stroke();

	/* Do the tuning curve first */
	cpdf_comments("\n%% plot the curve.\n");
	cpdf_setlinewidth(2.0);
	cpdf_setrgbcolorStroke(0.0, 0.0, 1.0);	/* blue */
	xbump = pow(xsfmax / xsfmin, 0.01);
	xp = xsfmin;
	for(i=0; i<100; i++) {
	    xp *= xbump;
	    /* A Gaussian curve */
	    yp = A * exp( - ((xp - fopt) / Sigma) * ((xp - fopt) / Sigma)) + D;
	    if(i==0) cpdf_moveto(xp, yp);
	    else     cpdf_lineto(xp, yp);
	}
	cpdf_stroke();

	/* Do markers after the curve is drawn, so that the markers are in front of the curve. */
	cpdf_comments("\n%% place data point markers.\n");
	cpdf_setgrayStroke(0.0);
	cpdf_setrgbcolorFill(1.0, 1.0, 0.0);		/* yellow inside */
	for(i=0; i<Nxsf; i++)
	    cpdf_marker(xsf[i], Resp[i], marktype, markersize);

	/* Try some centered text at (x, y) = (0.1, 50) */
	cpdf_setgrayFill(0.5);				/* Black */
	cpdf_beginText(0);
	cpdf_setFont("Helvetica", "MacRomanEncoding", 30.0);
	cpdf_textAligned(0.1, 50.0, 0.0, TEXTPOS_UM,"303");
	cpdf_setgrayFill(0.0);				/* Black */
	cpdf_textAligned(0.1, 50.0, 60.0, TEXTPOS_UM,"ABCDE");
	cpdf_endText();
	

	cpdf_setPlotDomain(oldDomain);		/* restore previous plot domain */
	cpdf_freePlotDomain(sfDomain);		/* deallocate the plot domain */

	/* xxA2 --------------------------------------------------------------------------------- */
	/* Time domain version */
	t = time(NULL);
	memcpy(&tm, localtime(&t), sizeof(struct tm));
	tm.tm_sec = 0;
	tm.tm_min = 0;
	tm.tm_hour = 0;
	tm.tm_mday = 1;		/* day starts at 1 */
	tm.tm_mon = 0;		/* month starts at 0 */
	tm.tm_isdst = -1;	/* This is needed.  Strange things happen ohterwise. */
	mktime(&tm);
	memcpy(&tm2, &tm, sizeof(struct tm));
	/* tm2.tm_year += 8; */		/* +2 years */
	/* tm2.tm_hour += 6; */		/* +6 hours */
	/* tm2.tm_mday += 4; */		/* days */
	tm2.tm_mon += 6;		/* months */

	/* Creat time plot domain and use previously created time axis */
	timeDomain = cpdf_createTimePlotDomain( 1.0*inch, 1.0*inch, 6.5*inch, 2.8*inch,
				&tm, &tm2, 0.0, respmax, TIME, LINEAR, 0);
	oldDomain = cpdf_setPlotDomain(timeDomain);	/* save oldDomain for later restore */
	cpdf_fillDomainWithGray(timeDomain, 0.85);				/* white */
	cpdf_drawMeshForDomain(timeDomain);

	cpdf_setgrayFill(0.0);
	xxA2 = cpdf_createTimeAxis(0.0, 6.5*inch, TIME, &tm, &tm2);
	cpdf_setTimeAxisNumberFormat(xxA2, MONTH_NAME, YEAR_2DIGIT, "Helvetica", 16.0);	/* YEAR_FULL for 1998 */
	cpdf_setAxisLabel(xxA2, "Date", "Helvetica-BoldOblique", "MacRomanEncoding", 20.0);
	cpdf_attachAxisToDomain(xxA2, timeDomain, 0.0, 0.0 );
	cpdf_drawAxis(xxA2);
	cpdf_freeAxis(xxA2);

	cpdf_setrgbcolorFill(0.0, 1.0, 0.0);
	cpdf_setgrayStroke(0.0);
	for(i=0; i<4; i++) {
	    xp = tm_to_NumDays(&tm, &tm2);
	    cpdf_marker(xp, 40.0 - 10.0*(float)i, 4, 18.0);	
	    tm2.tm_mon -= 1;		/* months */
	}
	cpdf_setPlotDomain(oldDomain);		/* restore previous plot domain */
	cpdf_freePlotDomain(timeDomain);		/* deallocate the plot domain */
}

void do_Lissa(void)
{
int i;
float x, y, angle;
float radius2= 1.2;
float xorig = 1.5, yorig = 1.5;

	for(i=0; i<=200; i++) {
	    angle = PI*(float)i/100.0;
	    x = radius2 * cos(3.0*angle) + xorig;
	    y = radius2 * sin(5.0*angle) + yorig;
	    if(i) cpdf_lineto(x, y);
	    else  cpdf_moveto(x, y);		/* first point */
	}
	cpdf_closepath();
	cpdf_setrgbcolorFill(1.0, 0.7, 0.7);	/* pink */
	cpdf_setrgbcolorStroke(0.0, 0.0, 1.0);	/* blue */
	cpdf_eofillAndStroke();			/* even-odd fill */
}

