/* -*-C-*-
*******************************************************************************
*
* File:         mailtoc.c
* RCS:          /usr/local/sources/CVS/mailapp-utilities/mailtoc.c,v 1.13 1999/04/20 21:03:36 tom Exp
* Description:  Table-of-contents access routines for Mail.app utilities
* Author:       Carl Edman
* Created:      Sun Apr 25 10:31:24 1993
* Modified:     Sun Jul 19 23:06:37 1998 Tom Hageman <tom@basil.icce.rug.nl>
* Language:     C
* Package:      EnhanceMail / mailapp-utilities
* Status:       
*
* (C) Copyright 1993, but otherwise this file is perfect freeware.
*
*******************************************************************************
*/

#import <libc.h>
#import <errno.h>

#import "mailtoc.h"

#if (NS_TARGET_MAJOR >= 4 || defined(__APPLE_CC__)) /* OPENSTEP/Rhapsody */
#  import <Foundation/NSByteOrder.h>
#  define NXSwapBigLongToHost		NSSwapBigLongToHost
#  define NXSwapBigFloatToHost		NSSwapBigFloatToHost
#  define NXConvertHostFloatToSwapped	NSConvertHostFloatToSwapped
#  define NXSwapHostLongToBig		NSSwapHostLongToBig
#  define NXSwapHostFloatToBig		NSSwapHostFloatToBig
#  define NXConvertSwappedFloatToHost	NSConvertSwappedFloatToHost
#endif

#define TOC_MESSAGE_INDEX_MAX 100000
   /* arbitrary limit on message index record size, to avoid crashes on corrupt mailboxes. */


#define INT_VALUE_BEFORE(p)   ((int)((p[-4]<<24)+(p[-3]<<16)+(p[-2]<<8)+p[-1]))

#define SET_INT_VALUE_BEFORE(p, val) \
	    (p[-4]=(val)>>24, p[-3]=(val)>>16, p[-2]=(val)>> 8, p[-1]=(val))

#define SKIP_OVER_EXTENDED_FIELD(p,mi,offset) \
	    (p = mi->data, \
	     p += strlen(p)+1, \
	     p += strlen(p)+1, \
	     p += strlen(p)+1, \
	     p += offset, \
	     (p - (const unsigned char *)mi) <= mi->record_length)

static void swap_table_of_contents_header_to_host(struct table_of_contents_header *toc)
{
   toc->magic=NXSwapBigLongToHost(toc->magic);
   toc->num_msgs=NXSwapBigLongToHost(toc->num_msgs);
   toc->mbox_time=NXSwapBigLongToHost(toc->mbox_time);
   toc->list=NXSwapBigFloatToHost(NXConvertHostFloatToSwapped(toc->list));
   toc->window.origin.x=NXSwapBigFloatToHost(NXConvertHostFloatToSwapped(toc->window.origin.x));
   toc->window.origin.y=NXSwapBigFloatToHost(NXConvertHostFloatToSwapped(toc->window.origin.y));
   toc->window.size.width=NXSwapBigFloatToHost(NXConvertHostFloatToSwapped(toc->window.size.width));
   toc->window.size.height=NXSwapBigFloatToHost(NXConvertHostFloatToSwapped(toc->window.size.height));
}

struct table_of_contents_header *get_table_of_contents_header(FILE *f,int createflag)
{
   int n;
   struct table_of_contents_header *toc;
   toc=malloc(sizeof(*toc));
   if(toc==NULL) return 0;
   errno=0;
   if ((n=fread((void *)toc,1,sizeof(*toc),f))!=sizeof(*toc))
   {
       fprintf(stderr, "\n *** creating new TOC ***\n");
      if (!(createflag && n==0))
      {
	 free(toc);
	 return 0;
      }
      /* Assume MacOS X(Server) MailViewer understands old magic number,
         so use it when creating a mailbox for backward compatibility. */
      toc->magic=MBOX_TOC_MAGIC;
      toc->num_msgs=0;
      toc->mbox_time=-1;
      toc->list=102.0;
      toc->window.origin.x=271.0;
      toc->window.origin.y=151.0;
      toc->window.size.width=557.0;
      toc->window.size.height=532.0;
      /* Write table of contents immediately, so space for it is allocated
         on-disk.  {{Does NOT fall through to byteswapping below anymore,
	 since this is handled in put_table_of_contents_header itself now.}} */
      if (put_table_of_contents_header(f,toc)!=0)
      {
	 free(toc);
	 return 0;
      }
   }
   else
   {
      swap_table_of_contents_header_to_host(toc);
   }
   if (toc->magic != MBOX_TOC_MAGIC &&
       toc->magic != MBOX_TOC_MAGIC_MACOS_X) /* basic sanity check. */
   {
      free(toc);
      return 0;
   }
   return toc;
}

static inline void swap_message_index_to_host(struct message_index *mi)
{
   mi->mes_offset=NXSwapBigLongToHost(mi->mes_offset);
   mi->mes_length=NXSwapBigLongToHost(mi->mes_length);
   mi->mes_date=NXSwapBigLongToHost(mi->mes_date);
}

struct message_index *get_message_index(FILE *f)
{
   unsigned long rl;
   struct message_index *mi;
   errno=0;
   if (fread(&rl,1,sizeof(rl),f)!=sizeof(rl)) return 0;
   rl=NXSwapBigLongToHost(rl);
   if (rl > TOC_MESSAGE_INDEX_MAX || rl < sizeof(*mi)) return 0;
   mi=malloc(rl);
   if(mi==NULL) return 0;
   mi->record_length=rl;
   if (fread(((char *)mi)+sizeof(rl),1,rl-sizeof(rl),f)!=rl-sizeof(rl))
   {
      free(mi);
      return 0;
   }
   swap_message_index_to_host(mi);
   return mi;
}

static inline void swap_message_index_to_disk(struct message_index *mi)
{
   mi->mes_offset=NXSwapHostLongToBig(mi->mes_offset);
   mi->mes_length=NXSwapHostLongToBig(mi->mes_length);
   mi->mes_date=NXSwapHostLongToBig(mi->mes_date);
}

int put_message_index(FILE *f,struct message_index *mi)
{
   long reclen=mi->record_length;
   int ret;
   mi->record_length=NXSwapHostLongToBig(mi->record_length);
   swap_message_index_to_disk(mi);
   errno=0;
   ret=(fwrite(mi,1,reclen,f)!=reclen);
   mi->record_length=NXSwapBigLongToHost(mi->record_length);
   swap_message_index_to_host(mi);
   return ret;
}

static void swap_table_of_contents_header_to_disk(struct table_of_contents_header *toc)
{
   toc->magic=NXSwapHostLongToBig(toc->magic);
   toc->num_msgs=NXSwapHostLongToBig(toc->num_msgs);
   toc->mbox_time=NXSwapHostLongToBig(toc->mbox_time);
   toc->list=NXConvertSwappedFloatToHost(NXSwapHostFloatToBig(toc->list));
   toc->window.origin.x=NXConvertSwappedFloatToHost(NXSwapHostFloatToBig(toc->window.origin.x));
   toc->window.origin.y=NXConvertSwappedFloatToHost(NXSwapHostFloatToBig(toc->window.origin.y));
   toc->window.size.width=NXConvertSwappedFloatToHost(NXSwapHostFloatToBig(toc->window.size.width));
   toc->window.size.height=NXConvertSwappedFloatToHost(NXSwapHostFloatToBig(toc->window.size.height));
}

int put_table_of_contents_header(FILE *f,struct table_of_contents_header *toc)
{
   int ret;
   swap_table_of_contents_header_to_disk(toc);
   errno=0;
   rewind(f);
   ret=(fwrite(toc,1,sizeof(*toc),f)!=sizeof(*toc));
   swap_table_of_contents_header_to_host(toc);
   return ret;
}

#define MESSAGE_DATE(y,m,d)   ((d)+((m)<<5)+((y)<<9))

long message_current_date(void)
{
   time_t clock;
   struct tm *ctm;
   time(&clock);
   ctm=localtime(&clock);
   return MESSAGE_DATE(ctm->tm_year+1900, ctm->tm_mon+1, ctm->tm_mday);
}

long message_date(int year,const char *month,int day)
{
   static const char months[][4]=
   {
      "Jan", "Feb", "Mar", "Apr", "May", "Jun",
      "Jul", "Aug", "Sep", "Oct", "Nov", "Dec"
   };
   int m;

   for(m=0;m<12;m++) if (!strncasecmp(month,months[m],3)) break;
   m=(m%12)+1; /* Wrap around to January for unidentifiable month
                  It's better than failing for an unattended program */

   if (year < 200) year += (year < 70 ? 2000 : 1900);
   /* Y2K fix -- we don't expect e-mail from before 200AD :-) */

   if (day <= 0) day = 1; else if (day > 31) day = 31;

   return MESSAGE_DATE(year, m, day);
}

char *message_from(const struct message_index *mi)
{
   char *p=mi->data;
   return p;
}

char *message_subject(const struct message_index *mi)
{
   char *p=mi->data;
   p+=strlen(p)+1;
   return p;
}

char *message_reference(const struct message_index *mi)
{
   char *p=mi->data;
   p+=strlen(p)+1;
   p+=strlen(p)+1;
   return p;
}

int message_attachsize(const struct message_index *mi)
{
   const unsigned char *p;

   if (!SKIP_OVER_EXTENDED_FIELD(p, mi, 4)) return -1;
   return INT_VALUE_BEFORE(p);   
}

int message_set_attachsize(struct message_index *mi,int size)
{
   unsigned char *p;

   if (!SKIP_OVER_EXTENDED_FIELD(p, mi, 4)) return -1;
   SET_INT_VALUE_BEFORE(p, size);
   return size;
}

time_t message_attachtime(const struct message_index *mi)
{
   const unsigned char *p;

   if (!SKIP_OVER_EXTENDED_FIELD(p, mi, 8)) return -1;
   return (time_t)INT_VALUE_BEFORE(p);
}

time_t message_set_attachtime(struct message_index *mi,time_t time)
{
   unsigned char *p;

   if (!SKIP_OVER_EXTENDED_FIELD(p, mi, 8)) return -1;
   SET_INT_VALUE_BEFORE(p, (int)time);
   return time;
}

int message_priority(const struct message_index *mi)
{
   const unsigned char *p;

   if (!SKIP_OVER_EXTENDED_FIELD(p, mi, 12)) return -1;
   return INT_VALUE_BEFORE(p);
}

int message_set_priority(struct message_index *mi,int priority)
{
   unsigned char *p;

   if (!SKIP_OVER_EXTENDED_FIELD(p, mi, 12)) return -1;
   SET_INT_VALUE_BEFORE(p, priority);
   return priority;
}

inline
void message_get_date(const struct message_index *mi, int *year, int *month, int *day)
{
   (*year) = mi->mes_date >> 9;
   (*month) = (mi->mes_date >> 5) & 0xf;
   (*day) = mi->mes_date & 0x1f;
}

/* New for Rhapsody / MacOS X -- with fallback to mes_date. */
time_t message_time(const struct message_index *mi)
{
   const unsigned char *p;

   if (!SKIP_OVER_EXTENDED_FIELD(p, mi, 16))
   {
      int y, m, d;
      struct tm tmbuf, *tm = &tmbuf;
      time_t time;

      message_get_date(mi, &y, &m, &d);
      tm->tm_year = y - 1900;
      tm->tm_mon = m - 1;
      tm->tm_mday = d;
      tm->tm_hour = 0;
      tm->tm_min = 0;
      tm->tm_sec = 0;
      tm->tm_gmtoff = 0;
      tm->tm_isdst = 0;
      tm->tm_zone = "";
      time = mktime(tm);
      /* this yields time in UTC (aka GMT).  We want 00:00:00, local time. */
      tm = localtime(&time);
      tm->tm_hour = 0;
      tm->tm_min = 0;
      tm->tm_sec = 0;
      return mktime(tm);
   }
   return (time_t)INT_VALUE_BEFORE(p);
}

time_t message_set_time(struct message_index *mi,time_t time)
{
   unsigned char *p;
   // Set mes_date regardless.
   struct tm *tm = localtime(&time);

   mi->mes_date = MESSAGE_DATE(tm->tm_year+1900, tm->tm_mon+1, tm->tm_mday);

   if (!SKIP_OVER_EXTENDED_FIELD(p, mi, 16))
   {
      tm->tm_hour = 0;
      tm->tm_min = 0;
      tm->tm_sec = 0;
      return mktime(tm);
   }
   SET_INT_VALUE_BEFORE(p, (int)time);
   return time;
}

int message_age(const struct message_index *mi)
{   
   time_t ctmt,mtmt;
   struct tm *tmp;
   int y, m, d;

   time(&ctmt);
   tmp=localtime(&ctmt);
   message_get_date(mi, &y, &m, &d);
   tmp->tm_year = y - 1900;
   tmp->tm_mon  = m - 1;
   tmp->tm_mday = d;
   mtmt=mktime(tmp);
   return ((ctmt-mtmt+(86400/2))/86400);
}
