/* 
	Mailbox.m created by dirk on Fri 10-Sep-1999
*/
#import <Foundation/Foundation.h>
#import "Mailbox.h"
#import "TableOfContents.h"
#import "globals.h"

@implementation Mailbox : NSObject

//------------------------------------------------------------------------------
// Class methods
//------------------------------------------------------------------------------
+ (NSArray *)mailboxesForDirectoryAtPath:(NSString *)path
/*"
Returns alls Mailboxes, which exist at %path. Takes care that the mailbox really exists and traverses links, if necessary
"*/
{
    NSMutableArray	*returnFileNames;
    NSEnumerator	*fileEnum;
    NSString		*element;

    returnFileNames = [NSMutableArray array];
    fileEnum = [[[NSFileManager defaultManager] directoryContentsAtPath:path] objectEnumerator];
    while ((element = [fileEnum nextObject]) != nil)
    {
        if (([element rangeOfString:@".mbox"]).length > 0)
        {
            [returnFileNames addObject:[path stringByAppendingPathComponent:element]];
        }
        else if ([[[NSFileManager defaultManager] fileAttributesAtPath:[path stringByAppendingPathComponent:element] traverseLink:YES] objectForKey:NSFileType] == NSFileTypeDirectory)
        {
            // we have found a subdirectory. Search for mailboxes as well
            NSArray	*mboxes;

            mboxes = [self mailboxesForDirectoryAtPath:[path stringByAppendingPathComponent:element]];
            if ([mboxes count] > 0)
            {
                [returnFileNames addObjectsFromArray:mboxes];
            }
        }
    }
    return returnFileNames;
}

+ (NSString *)mailViewerMailboxPath
/*"
Reads in MailViewer's defaults to find out the path, where the mailboxes are located.
"*/
{
    NSArray			*searchList;
    NSUserDefaults	*userDefaults;
    NSString 		*mailDir;

    userDefaults = [NSUserDefaults standardUserDefaults];
    searchList = [userDefaults searchList];

    // search only MailViewer's defaults
    [userDefaults setSearchList:[NSArray arrayWithObject:@"MailViewer"]];
    mailDir = [userDefaults stringForKey:gvMailViewerMailDir];
    NSAssert1(mailDir != nil, @"MailViewer has no default named %@", gvMailViewerMailDir);

    [userDefaults setSearchList:searchList];
    return mailDir;
}

//------------------------------------------------------------------------------
// Instance methods
//------------------------------------------------------------------------------
- (id)initWithMailboxPath:(NSString *)mailboxPath;
{
    NSCalendarDate	*modDate;
    
    [super init];
    ivMailboxPath = [mailboxPath retain];
    ivHasNewMail = NO;

    // check if mailbox really exists
    if ([[NSFileManager defaultManager] fileExistsAtPath:mailboxPath] == NO)
    {
        [self autorelease];
        return nil;
    }

    ivTableOfContents = [[TableOfContents alloc] initWithMailboxPath:[self mailboxPath]];
    modDate = [ivTableOfContents modificationDate];
    [self setPreviousModificationDate:modDate];

    return self;
}

- (void)dealloc
{
    [ivPreviousModificationDate release];
    [ivMailboxPath release];
    [ivTableOfContents release];
    [super dealloc];
}

- (BOOL)hasNewMail
/*"
Check if this mailbox has new mail. This happens in two steps: first, check the modification date of the mailbox. If it is still the same, no new mail arrived. If it differs scan the table_of_contents of that mailbox.
"*/
{
    NSCalendarDate	*modDate;

    modDate = [ivTableOfContents modificationDate];
	if ([modDate compare:ivPreviousModificationDate] != NSOrderedSame)
    {
        // Mailbox modified, check table of contents for unread mails
        [self setPreviousModificationDate:modDate];

        ivHasNewMail = [ivTableOfContents hasUnreadMails];
    }
    return ivHasNewMail;
}

- (NSComparisonResult)compare:(Mailbox *)otherMailbox
/*"
Compare the names of two mailboxes.
"*/
{
	return [[self mailboxName] compare:[otherMailbox mailboxName]];
}

- (void)checkTOCForUnreadMail
/*"
This method will be called on app's start when the 'CheckOnActivate' default is set.
"*/
{
    [ivTableOfContents hasUnreadMails];
}

//------------------------------------------------------------------------------
// Accessor methods
//------------------------------------------------------------------------------
- (NSString *)mailboxName
{
    return [ivMailboxPath lastPathComponent];
}

- (NSString *)mailboxPath
{
    return ivMailboxPath;
}

- (NSString *)relativePath
{
    NSRange	range;

    range = [ivMailboxPath rangeOfString:[[self class] mailViewerMailboxPath]];
    return [ivMailboxPath substringFromIndex:(range.length + 1)];
}

- (void)setPreviousModificationDate:(NSCalendarDate *)date
{
    [date retain];
    [ivPreviousModificationDate release];
    ivPreviousModificationDate = date;
}

@end
