//---------------------------------------------------------------------------------------
//	DockController.m created by erik on Mon 13-Jul-1998
//	This code is part of the Spectre Project by Erik Doernenburg. For copyright details
//	see GNU public license version 2 or above. No warranties implied. Use at own risk.
//	More information can be found at <http://www.object-factory.com/~erik/Spectre>.
//	@(#)$Id: DockController.m,v 1.7 1998/09/27 16:08:23 erik Exp $
//---------------------------------------------------------------------------------------

#import <AppKit/AppKit.h>
#include <unistd.h>
#import "LocalizableStrings.h"
#import "PrefController.h"
#import "SPCTileView.h"
#import "SPCGrid.h"
#import "SPCPoint.h"
#import "DockController.h"


@interface DockController(PrivateAPI)
- (void)_storeConfigInDefaults;
- (void)_retrieveConfigFromDefaults;
- (NSWindow *)_newTileWindowForPath:(NSString *)path;
- (NSPoint)_screenPointFromPoint:(SPCPoint *)point;
- (NSString *)_getAppStateForPath:(NSString *)path;
- (NSString *)__getAppStateForPath:(NSString *)path withProcessSet:(NSSet *)processes;
- (NSSet *)__runningProcesses;
@end


static NSString *SPCDockPathSubkey = @"path";
static NSString *SPCDockXPosSubkey = @"x";
static NSString *SPCDockYPosSubkey = @"y";

#define USER_DEFAULTS [NSUserDefaults standardUserDefaults]

//---------------------------------------------------------------------------------------
    @implementation DockController
//---------------------------------------------------------------------------------------


//---------------------------------------------------------------------------------------
//	INITIALISATION
//---------------------------------------------------------------------------------------

- init
{
    NSNotificationCenter *wsnc;

    NSLog(@"creating dock window");

    [super init];

    tileWindows = [[SPCGrid allocWithZone:[self zone]] init];
    tileSize = [SPCTileView tileSize];
    [self _retrieveConfigFromDefaults];
    
    wsnc = [[NSWorkspace sharedWorkspace] notificationCenter];
    [wsnc addObserver:self selector:@selector(appStateChanged:) name:NSWorkspaceDidLaunchApplicationNotification object:nil];
    [wsnc addObserver:self selector:@selector(appStateChanged:) name:NSWorkspaceWillLaunchApplicationNotification object:nil];
    [wsnc addObserver:self selector:@selector(appStateChanged:) name:NSWorkspaceDidTerminateApplicationNotification object:nil];
    [[NSNotificationCenter defaultCenter] addObserver:self selector:@selector(userDefaultsChanged:) name:NSUserDefaultsDidChangeNotification object:nil];

   return self;
}


//---------------------------------------------------------------------------------------
//	DEALLOC
//---------------------------------------------------------------------------------------

- (void)dealloc
{
    NSLog(@"removing dock window");
    [[[NSWorkspace sharedWorkspace] notificationCenter] removeObserver:self];
    [[NSNotificationCenter defaultCenter] removeObserver:self];
    [myWindow orderOut:self];
    [myWindow release];
    [[tileWindows allObjects] makeObjectsPerformSelector:@selector(orderOut:) withObject:self];
    [tileWindows release];
    [super dealloc];
}


//---------------------------------------------------------------------------------------
//	DERIVED ATTRIBUTES
//---------------------------------------------------------------------------------------

- (NSPoint)dockOrigin;
{
    return [myWindow frame].origin;
}


- (NSScreen *)dockScreen;
{
    return [NSScreen mainScreen];
}


//---------------------------------------------------------------------------------------
//	TARGET/DELGATE FOR TILES
//---------------------------------------------------------------------------------------
- (void)launchApplication:(id)sender andHideOthers:(BOOL)hideOthers {
    if(hideOthers) {
        /* the approach of hideOtherApplications and then launching a application does not always work. It seems that the events are not ordered so that sometimes an application is launched BEFORE it is hidden, resulting in a  hidden app. To get arround this we wait a little and then launch the app again.
        This is neither elegant nor does not always work however I felt that the command-doubleclick is still usefull.
        eike@mcy.com
        */
        [[NSWorkspace sharedWorkspace] hideOtherApplications];
        // force task switch to give applications time to hide & launch
        [NSThread sleepUntilDate:[NSDate dateWithTimeIntervalSinceNow:10]];
    }
    if([sender appState] == NSWorkspaceDidTerminateApplicationNotification)
        [sender setAppState:NSWorkspaceWillLaunchApplicationNotification];
    [[NSWorkspace sharedWorkspace] launchApplication:[sender appPath]];
}


- (void)launchApplication:(id)sender;
{
    [self launchApplication:sender andHideOthers:NO];
}

- (void)launchApplicationAndHideOthers:(id)sender;
{
    [self launchApplication:sender andHideOthers:YES];
}

- (void)collapseOrExpand:(id)sender {
    if(! collapsed) {
        [[tileWindows allObjects] makeObjectsPerformSelector:@selector(orderOut:) withObject:self];
    } else {
        [[tileWindows allObjects] makeObjectsPerformSelector:@selector(orderFront:) withObject:self];
    }
    collapsed	= ! collapsed;
}


- (void)dragControlTile:(id)sender;
{
    NSAutoreleasePool *pool;
    NSEvent	 		  *theEvent;
    NSPoint	 		  windowOrigin, eventLocation;
    float	 		  dx, dy;
    NSEnumerator	  *positionEnum;
    NSWindow		  *tile;
    SPCPoint	  	  *position;

    theEvent = [myWindow currentEvent];
    eventLocation = [myWindow convertBaseToScreen:[theEvent locationInWindow]];
    windowOrigin = [myWindow frame].origin;
    dx = windowOrigin.x - eventLocation.x;
    dy = windowOrigin.y - eventLocation.y;
    
    while([theEvent type] != NSLeftMouseUp)
        {
        pool = [[NSAutoreleasePool allocWithZone:[self zone]] init];
        eventLocation = [myWindow convertBaseToScreen:[theEvent locationInWindow]];
        [myWindow setFrameOrigin:NSMakePoint(eventLocation.x + dx, eventLocation.y + dy)];
        positionEnum = [tileWindows positionEnumerator];
        while((position = [positionEnum nextObject]) != nil)
            {
            tile = [tileWindows objectAtPosition:position];
            [tile setFrameOrigin:[self _screenPointFromPoint:position]];
            }
        [pool release];
        // this'll "leak" events but who'd drag a tile that long...
        theEvent = [myWindow nextEventMatchingMask:(NSLeftMouseDraggedMask | NSLeftMouseUpMask)];
        }

    [self _storeConfigInDefaults];
}


- (void)toggleDockLevel:(id)sender;
{
    if([USER_DEFAULTS integerForKey:SPCDockLevelKey] > NSNormalWindowLevel)
        [USER_DEFAULTS setInteger:NSNormalWindowLevel - 1 forKey:SPCDockLevelKey];
    else
        [USER_DEFAULTS setInteger:NSMainMenuWindowLevel - 1 forKey:SPCDockLevelKey];
    [self _storeConfigInDefaults];
}


- (void)dragAppTile:(id)sender;
{
    NSAutoreleasePool *pool;
    NSEvent	 		  *theEvent;
    NSPoint	 		  windowOrigin, dockOrigin, eventLocation;
    float	 		  dx, dy;
    NSWindow		  *tile, *destMarker;
    NSEnumerator	  *posEnum;
    SPCPoint	  	  *destPosition, *position;

    tile = [sender window];
    [tile retain];
    posEnum = [tileWindows positionEnumerator];
    while((position = [posEnum nextObject]) != nil)
        if([tileWindows objectAtPosition:position] == tile)
            break;
    NSAssert(position != nil, @"Tile not found in grid.");
    destPosition = [position retain];
    [tileWindows removeObjectAtPosition:position];

    theEvent = [tile currentEvent];
    eventLocation = [tile convertBaseToScreen:[theEvent locationInWindow]];
    dockOrigin = [myWindow frame].origin;
    windowOrigin = [tile frame].origin;
    dx = windowOrigin.x - eventLocation.x;
    dy = windowOrigin.y - eventLocation.y;

    destMarker = [self _newTileWindowForPath:[sender appPath]];
    [destMarker setFrameOrigin:windowOrigin];
    [[destMarker contentView] setAppState:NSWorkspaceWillLaunchApplicationNotification];
    [destMarker orderBack:self];

    while([theEvent type] != NSLeftMouseUp)
        {
        pool = [[NSAutoreleasePool alloc] init];
        eventLocation = [tile convertBaseToScreen:[theEvent locationInWindow]];
        [tile setFrameOrigin:NSMakePoint(eventLocation.x + dx, eventLocation.y + dy)];
        position = [SPCPoint pointWithX:floor((eventLocation.x - dockOrigin.x)  / tileSize.width) andY:floor((eventLocation.y - dockOrigin.y) / tileSize.height)];
        if([tileWindows objectAtPosition:position] == nil)
            {
            [destPosition release];
            destPosition = [position retain];
            [destMarker setFrameOrigin:[self _screenPointFromPoint:position]];
            }
        [pool release];
        // this'll "leak" events but who'd drag a tile that long...
        theEvent = [myWindow nextEventMatchingMask:(NSLeftMouseDraggedMask | NSLeftMouseUpMask)];
        }

    [destMarker orderOut:self];
    [destMarker release];

    if(([destPosition x] != 0) || ([destPosition y] != 0))
        {
        [tileWindows setObject:tile atPosition:destPosition];
        [tile setFrameOrigin:[self _screenPointFromPoint:destPosition]];
        }
    else
        {
        [tile orderOut:self];
        }
    [tile release];
    [destPosition release];
    
    [self _storeConfigInDefaults];
}


- (void)tileView:(SPCTileView *)view acceptedDroppedPath:(NSString *)path;
{
    if(view == [myWindow contentView])
        {
        NSString	*app, *type;
        NSRect 	 	screenFrame;
        NSWindow 	*tile;
        SPCPoint 	*p, *dest;
        NSPoint		sp;
        int			x, y, sx, sy;
        
        [[NSWorkspace sharedWorkspace] getInfoForFile:path application:&app type:&type];
        if(([type isEqualToString:NSApplicationFileType]) || ([type isEqualToString:NSShellCommandFileType]))
            {
            screenFrame = [[NSScreen mainScreen] visibleFrame];
            x = 0; y = 1; sy = -1; sx = -1; dest = nil;
            while(dest == nil)
                {
                p = [SPCPoint pointWithX:(x * sx) andY:(y * sy)];
                sp = [self _screenPointFromPoint:p];
                if(sp.x > NSMaxX(screenFrame))
                    { break; }
                else if(sp.x < NSMinX(screenFrame))
                    { sx = 1; x = 1; sy = -1; y = 0; }
                else if(sp.y > NSMaxY(screenFrame))
                    { x += 1; sy = - 1; y = 0; }
                else if(sp.y < NSMinY(screenFrame))
                    { sy = 1; y = 1; }
                else if([tileWindows objectAtPosition:p] != nil)
                    { y += 1; }
                else
                    { dest = p; }
                }
            if(dest != nil)
                {
                tile = [self _newTileWindowForPath:path];
                [tile setFrameOrigin:[self _screenPointFromPoint:p]];
                [tileWindows setObject:tile atPosition:p];
                [[tile contentView] setAppState:[self _getAppStateForPath:path]];
                [tile orderFront:self];
                }
            else
                {
                NSRunAlertPanel(ATITLE_NO_MORE_SPACE, ATEXT_NO_MORE_SPACE, BTN_CANCEL, nil, nil);
                }
            
            }
        else
            {
            NSLog(@"don't drop files of type '%@'", type);
            NSBeep();
            }
        }
    else
        {
        if([view appState] == NSWorkspaceDidTerminateApplicationNotification)
            [view setAppState:NSWorkspaceWillLaunchApplicationNotification];
        [[NSWorkspace sharedWorkspace] openFile:path withApplication:[view appPath]];
        }
}


//---------------------------------------------------------------------------------------
//	NOTIFICATIONS 
//---------------------------------------------------------------------------------------

- (void)appStateChanged:(NSNotification *)notification;
{
    NSString 		*anAppPath;
    NSEnumerator	*tileEnum;
    NSWindow		*tile;

    anAppPath = [[notification userInfo] objectForKey:@"NSApplicationName"];
    NSLog(@"%@: %@", [notification name], anAppPath);
    tileEnum = [tileWindows objectEnumerator];
    while((tile = [tileEnum nextObject]) != nil)
        {
        if([[[tile contentView] appPath] isEqualToString:anAppPath])
            [[tile contentView] setAppState:[notification name]];
        }
}


- (void)userDefaultsChanged:(NSNotification *)notification;
{
    int windowLevel;

    windowLevel = [USER_DEFAULTS integerForKey:SPCDockLevelKey];
    [myWindow setLevel:windowLevel];
    NSAssert(sizeof(int) <= sizeof(id), @"Cannot pass int as id.");
    [[tileWindows allObjects] makeObjectsPerformSelector:@selector(setLevel:) withObject:(id)windowLevel];
}


//---------------------------------------------------------------------------------------
//	UTILITIES
//---------------------------------------------------------------------------------------

- (void)_storeConfigInDefaults;
{
    NSMutableArray	*defaults;
    NSDictionary	*tileConfig;
    NSEnumerator	*positionEnum;
    SPCPoint		*position;
    SPCTileView		*tileView;

    [USER_DEFAULTS setObject:NSStringFromPoint([self dockOrigin]) forKey:SPCDockOriginKey];
    defaults = [NSMutableArray array];
    positionEnum = [tileWindows positionEnumerator];
    while((position = [positionEnum nextObject]) != nil)
        {
        tileView = [[tileWindows objectAtPosition:position] contentView];
        tileConfig = [NSDictionary dictionaryWithObjectsAndKeys:[tileView appPath], SPCDockPathSubkey, [NSString stringWithFormat:@"%d", [position x]], SPCDockXPosSubkey, [NSString stringWithFormat:@"%d", [position y]], SPCDockYPosSubkey, nil];
        [defaults addObject:tileConfig];
        }
    [USER_DEFAULTS setObject:defaults forKey:SPCDockItemsKey];
}


- (void)_retrieveConfigFromDefaults;
{
    NSString		*string;
    NSPoint			dockOrigin;
    NSSet		 	*runningProcesses;
    NSEnumerator 	*tileConfigEnum;
    NSDictionary	*tileConfig;
    NSWindow		*tile;
    SPCPoint		*position;

    if((string = [USER_DEFAULTS objectForKey:SPCDockOriginKey]) != nil)
        {
        dockOrigin = NSPointFromString(string);
        }
    else
        {
        NSRect screenFrame = [[NSScreen mainScreen] visibleFrame];
        dockOrigin = NSMakePoint(NSMaxX(screenFrame) - tileSize.width, NSMaxY(screenFrame) - tileSize.height);
        }
    myWindow = [self _newTileWindowForPath:nil];
    [myWindow setFrameOrigin:dockOrigin];

    runningProcesses = [self __runningProcesses];
    tileConfigEnum = [[USER_DEFAULTS objectForKey:SPCDockItemsKey] objectEnumerator];
    while((tileConfig = [tileConfigEnum nextObject]) != nil)
        {
        tile = [self _newTileWindowForPath:[tileConfig objectForKey:SPCDockPathSubkey]];
        position = [SPCPoint pointWithX:[[tileConfig objectForKey:SPCDockXPosSubkey] intValue] andY:[[tileConfig objectForKey:SPCDockYPosSubkey] intValue]];
        [tile setFrameOrigin:[self _screenPointFromPoint:position]];
        [[tile contentView] setAppState:[self __getAppStateForPath:[tileConfig objectForKey:SPCDockPathSubkey] withProcessSet:runningProcesses]];
        [tileWindows setObject:tile atPosition:position];
        }

    [myWindow orderFront:self];
    [[tileWindows allObjects] makeObjectsPerformSelector:@selector(orderFront:) withObject:self];
    
}


- (NSWindow *)_newTileWindowForPath:(NSString *)path;
{
    NSWindow	 *window;
    SPCTileView	 *tileView;
    NSRect		 tileFrame;

    tileFrame = NSMakeRect(0,0, tileSize.width, tileSize.height);
    window = [[NSWindow allocWithZone:[self zone]] initWithContentRect:tileFrame styleMask:NSBorderlessWindowMask backing:NSBackingStoreBuffered defer:NO screen:[self dockScreen]];
    [window setLevel:[USER_DEFAULTS integerForKey:SPCDockLevelKey]];
    [window setDelegate:self];

    tileView = [[SPCTileView allocWithZone:[self zone]] initWithFrame:tileFrame];
    [tileView setDelegate:self];
    [tileView setTarget:self];
    if(path == nil)
        {
        [tileView setAppPath:[[NSWorkspace sharedWorkspace] fullPathForApplication:@"Spectre"]];
        [tileView setAppState:NSWorkspaceDidLaunchApplicationNotification];
        [tileView setAction:@selector(dragControlTile:)];
        [tileView setAltAction:@selector(toggleDockLevel:)];
        [tileView setDoubleAction:@selector(collapseOrExpand:)];	// hide or show app icons
        [tileView setDoubleCommandAction:@selector(launchApplicationAndHideOthers:)];	// hide all apps except myself
        }
    else
        {
        [tileView setAppPath:path];
        [tileView setAction:@selector(dragAppTile:)];
        [tileView setDoubleAction:@selector(launchApplication:)];
        [tileView setDoubleCommandAction:@selector(launchApplicationAndHideOthers:)];
        }
    [window setContentView:tileView];
    [tileView release];

    return window;
}


- (NSPoint)_screenPointFromPoint:(SPCPoint *)point;
{
    NSPoint	screenPoint;

    screenPoint = [myWindow frame].origin;
    screenPoint.x += [point x] * tileSize.width;
    screenPoint.y += [point y] * tileSize.height;

    return screenPoint; 
}


- (NSString *)_getAppStateForPath:(NSString *)path;
{
    return [self __getAppStateForPath:path withProcessSet:[self __runningProcesses]];
}


- (NSString *)__getAppStateForPath:(NSString *)path withProcessSet:(NSSet *)processes;
{
    if([processes containsObject:path])
        return NSWorkspaceDidLaunchApplicationNotification;
    return NSWorkspaceDidTerminateApplicationNotification;
}


- (NSSet *)__runningProcesses;
{
#if unix
    NSTask 			*pstask;
    NSPipe			*pspipe;
    NSMutableString	*rawOutput;
    NSCharacterSet	*wsCharset;
    NSEnumerator	*lineEnum;
    NSString		*line, *appPath;
    NSMutableSet	*paths;
    NSRange			pathRange;

    pspipe = [NSPipe pipe];
    pstask = [[NSTask alloc] init];
    [pstask setLaunchPath:@"/bin/ps"];
    [pstask setArguments:[NSArray arrayWithObject:@"axww"]];
    [pstask setStandardOutput:pspipe];
    [pstask launch];
    rawOutput = [[[NSString alloc] initWithData:[[pspipe fileHandleForReading] readDataToEndOfFile] encoding:[NSString defaultCStringEncoding]] autorelease];
    [pstask release];

    wsCharset = [NSCharacterSet whitespaceCharacterSet];
    paths = [NSMutableSet set];
    lineEnum = [[rawOutput componentsSeparatedByString:@"\n"] objectEnumerator];
    while(((line = [lineEnum nextObject]) != nil) && ([line length] > 20))
        {
        line = [line substringFromIndex:20];
        pathRange = [line rangeOfCharacterFromSet:wsCharset];
        if(pathRange.length != 0)
            line = [line substringWithRange:NSMakeRange(0, pathRange.location)];
        appPath = [line stringByDeletingLastPathComponent];
        if([[appPath pathExtension] isEqualToString:@"app"] == NO)
            appPath = line;
        [paths addObject:appPath];
        }

    return paths;

#else

    NSLog(@"Warning. Cannot determine running processes on this platform.");
    return [NSArray array];

#endif
}

//---------------------------------------------------------------------------------------
    @end
//---------------------------------------------------------------------------------------
