/* Sample code for WWDC 1998 session 122: Yellow Box Foundation in Depth

You may freely use this code.  Apple provides this code as is without
warranty to its correctness or fitness for use for any purpose.
*/

/* Implementation for ThreadPerformer class */

#import "ThreadPerformer.h"

// Structure containing information for a method invocation
typedef struct {
    id target;
    SEL selector;
    id arg1;
} ThreadPerformerQuery;

// Structure for the return value for a method invocation
typedef struct {
    id returnVal;
} ThreadPerformerResponse;

// Request another thread to perform a method and return the
// return value using messages to ports.  This is a simple
// illustration of sending messages to ports, and running a
// run loop.
//
// This could be done with condition locks, a dictionary of
// requests (mapping threads to requests or vice versa), a
// dictionary to hold per-thread return values (since multiple
// threads may be performing simultaneously) and a trivial
// message to the send port of the performer (to wake up the
// performer's thread, which is assumed to be running its own
// run loop).  Also, this could be simpler if there was only
// some specific action that had to be performed by the
// performer (such as posting a notification).

@implementation ThreadPerformer

+ (ThreadPerformer *)defaultPerformer {
    ThreadPerformer *performer;
    NSMutableDictionary *mdict;

    // You should only access a thread's dictionary
    // from the execution context of that thread itself.
    mdict = [[NSThread currentThread] threadDictionary];
    performer = [mdict objectForKey:@"ThreadPerformer"];
    if (nil == performer) {
	performer = [[self allocWithZone:NULL] init];
        [mdict setObject:performer forKey:@"ThreadPerformer"];
	[performer release];
	// (thread dictionary keeps a retain on the performer)
    }
    return performer;
}

- (id)init {
    self = [super init];
    if (!self) return nil;

    _sendPort = [[NSPort port] retain];
    [_sendPort setDelegate:self];
    _replyPort = [[NSPort port] retain];
    [_replyPort setDelegate:self];

    _lock = [[NSLock alloc] init];

    // It is only safe to use a thread's run loop from the
    // within the execution context of the thread itself
    // (in other words, a thread should not access another
    // thread's run loop).  Keep that in mind if you factor
    // this code into an "enablePerformerForMode:" type method.
    [[NSRunLoop currentRunLoop] addPort:_sendPort
	forMode:NSDefaultRunLoopMode];

    return self;
}

- (void)dealloc {
    [_lock release];
    [_sendPort release];
    [_replyPort release];
    [super dealloc];
}

// This method can be called from any other thread than the one
// with which the performer instance is associated.  It is not
// safe in its current form to call from that thread, but that
// would be simple to add (the selector should just be performed).
// It is not "safe", however, to have the reply port registered
// in multiple threads at the same time, since a thread may pick
// up a return value targeted for another thread.  There are
// various ways to fix that; here, we lock.  Note that it would
// not be correct to lock -handlePortMessage: similarly.
- (id)performSelector:(SEL)selector
	onTarget:(id)target
	withObject:(id)arg1 {
    NSData *methodData;
    NSPortMessage *message;
    id result;
    ThreadPerformerQuery query = {target, selector, arg1};

    // Retain these so they aren't invalidated
    // during the operation
    [target retain];
    [arg1 retain];

    // It is tempting to lock after the message is sent, but that
    // might result in out-of-order delivery of return values.
    [_lock lock];

    // Note: this class only works within a single process,
    // so there are no byte-ordering or architecture alignment
    // issues we need to worry about.
    methodData = [NSData dataWithBytes:&query length:sizeof(query)];
    message = [[NSPortMessage allocWithZone:NULL]
		initWithSendPort:_sendPort receivePort:_replyPort
		components:[NSArray arrayWithObject:methodData]];
    // msgid 0 means request
    [message setMsgid:0];

    // A timeout could be substituted here; we use
    // +distantFuture to mean no timeout.
    if (![message sendBeforeDate:[NSDate distantFuture]]) {
	[message release];
	[_lock unlock];
	[NSException raise:NSInternalInconsistencyException
		format:@"Could not perform selector"];
    }
    [message release];	// Done with message

    // Set up current thread's run loop and
    // use it to wait for a reply.
    _returnValue = nil;
    [[NSRunLoop currentRunLoop] addPort:_replyPort
	forMode:@"ThreadPerformerReplyMode"];

    // A timeout could be substituted here; we use
    // +distantFuture to mean no timeout.
    [[NSRunLoop currentRunLoop] runMode:@"ThreadPerformerReplyMode"
	beforeDate:[NSDate distantFuture]];

    // Got reply; it's in _retainValue, and has been retained
    [[NSRunLoop currentRunLoop] removePort:_replyPort
	forMode:@"ThreadPerformerReplyMode"];
    result = _returnValue;
    _returnValue = nil;

    [_lock unlock];

    // Release our previous retains.
    [target release];
    [arg1 release];

    // Autorelease our retain on the return value
    // in the context of the current thread's current
    // autorelease pool.
    return [result autorelease];
}

// The delegate of a port receives the -handleMachMessage: or
// -handlePortMessage: method when a message arrives on a port,
// if the delegate implements either of those.
- (void)handlePortMessage:(NSPortMessage *)message {
    unsigned int msgid = [message msgid];
    NSArray *components = [message components];
    switch (msgid) {
    case 0: { // msgid 0 means request
	NSData *methodData;
	ThreadPerformerQuery query;
	ThreadPerformerResponse response;

	methodData = [components objectAtIndex:0];
	memmove(&query, [methodData bytes], sizeof(query));

	// Perform the method
	response.returnVal =
	    [query.target performSelector:query.selector
		withObject:query.arg1];

	// Retain return value so it isn't freed while we
	// return it to the other thread.
	[response.returnVal retain];

	// We send the return value back to the requesting thread
	// with a message to provide a way for the requestor to
	// sleep, and be woken up.  Also to illustrate running a
	// run loop.  Since -performSelector:.... locks a lock,
	// an NSConditionLock could also have been used for the
	// same purpose (which is somewhat faster and uses memory).

	// Send return value back to thread
	methodData = [NSData dataWithBytes:&response
		length:sizeof(response)];
	message = [[NSPortMessage allocWithZone:NULL]
		initWithSendPort:[message sendPort] receivePort:nil
		components:[NSArray arrayWithObject:methodData]];
	// msgid 1 means reply
	[message setMsgid:1];

	// A timeout could be substituted here; we use
	// +distantFuture to mean no timeout.
	if (![message sendBeforeDate:[NSDate distantFuture]]) {
	    [message release];

	    // It can be argued that this exception should be
	    // returned to 
	    [NSException raise:NSInternalInconsistencyException
		format:@"Could not return value"];
	}
	[message release];	// Done with message
	break;
    }
    case 1: { // msgid 1 means reply
	NSData *methodData;
	ThreadPerformerResponse response;

	methodData = [components objectAtIndex:0];
	memmove(&response, [methodData bytes], sizeof(response));
	// The return value was previously retained by the other
	// thread, and will be autoreleased by -performSelector:...
	_returnValue = response.returnVal;
	break;
    }
    }
}

@end

